/**
 * $Id$
 * $Author$
 * $Log$
*/


#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include "scale/commons.h"
#include "scale/alloc.h"
#include "scale/hash.h"
#include "scale/list.h"
#include "scale/heap.h"
#include "scale/vector.h"
#include "scale/btree.h"
#include "scale/stack.h"
#include "scale/queue.h"


/* test-data list */
size_t tdsize = -1;
char **tdlist = NULL;


double 
bench_xarray(sxarray_t *xarr)
{
	clock_t stime, etime;
	int i;

	stime = clock();

	for (i=0; i<tdsize; ++i)
		s_xarray_add(xarr, tdlist[i]);
	for (i=tdsize-1; i>=0; --i)
		s_xarray_remove(xarr, i);

	etime = clock();

	return (etime - stime) / (double)CLOCKS_PER_SEC;
}


int
bench_init(size_t tdc)
{
	char buf[32];
	int i;

	tdsize = tdc;
	memset(buf, 0x00, 32);
	buf[0] = 'a' - 1; /* init to before the 'a'. */

	tdlist = (char **)s_malloc(sizeof(char *) * tdsize);

	for (i=0; i<tdsize; i++) {
		snprintf(buf, 32, "%d", i);
		tdlist[i] = strdup(buf);
		
	}
	return 1;
}


void
finalize()
{
	int i;

	for (i=0; i<tdsize; ++i)
		free(tdlist[i]);
	s_freen(tdlist, sizeof(char *) * tdsize);
}


double 
bench_vector()
{
	sxarray_t *xarr;
	double time;

	xarr = s_vector_xarray();
	time = bench_xarray(xarr);
	s_xarray_free(xarr);

	return time;
}


double
bench_list()
{
	sxarray_t *xarr;
	double time;

	xarr = s_list_xarray();
	time = bench_xarray(xarr);
	s_xarray_free(xarr);

	return time;
}


double
bench_heap()
{
	sheap_t *heap;
	double etime, stime;
	int i;

	heap = s_heap_alloc(s_string_compare);

	stime = clock();

	for (i=0; i<tdsize; ++i) 
		s_heap_push(heap, tdlist[i]);

	for (i=0; i<tdsize; ++i) {
		s_heap_pop(heap);
	}

	etime = clock();

	s_heap_free(heap);

	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_stack()
{
	sstack_t *stack;
	double stime, etime;
	int i;

	stack = s_stack_alloc();

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_stack_push(stack, tdlist[i]);
	for (i=0; i<tdsize; ++i)
		s_stack_pop(stack);
	etime = clock();

	s_stack_free(stack);
	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_hash()
{
	shash_t *hash;
	double stime, etime;
	int i;

	hash = s_hash_alloc(DEFAULT_HASH_SIZE, s_string_hashfunc,
						s_string_compare);

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_hash_put(hash, tdlist[i], tdlist[i]);

	for (i=0; i<tdsize; ++i)
		s_hash_remove(hash, tdlist[i]);
	etime = clock();

	s_hash_free(hash);

	return (etime - stime) / CLOCKS_PER_SEC;
}


double
bench_btree()
{
	sbtree_t *bt;
	double stime, etime;
	int i;

	bt = s_btree_alloc(s_string_compare);

	stime = clock();
	for (i=0; i<tdsize; ++i)
		s_btree_add(bt, tdlist[i], tdlist[i]);

	for (i=0; i<tdsize; ++i) 
		s_btree_remove(bt, tdlist[i]);
	etime = clock();

	s_btree_free(bt);
	return (etime - stime) / CLOCKS_PER_SEC;
}


int
main(int argc, char *argv[])
{
	int benchlist[] = {
		123, 1234, 12345, 123456, 10, 12, 19,
		54321, 4321, 321, 21, 1, 99, 999, 9999, 99999,
		8989, 9898, 7878, 8787, 6767, 5643, 1357, 2468,
		123456, 654321,
		-1
	};
	int i;

	i = 0;
	while (benchlist[i] != -1) {
		bench_init(benchlist[i]);
		printf("Bench-mark test(count = %d)\n", benchlist[i]);

		printf("  Vector: %.3f\n", bench_vector());
		printf("  List  : %.3f\n", bench_list());
		printf("  Heap  : %.3f\n", bench_heap());
		printf("  Stack : %.3f\n", bench_stack());
		printf("  Hash  : %.3f\n", bench_hash());
		printf("  BTree : %.3f\n", bench_btree());
		printf("\n");
		finalize();
		i++;
	}

	return EXIT_SUCCESS;
}
