/**
 * $Id: sc_buffer.c,v 1.7 2004/08/05 13:54:30 jklein Exp $
 * $Author: jklein $
 * $Log: sc_buffer.c,v $
 * Revision 1.7  2004/08/05 13:54:30  jklein
 * Fixed so that Scale can be compiled on Mac OSX(10.3 or higher).
 *
 * Revision 1.6  2004/08/04 08:55:28  jklein
 * fixed off-by-one bug in sc_buffer.c
 *
 * Revision 1.5  2004/06/28 09:51:09  jklein
 * interface was changed in scale/iterator.h
 *
 * Revision 1.4  2004/06/27 13:35:17  jklein
 * Added few functions for sbuffer_t and few macros for sxarray_t.
 *
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */

#include <stdio.h>
#include <stdarg.h>

#include "scale/buffer.h"
#include "scale/alloc.h"


#if ! defined va_copy && defined __va_copy
#  define va_copy(s,d) __va_copy((s),(d))
#endif

#define DEF_INITSIZE 128
#define EXTRA_MEM_GROUTH_RATIO 32


void
_s_buffer_grouth(sbuffer_t *sbuf, size_t newsize)
{
	sbuf->string = s_reallocn(sbuf->string, sbuf->memsize, newsize);
	sbuf->memsize = newsize;
}


sbuffer_t *
s_buffer_alloc(void)
{
	return s_buffer_allocn(DEF_INITSIZE);
}


sbuffer_t *
s_buffer_allocn(size_t initsize)
{
	sbuffer_t *sbuf;

	if(initsize == 0)
		initsize = DEF_INITSIZE;
	
	sbuf = typed_malloc(sbuffer_t);
	sbuf->memsize = initsize;
	sbuf->string = s_calloc(sbuf->memsize, sizeof(char));

	return sbuf;
}


sbuffer_t *
s_buffer_alloc_str(const char *str)
{
	sbuffer_t *sbuf;

	sbuf = s_buffer_allocn(strlen(str) + 1);
	strcpy(sbuf->string, str);
	return sbuf;
}


void
s_buffer_free(sbuffer_t *sbuf)
{
	s_buffer_clear(sbuf);

	s_freen(sbuf->string, sbuf->memsize);
	s_freen(sbuf, sizeof(sbuffer_t));
}


void
s_buffer_clear(sbuffer_t *sbuf)
{
	memset(sbuf->string, '\0', sbuf->memsize);
}


size_t
s_buffer_strlen(sbuffer_t *sbuf)
{
	return strlen(sbuf->string);
}


sbuffer_t *
s_buffer_strcat(sbuffer_t *sbuf, const char *src)
{ 
	register const size_t newlen = strlen(src) + strlen(sbuf->string);
	
	if (newlen + 1 > sbuf->memsize)
		_s_buffer_grouth(sbuf, newlen + 1 + EXTRA_MEM_GROUTH_RATIO);

	strcat(sbuf->string, src);
	return sbuf;
}


sbuffer_t *
s_buffer_strncat(sbuffer_t *sbuf, const char *src, size_t n)
{
	register const size_t newlen = n + strlen(sbuf->string);
	
	if (n == 0)
		return sbuf;
	
	if (newlen + 1 > sbuf->memsize)
		_s_buffer_grouth(sbuf, newlen + 1 + EXTRA_MEM_GROUTH_RATIO);

	strncat(sbuf->string, src, n);
	sbuf->string[newlen] = '\0';

	return sbuf;
}


sbuffer_t *
s_buffer_strcpy(sbuffer_t *sbuf, const char *src)
{
	return s_buffer_strncpy(sbuf, src, strlen(src));
}


sbuffer_t *
s_buffer_strncpy(sbuffer_t *sbuf, const char *src, size_t n)
{
	if (n == 0)
		return sbuf;

	if (n + 1 > sbuf->memsize)
		_s_buffer_grouth(sbuf, n + 1 + EXTRA_MEM_GROUTH_RATIO);

	memset(sbuf->string, '\0', sbuf->memsize);
	strncpy(sbuf->string, src, n);
	
	return sbuf;
}


size_t
s_buffer_sprintf(sbuffer_t *sbuf, const char *fmt, ...)
{
	va_list ap;
	sint ret;

	va_start(ap, fmt);
	ret = s_buffer_vsprintf(sbuf, fmt, ap);
	va_end(ap);

	return ret;
}


size_t
s_buffer_vsprintf(sbuffer_t *sbuf, const char *fmt, va_list ap)
{
	va_list dupap;
	size_t newlen;

	/* allocate new memoty if we need it.
	   warning: Until glibc 2.0.6 they would return -1 when the output
	   was truncated. So this code don't work on it. */
	va_copy(dupap, ap);
	newlen = vsnprintf(NULL, 0, fmt, dupap);
	va_end(dupap);

	if(sbuf->memsize < newlen + 1)
		_s_buffer_grouth(sbuf, newlen + 1 + EXTRA_MEM_GROUTH_RATIO);
	
	vsprintf(sbuf->string, fmt, ap);
	return newlen;
}


void
s_buffer_chrrep(sbuffer_t *sbuf, char src, char dst)
{
	register unsigned char *p = (unsigned char *)sbuf->string;

	for (; *p != '\0'; ++p) {
		if (*p == src)
			*p = dst;
	}
}


void
s_buffer_strrep(sbuffer_t *sbuf, const char *src, const char *dst)
{
	sbuffer_t * const tmpbuf = s_buffer_allocn(s_buffer_strlen(sbuf) + 16);
	const char *p, *repp;
	const size_t dstlen = strlen(dst);
	const size_t srclen = strlen(src);
	
	p = sbuf->string;
	while ((repp = strstr(p, src)) != NULL) {
		s_buffer_strncat(tmpbuf, p, repp - p);
		s_buffer_strncat(tmpbuf, dst, dstlen);
		p = repp + srclen;
	}
	s_buffer_strcat(tmpbuf, p);
	
	s_buffer_strncpy(sbuf, tmpbuf->string, strlen(tmpbuf->string));
	s_buffer_free(tmpbuf);
}


sint
s_buffer_strcmp(const sbuffer_t *lbuf, const sbuffer_t *rbuf)
{
	return strcmp(lbuf->string, rbuf->string);
}


sint
s_buffer_strncmp(const sbuffer_t *lbuf, const sbuffer_t *rbuf, size_t n)
{
	return strncmp(lbuf->string, rbuf->string, n);
}


char *
s_buffer_cstr(const sbuffer_t *sbuf)
{
	return sbuf->string;
}
