/**
 * $Id: sc_list.c,v 1.9 2004/08/05 13:54:30 jklein Exp $
 * $Author: jklein $
 * $Log: sc_list.c,v $
 * Revision 1.9  2004/08/05 13:54:30  jklein
 * Fixed so that Scale can be compiled on Mac OSX(10.3 or higher).
 *
 * Revision 1.8  2004/07/10 17:04:48  jklein
 * Added few function to sxarray_t.
 *
 * Revision 1.7  2004/06/28 09:51:09  jklein
 * interface was changed in scale/iterator.h
 *
 * Revision 1.6  2004/06/27 13:35:17  jklein
 * Added few functions for sbuffer_t and few macros for sxarray_t.
 *
 * Revision 1.5  2004/06/26 06:29:20  jklein
 * Added sort function and IO utilities.
 *
 * Revision 1.4  2004/06/24 15:16:50  jklein
 * sxarray_t was added for abstruction of collection structure.
 *
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */


#include <stdio.h>
#include "scale/list.h"
#include "scale/alloc.h"


#define _CLEAN_HEAD_TAIL(__list) \
__list->head->contents = __list->tail->contents = NULL


#define _SWAP_LIST_NODE(__lhs, __rhs)\
{\
	spointer temp = __lhs->contents;\
	__lhs->contents = __rhs->contents;\
	__rhs->contents = temp;\
} while (0)



void
_sc_dump_list(const char *prompt, slist_t *list)
{
	int i;
	printf("%s {", prompt);
	for (i=0; i<list->count; i++) {
		printf("%s", (char *)s_list_get(list, i));
		if (i != list->count - 1)
			printf(", ");
	}
	printf("}\n");
}


/* returns node at 'idx'. if idx==tail, returns tail-node. */
static slist_node_t *
_get_node_at(slist_t *list, suint idx)
{
	slist_node_t *node;
	size_t i;

	if (list->count < idx)
		return NULL;

	/**
	 * use firstest way to find the 'idx'.
	 */
	if (list->count - idx > list->count / 2) {
		node = list->head->next;
		i = 0;
		while (i != idx) {
			node = node->next;
			++i;
		}
	}
	else {
		node = list->tail;
		i = list->count;
		while (i != idx) {
			node = node->prev;
			--i;
		}
	}

	return node;
}


slist_t *
s_list_alloc()
{
	slist_t *list;

	list = typed_malloc(slist_t);

	list->count = 0;
	
	list->head = typed_malloc(slist_node_t);
	list->tail = typed_malloc(slist_node_t);

	list->head->contents = NULL;
	list->head->next = list->tail;
	list->head->prev = list->tail;

	list->tail->contents = NULL;
	list->tail->next = list->tail;
	list->tail->prev = list->head;

	return list;
}


void
s_list_free(slist_t *list)
{
	slist_node_t *node, *next;

	for (node=list->head; node != list->tail; node=next) {
		next = node->next;

		node->contents = node->next = node->prev = NULL;

		s_freen(node, sizeof(slist_node_t));
	}
	/* 'node' must be list->tail. */
	node->contents = node->next = node->prev = NULL;
	s_freen(node, sizeof(slist_node_t));
	s_freen(list, sizeof(slist_t));
}


size_t
s_list_size(slist_t *list)
{
	return list->count;
}


void
s_list_insert_at(slist_t *list, suint idx, const spointer contents)
{
	slist_node_t *node;
	slist_node_t *newnode;

	newnode = typed_malloc(slist_node_t);
	newnode->contents = contents;
	newnode->next = newnode->prev = NULL;

	node = _get_node_at(list, idx);

	newnode->next = node;
	newnode->prev = node->prev;
	node->prev->next = newnode;
	node->prev = newnode;

	++list->count;
	_CLEAN_HEAD_TAIL(list);
}


void
s_list_add(slist_t *list, const spointer contents)
{
	s_list_insert_at(list, list->count, contents);
}


spointer
s_list_get(slist_t *list, suint idx)
{
	slist_node_t *node;
	
	if ((node = _get_node_at(list, idx)) == NULL) {
		return NULL;
	}

	return node->contents;
}


spointer
s_list_remove(slist_t *list, suint idx)
{
	slist_node_t *node;
	spointer contents;

	if ((node = _get_node_at(list, idx)) == NULL) {
		return NULL;
	}

	contents = node->contents;

	/* state of link list.
	 * old: [node::prev]->[node]->[node::next]
	 * new: [node::prev]->[node::next]
	 */
	node->prev->next = node->next;
	node->next->prev = node->prev;

	s_freen(node, sizeof(slist_node_t));
	--list->count;

	return contents;
}


sint32
s_list_find(slist_t *list, sc_compf_t compare, spointer contents)
{
	slist_node_t *node;
	size_t idx;
	sint8 cmpres;

	_CLEAN_HEAD_TAIL(list);

	list->tail->contents = contents;
	node = list->head->next;
	idx = 0;

	while ((cmpres = compare(contents, node->contents)) != 0) {
		++idx;
		node = node->next;
	}

	if (node != list->tail) 
		return idx;
	else
		return -1;
}


sboolean
s_list_join(slist_t *dst, slist_t *src)
{
	slist_node_t *dstnode, *srcnode;
	slist_node_t *newnode;

	_CLEAN_HEAD_TAIL(dst);
	_CLEAN_HEAD_TAIL(src);

	dstnode = dst->tail->prev;
	srcnode = src->head->next;

	while (srcnode != src->tail) {
		/**
		   old: [dstnode] --> [dstnode::next]
		   new: [dstnode] --> [newnode] --> [dstnode::next]
		*/
		newnode = typed_malloc(slist_node_t);
		newnode->contents = srcnode->contents;
		newnode->prev = dstnode;
		newnode->next = dstnode->next;

		dstnode->next->prev = newnode;
		dstnode->next = newnode;

		dstnode = dstnode->next;
		srcnode = srcnode->next;
	}

	dst->count += src->count;

	return true;
}


static slist_node_t *
_s_list_qsort_pivot(slist_node_t *start, slist_node_t *end, sc_compf_t cmpf)
{
	slist_node_t *pnode;

	pnode = start;

	while (pnode != end->next && cmpf(start->contents, pnode->contents) == 0)
		pnode = pnode->next;

	if (pnode == end->next)
		return NULL;

	return (cmpf(start->contents, pnode->contents) >= 0)?start:pnode;
}


/* static void */
/* _node_dump(char *prompt, slist_node_t *s, slist_node_t *e) */
/* { */
/* 	slist_node_t *p = s; */

/* 	printf("%s[", prompt); */
/* 	while (p != e->next) { */
/* 		printf("%s, ", (char*)p->contents); */
/* 		p = p->next; */
/* 	} */
/* 	printf("]\n"); */
/* } */


static void
_s_list_qsort(slist_node_t *start, slist_node_t *end, sc_compf_t cmpf)
{
	slist_node_t *pnode;

	if (start == end) return;

	
	if ((pnode = _s_list_qsort_pivot(start, end, cmpf)) != NULL) {
		spointer x = pnode->contents;
		slist_node_t *inode = start;
		slist_node_t *jnode = end;
		sboolean clossed = false;

		while (clossed == false && inode != jnode) {
			while (cmpf(inode->contents, x) < 0)
				inode = inode->next;
			while (cmpf(jnode->contents, x) > 0) {
				if (jnode == inode)
					clossed = true;

				jnode = jnode->prev;
			}

			if (inode == jnode) {
				inode = inode->next;
				jnode = jnode->prev;
				clossed = true;
				continue;
			}

			/* swap */
			if (clossed == false) {
				_SWAP_LIST_NODE(inode, jnode);

				inode = inode->next;
				jnode = jnode->prev;
			}

			clossed = ((inode == jnode) ||
					   (inode->prev == jnode));
		}

		_s_list_qsort(start, jnode, cmpf);
		_s_list_qsort(inode, end, cmpf);
	}
}


void
s_list_sort(slist_t *list, sc_compf_t cmpf)
{
	_s_list_qsort(list->head->next, list->tail->prev, cmpf);
}


/***************************************************************************
 * These functions are for siterator_t interface.
 ***************************************************************************/


typedef struct {
	slist_node_t *next;
} list_itr_manager_t;


static spointer
_s_list_iterator_next(siterator_t *itr)
{
	list_itr_manager_t *mng;
	slist_t *list;
	spointer contents;

	list = (slist_t *)itr->listdata;
	mng = (list_itr_manager_t *)itr->data;

	if (mng->next == NULL)
		return NULL;

	contents = mng->next->contents;

	mng->next = (list->tail != mng->next->next)?
		mng->next->next:NULL;
	return contents;
}


static void
_s_list_iterator_clear(siterator_t *itr)
{
	list_itr_manager_t *mng;
	slist_t *list;

	list = (slist_t *)itr->listdata;
	mng = (list_itr_manager_t *)itr->data;

	mng->next = list->head->next;
}


static void
_s_list_iterator_free(siterator_t *itr)
{
	s_freen(itr->data, sizeof(list_itr_manager_t));
	s_freen(itr, sizeof(siterator_t));
}


siterator_t *
s_list_iterator(slist_t *list)
{
	siterator_t *itr;

	itr = typed_malloc(siterator_t);
	itr->listdata = list;
	itr->data = typed_malloc(list_itr_manager_t);
	itr->next = _s_list_iterator_next;
	itr->clear = _s_list_iterator_clear;
	itr->free = _s_list_iterator_free;

	itr->clear(itr);

	return itr;
}




/***************************************************************************
 * These functions are for sxarray_t interface.
 ***************************************************************************/


#define __XARR_GET_LIST(__xarr) ((slist_t *)__xarr->listdata)


static void
_s_list_xarr_free(sxarray_t *xarr)
{
	slist_t *list = __XARR_GET_LIST(xarr);

	if (xarr->is_origin)
		s_list_free(list);
	s_freen(xarr, sizeof(sxarray_t));
}


static size_t
_s_list_xarr_size(sxarray_t *xarr)
{
	slist_t *list = __XARR_GET_LIST(xarr);
	return s_list_size(list);
}


static void
_s_list_xarr_insert_at(sxarray_t *xarr, suint idx, spointer contents)
{
	s_list_insert_at(__XARR_GET_LIST(xarr), idx, contents);
}


static void
_s_list_xarr_add(sxarray_t *xarr, spointer contents)
{
	s_list_add(__XARR_GET_LIST(xarr), contents);
}


static spointer
_s_list_xarr_get(sxarray_t *xarr, size_t ind)
{
	return s_list_get(__XARR_GET_LIST(xarr), ind);
}


static spointer
_s_list_xarr_remove(sxarray_t *xarr, size_t ind)
{
	return s_list_remove(__XARR_GET_LIST(xarr), ind);
}


static sint32
_s_list_xarr_find(sxarray_t *xarr, sc_compf_t cmpf, spointer contents)
{
	return s_list_find(__XARR_GET_LIST(xarr), cmpf, contents);
}


static siterator_t *
_s_list_xarr_iterator(sxarray_t *xarr)
{
	return s_list_iterator(__XARR_GET_LIST(xarr));
}


static void
_s_list_xarr_sort(sxarray_t *xarr, sc_compf_t cmpf)
{
	s_list_sort(__XARR_GET_LIST(xarr), cmpf);
}


static void
_s_list_xarr_join(sxarray_t *dest, sxarray_t *src)
{
	slist_t *dlist = __XARR_GET_LIST(dest);
	slist_node_t *node, *newnode;
	suint i;
	size_t srcsize = s_xarray_size(src);

	node = dlist->tail->prev;

	for (i=0; i<srcsize; i++) {
		newnode = typed_malloc(slist_node_t);
		newnode->contents = s_xarray_get(src, i);
		newnode->prev = newnode->next = NULL;

		newnode->next = node->next;
		newnode->prev = node;
		node->next->prev = newnode;
		node->next = newnode;

		node = newnode;
	}

	dlist->count += srcsize;
}




sxarray_t *
s_list_xarray(void)
{
	slist_t *list = s_list_alloc();
	sxarray_t *xarr = s_list_attach_xarray(list);
	xarr->is_origin = true;
	return xarr;
}


sxarray_t *
s_list_attach_xarray(slist_t *list)
{
	sxarray_t *xarr;

	xarr = typed_malloc(sxarray_t);
	xarr->listdata = list;
	xarr->free     = _s_list_xarr_free;
	xarr->size     = _s_list_xarr_size;
	xarr->add      = _s_list_xarr_add;
	xarr->get      = _s_list_xarr_get;
	xarr->remove   = _s_list_xarr_remove;
	xarr->find     = _s_list_xarr_find;
	xarr->iterator = _s_list_xarr_iterator;
	xarr->sort     = _s_list_xarr_sort;
	xarr->join	   = _s_list_xarr_join;
	xarr->insert_at	   = _s_list_xarr_insert_at;

	xarr->is_origin = false;
	
	return xarr;
}

