/**
 * $Id: tester.c,v 1.14 2004/08/05 13:54:30 jklein Exp $
 * $Author: jklein $
 * $Log: tester.c,v $
 * Revision 1.14  2004/08/05 13:54:30  jklein
 * Fixed so that Scale can be compiled on Mac OSX(10.3 or higher).
 *
 * Revision 1.13  2004/08/03 05:49:50  jklein
 * Add sxtable_t structure.
 *
 * Revision 1.12  2004/08/02 16:39:13  jklein
 * Fixed bug in the btree.c
 *
 * Revision 1.11  2004/07/22 17:27:56  jklein
 * few changes in stack.h
 *
 * Revision 1.10  2004/07/11 04:06:07  jklein
 * fixed s_vector_sort.
 *
 * Revision 1.9  2004/07/10 17:04:48  jklein
 * Added few function to sxarray_t.
 *
 * Revision 1.8  2004/06/27 13:35:17  jklein
 * Added few functions for sbuffer_t and few macros for sxarray_t.
 *
 * Revision 1.7  2004/06/26 06:29:20  jklein
 * Added sort function and IO utilities.
 *
 * Revision 1.6  2004/06/24 15:16:50  jklein
 * sxarray_t was added for abstruction of collection structure.
 *
 * Revision 1.5  2004/06/23 12:59:01  jklein
 * Added scale/btree.h, sc_btree.c.
 *
 * Revision 1.4  2004/06/22 12:23:39  jklein
 * correcting mistakes.
 *
 * Revision 1.3  2004/06/21 16:44:17  jklein
 * Added iterator.
 *
 */


#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include "scale/commons.h"
#include "scale/alloc.h"
#include "scale/list.h"
#include "scale/vector.h"
#include "scale/buffer.h"
#include "scale/hash.h"
#include "scale/stack.h"
#include "scale/heap.h"
#include "scale/commons.h"
#include "scale/queue.h"
#include "scale/btree.h"
#include "scale/iterator.h"
#include "scale/xarray.h"
#include "scale/graphviz.h"


#define FUNC_START() printf("* %s\n", __func__)
#define FUNC_END()   

typedef int (*testfunc)(void);


void
iterator_tester(siterator_t *itr)
{
	sint i = 0;
	spointer contents;

	while ((contents = itr->next(itr)) != NULL) {
		switch (i) {
		case 0: assert(strcmp(contents, "A") == 0); break;
		case 1: assert(strcmp(contents, "B") == 0); break;
		case 2: assert(strcmp(contents, "C") == 0); break;
		default:
			assert(false);
		}
		++i;
	}

	assert(i == 3);
}


void
sxarray_join_test(sxarray_t *dx, sxarray_t *sx)
{
	assert(s_xarray_size(dx) == 0);
	assert(s_xarray_size(sx) == 0);

	s_xarray_add(dx, "A");
	s_xarray_add(dx, "B");
	s_xarray_add(dx, "C");

	s_xarray_add(sx, "a");
	s_xarray_add(sx, "b");
	s_xarray_add(sx, "c");

	assert(s_xarray_size(dx) == 3);
	assert(s_xarray_size(sx) == 3);

	s_xarray_join(dx, sx);

	assert(s_xarray_size(dx) == 6);

	assert(strcmp(s_xarray_get(dx, 0), "A") == 0);
	assert(strcmp(s_xarray_get(dx, 1), "B") == 0);
	assert(strcmp(s_xarray_get(dx, 2), "C") == 0);
	assert(strcmp(s_xarray_get(dx, 3), "a") == 0);
	assert(strcmp(s_xarray_get(dx, 4), "b") == 0);
	assert(strcmp(s_xarray_get(dx, 5), "c") == 0);

	assert(strcmp(s_xarray_remove(dx, 0), "A") == 0);
	assert(strcmp(s_xarray_remove(dx, 0), "B") == 0);
	assert(strcmp(s_xarray_remove(dx, 0), "C") == 0);
	assert(strcmp(s_xarray_remove(dx, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(dx, 0), "b") == 0);
	assert(strcmp(s_xarray_remove(dx, 0), "c") == 0);

	assert(s_xarray_size(dx) == 0);
}


void
sxarray_iterator_tester(sxarray_t *xarr)
{
	siterator_t *itr;

	s_xarray_add(xarr, "B");
	s_xarray_add(xarr, "C");
	s_xarray_add(xarr, "A");
	s_xarray_add(xarr, "a");

	s_xarray_sort(xarr, s_string_compare);

	assert(s_xarray_size(xarr) == 4);
	assert(strcmp(s_xarray_get(xarr, 0), "A") == 0);
	assert(strcmp(s_xarray_get(xarr, 1), "B") == 0);
	assert(strcmp(s_xarray_get(xarr, 2), "C") == 0);
	assert(strcmp(s_xarray_get(xarr, 3), "a") == 0);

	itr = s_xarray_iterator(xarr);

	assert(strcmp(s_iterator_next(itr), "A") == 0);
	assert(strcmp(s_iterator_next(itr), "B") == 0);
	assert(strcmp(s_iterator_next(itr), "C") == 0);
	assert(strcmp(s_iterator_next(itr), "a") == 0);

	s_iterator_free(itr);

	assert(strcmp(s_xarray_remove(xarr, 0), "A") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "B") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "C") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);

}



void
sxarray_sort_tester(sxarray_t *xarr)
{
	assert(s_xarray_size(xarr) == 0);

	s_xarray_add(xarr, "a");	s_xarray_add(xarr, "w");
	s_xarray_add(xarr, "b");	s_xarray_add(xarr, "x");
	s_xarray_add(xarr, "c");	s_xarray_add(xarr, "y");
	s_xarray_add(xarr, "d");	s_xarray_add(xarr, "z");
	s_xarray_sort(xarr, s_string_compare);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "b") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "c") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "d") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "w") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "x") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "y") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "z") == 0);


	s_xarray_add(xarr, "a");	s_xarray_add(xarr, "w");
	s_xarray_sort(xarr, s_string_compare);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "w") == 0);

	
	s_xarray_add(xarr, "a");	s_xarray_add(xarr, "b");
	s_xarray_add(xarr, "c");	s_xarray_add(xarr, "d");
	s_xarray_sort(xarr, s_string_compare);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "b") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "c") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "d") == 0);

	s_xarray_add(xarr, "b");	s_xarray_add(xarr, "a");
	s_xarray_sort(xarr, s_string_compare);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "b") == 0);

	s_xarray_add(xarr, "a");	s_xarray_add(xarr, "b");
	s_xarray_add(xarr, "b");	s_xarray_add(xarr, "a");
	s_xarray_sort(xarr, s_string_compare);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "a") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "b") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "b") == 0);


}


void
sxarray_tester(sxarray_t *xarr)
{
	sxarray_t *vx, *lx; /* vector xarray, list xarray */

	assert(xarr->size(xarr) == 0);

	s_xarray_add(xarr, "B");
	s_xarray_add(xarr, "C");
	s_xarray_add(xarr, "A");

	s_xarray_sort(xarr, s_string_compare);

	assert(s_xarray_size(xarr) == 3);
	assert(strcmp(s_xarray_get(xarr, 0), "A") == 0);
	assert(strcmp(s_xarray_get(xarr, 1), "B") == 0);
	assert(strcmp(s_xarray_get(xarr, 2), "C") == 0);
	assert(s_xarray_find(xarr, s_string_compare, "A") == 0);
	assert(s_xarray_find(xarr, s_string_compare, "B") == 1);
	assert(s_xarray_find(xarr, s_string_compare, "C") == 2);
	assert(strcmp(s_xarray_remove(xarr, 0), "A") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "B") == 0);
	assert(strcmp(s_xarray_remove(xarr, 0), "C") == 0);
	assert(s_xarray_size(xarr) == 0);

	/* join test */
	vx = s_vector_xarray();
	lx = s_list_xarray();

	sxarray_join_test(xarr, vx);
	sxarray_join_test(xarr, lx);

	s_xarray_free(vx);
	s_xarray_free(lx);

	/* iterator test */
	assert(s_xarray_size(xarr) == 0);
	sxarray_iterator_tester(xarr);

	/* sort test */
	sxarray_sort_tester(xarr);
}


int
sxtable_test(sxtable_t *xt)
{
	assert(s_xtable_size(xt) == 0);

	s_xtable_put(xt, "a", "A");
	s_xtable_put(xt, "b", "B");
	s_xtable_put(xt, "c", "C");

	assert(s_xtable_size(xt) == 3);

	assert(strcmp(s_xtable_get(xt, "a"), "A") == 0);
	assert(strcmp(s_xtable_get(xt, "b"), "B") == 0);
	assert(strcmp(s_xtable_get(xt, "c"), "C") == 0);

	s_xtable_remove(xt, "a");
	s_xtable_remove(xt, "b");
	s_xtable_remove(xt, "c");

	assert(s_xtable_size(xt) == 0);

	{/* test iterator */
		siterator_t *keys;
		char *key;
		int has_read = 0;

		s_xtable_put(xt, "1", "a");
		s_xtable_put(xt, "2", "b");
		s_xtable_put(xt, "4", "c");
		s_xtable_put(xt, "8", "d");

		assert(keys = s_xtable_keys(xt));

		while ( (key = s_iterator_next(keys)) != NULL) {
			if (strcmp(key, "1") == 0) {
				if (has_read & 1)
					fprintf(stderr, "ERROR(%d)\n", __LINE__);
				has_read = has_read | 1;
			}
			else if (strcmp(key, "2") == 0) {
				if (has_read & 2)
					fprintf(stderr, "ERROR(%d)\n", __LINE__);
				has_read = has_read | 2;
			}
			else if (strcmp(key, "4") == 0) {
				if (has_read & 4)
					fprintf(stderr, "ERROR(%d)\n", __LINE__);
				has_read = has_read | 4;
			}
			else if (strcmp(key, "8") == 0) {
				if (has_read & 8)
					fprintf(stderr, "ERROR(%d)\n", __LINE__);
				has_read = has_read | 8;
			}
		}

		if (has_read != (1|2|4|8)) {
			fprintf(stderr, "not read all\n");
			return 0;
		}

		s_iterator_free(keys);
	}

	return 1;
}


int
test_list(void)
{
	slist_t *list;
	slist_t *sublist;

	FUNC_START();

	list = s_list_alloc();

	s_list_add(list, "A");
	s_list_add(list, "B");
	s_list_add(list, "C");

	s_list_insert_at(list, 0, "x");
	s_list_insert_at(list, 2, "y");
	s_list_insert_at(list, 5, "z");

	assert(list->count == 6);
	assert(strcmp(s_list_get(list, 0), "x") == 0);
	assert(strcmp(s_list_get(list, 1), "A") == 0);
	assert(strcmp(s_list_get(list, 2), "y") == 0);	
	assert(strcmp(s_list_get(list, 3), "B") == 0);
	assert(strcmp(s_list_get(list, 4), "C") == 0);
	assert(strcmp(s_list_get(list, 5), "z") == 0);


	assert(strcmp(s_list_remove(list, 5), "z") == 0);
	assert(strcmp(s_list_remove(list, 2), "y") == 0);
	assert(strcmp(s_list_remove(list, 0), "x") == 0);

	assert(strcmp(s_list_remove(list, 1), "B") == 0);
	assert(list->count == 2);

	assert(strcmp(s_list_get(list, 0), "A") == 0);
	assert(strcmp(s_list_get(list, 1), "C") == 0);

	assert(s_list_find(list, s_string_compare, "A") == 0);
	assert(s_list_find(list, s_string_compare, "C") == 1);

	sublist = s_list_alloc();
	s_list_add(sublist, "F");
	s_list_add(sublist, "D");
	s_list_add(sublist, "E");
	s_list_add(sublist, "Z");
	s_list_add(sublist, "S");

	s_list_sort(sublist, s_string_compare);

	assert(strcmp(s_list_remove(sublist, 3), "S") == 0);
	assert(strcmp(s_list_remove(sublist, 3), "Z") == 0);

	assert(sublist->count == 3);
	assert(s_list_find(sublist, s_string_compare, "D") == 0);
	assert(s_list_find(sublist, s_string_compare, "E") == 1);
	assert(s_list_find(sublist, s_string_compare, "F") == 2);

	assert(s_list_join(list, sublist));

	assert(list->count == 5);

	assert(strcmp(s_list_get(list, 0), "A") == 0);
	assert(strcmp(s_list_get(list, 1), "C") == 0);
	assert(strcmp(s_list_get(list, 2), "D") == 0);
	assert(strcmp(s_list_get(list, 3), "E") == 0);
	assert(strcmp(s_list_get(list, 4), "F") == 0);
	
	assert(s_list_find(list, s_string_compare, "A") == 0);
	assert(s_list_find(list, s_string_compare, "C") == 1);
	assert(s_list_find(list, s_string_compare, "D") == 2);
	assert(s_list_find(list, s_string_compare, "E") == 3);
	assert(s_list_find(list, s_string_compare, "F") == 4);

	{ /* list iterator test */
		slist_t *itrlist = s_list_alloc();
		siterator_t *itr;
		s_list_add(itrlist, "A");
		s_list_add(itrlist, "B");
		s_list_add(itrlist, "C");

		assert(strcmp(s_list_get(itrlist, 0), "A") == 0);
		assert(strcmp(s_list_get(itrlist, 1), "B") == 0);
		assert(strcmp(s_list_get(itrlist, 2), "C") == 0);

		itr = s_list_iterator(itrlist);

		iterator_tester(itr);
		itr->clear(itr);
		iterator_tester(itr);

		itr->free(itr);
		s_list_free(itrlist);
	}

	{ /* list sxarray test */
		sxarray_t *xarr = s_list_xarray();
		sxarray_tester(xarr);
		xarr->free(xarr);
	}

	s_list_free(list);
	s_list_free(sublist);

	FUNC_END();
	return 1;
}


int
test_vector(void)
{
	svector_t *vect;
	svector_t *subvect;

	FUNC_START();
	
	vect = s_vector_alloc();

	s_vector_add(vect, "A");
	s_vector_add(vect, "B");
	s_vector_add(vect, "C");

	s_vector_insert_at(vect, 0, "x");
	s_vector_insert_at(vect, 2, "y");
	s_vector_insert_at(vect, 5, "z");

	assert(s_vector_size(vect) == 6);
	assert(strcmp(s_vector_get(vect, 0), "x") == 0);
	assert(strcmp(s_vector_get(vect, 1), "A") == 0);
	assert(strcmp(s_vector_get(vect, 2), "y") == 0);
	assert(strcmp(s_vector_get(vect, 3), "B") == 0);
	assert(strcmp(s_vector_get(vect, 4), "C") == 0);
	assert(strcmp(s_vector_get(vect, 5), "z") == 0);

	assert(strcmp(s_vector_remove(vect, 5), "z") == 0);
	assert(strcmp(s_vector_remove(vect, 2), "y") == 0);
	assert(strcmp(s_vector_remove(vect, 0), "x") == 0);	

	assert(s_vector_find(vect, s_string_compare, "A") == 0);
	assert(s_vector_find(vect, s_string_compare, "B") == 1);
	assert(s_vector_find(vect, s_string_compare, "C") == 2);

	assert(strcmp(s_vector_remove(vect, 1), "B") == 0);
	assert(s_vector_size(vect) == 2);

	assert(strcmp(s_vector_get(vect, 0), "A") == 0);
	assert(strcmp(s_vector_get(vect, 1), "C") == 0);


	subvect = s_vector_alloc();
	s_vector_add(subvect, "c");
	s_vector_add(subvect, "a");
	s_vector_add(subvect, "b");

	s_vector_sort(subvect, s_string_compare);

	assert(s_vector_size(subvect) == 3);

	assert(s_vector_join(vect, subvect));

	assert(s_vector_size(vect) == 5);
	assert(strcmp(s_vector_get(vect, 0), "A") == 0);
	assert(strcmp(s_vector_get(vect, 1), "C") == 0);
	assert(strcmp(s_vector_get(vect, 2), "a") == 0);
	assert(strcmp(s_vector_get(vect, 3), "b") == 0);
	assert(strcmp(s_vector_get(vect, 4), "c") == 0);
	
	{ /* vector iterator test */
		svector_t *itrvect = s_vector_allocn(3);
		siterator_t *itr;

		s_vector_add(itrvect, "A");
		s_vector_add(itrvect, "B");
		s_vector_add(itrvect, "C");

		assert(s_vector_size(itrvect) == 3);
		assert(strcmp(s_vector_get(itrvect, 0), "A") == 0);
		assert(strcmp(s_vector_get(itrvect, 1), "B") == 0);
		assert(strcmp(s_vector_get(itrvect, 2), "C") == 0);

		itr = s_vector_iterator(itrvect);

		iterator_tester(itr);
		itr->clear(itr);
		iterator_tester(itr);
		itr->free(itr);

		s_vector_free(itrvect);
	}

	{ /* vector sxarray test */
		sxarray_t *xarr = s_vector_xarray();
		sxarray_tester(xarr);
		xarr->free(xarr);
	}

	s_vector_free(vect);
	s_vector_free(subvect);
	FUNC_END();
	return 1;
}


int
test_buffer(void)
{
	sbuffer_t *sbuf;

	FUNC_START();

	sbuf = s_buffer_allocn(1024);

	assert(s_buffer_strlen(sbuf) == 0);

	s_buffer_strcat(sbuf, "ABC");
	s_buffer_strcat(sbuf, "DEF");

	assert(strcmp(sbuf->string, "ABCDEF") == 0);

	s_buffer_strcpy(sbuf, "abc");

	assert(strcmp(sbuf->string, "abc") == 0);

	assert(s_buffer_sprintf(sbuf, "NO.%d", 1) == 4);
	assert(strcmp(sbuf->string, "NO.1") == 0);

	s_buffer_chrrep(sbuf, '1', '2');
	assert(strcmp(sbuf->string, "NO.2") == 0);

	s_buffer_strrep(sbuf, "NO", "XX");
	assert(strcmp(sbuf->string, "XX.2") == 0);

	s_buffer_free(sbuf);
	FUNC_END();
	return 1;
}


int
test_hash(void)
{
	shash_t *hash;
	
	FUNC_START();

	hash = s_hash_alloc(128, s_string_hashfunc, s_string_compare);

	assert(s_hash_size(hash) == 0);

	s_hash_put(hash, "a", "A");
	s_hash_put(hash, "b", "B");
	s_hash_put(hash, "c", "C");

	assert(s_hash_size(hash) == 3);
	assert(strcmp(s_hash_get(hash, "a"), "A") == 0);
	assert(strcmp(s_hash_get(hash, "b"), "B") == 0);
	assert(strcmp(s_hash_get(hash, "c"), "C") == 0);

	s_hash_remove(hash, "b");

	assert(s_hash_size(hash) == 2);
	assert(strcmp(s_hash_get(hash, "a"), "A") == 0);
	assert(s_hash_get(hash, "b") == NULL);
	assert(strcmp(s_hash_get(hash, "c"), "C") == 0);

	{ /* test iterater */
		siterator_t *itr = s_hash_keys(hash);
		void *key, *value;
		size_t count = 0;
		
		while ((key = itr->next(itr)) != NULL) {
			value = s_hash_get(hash, key);
			++count;

			if ((strcmp(key, "a") && strcmp(value, "A")) ||
				(strcmp(key, "c") && strcmp(value, "C"))) {
				/* OK */
			}
			else {
				/* NG */
				assert(false);
			}
		}
		assert(count == 2);

		itr->free(itr);
	}

	s_hash_remove(hash, "a");
	s_hash_remove(hash, "c");

	{
		sxtable_t *xt = s_hash_attach_xtable(hash);
		sxtable_test(xt);
		s_xtable_free(xt);
	}

	s_hash_free(hash);
	FUNC_END();
	return 1;
}


int
test_stack(void)
{
	sstack_t *stack;

	FUNC_START();

	stack = s_stack_allocn(10000);

	s_stack_push(stack, "A");
	s_stack_push(stack, "B");
	s_stack_push(stack, "C");
	
	assert(s_stack_size(stack) == 3);
	assert(strcmp(s_stack_pop(stack), "C") == 0);
	assert(strcmp(s_stack_pop(stack), "B") == 0);
	assert(strcmp(s_stack_pop(stack), "A") == 0);
	assert(s_stack_size(stack) == 0);

	s_stack_push(stack, "a");
	s_stack_push(stack, "b");
	
	assert(strcmp(s_stack_peek(stack), "b") == 0);
	assert(s_stack_size(stack) == 2);
	assert(strcmp(s_stack_pop(stack), "b") == 0);
	assert(strcmp(s_stack_pop(stack), "a") == 0);
	assert(s_stack_size(stack) == 0);

	s_stack_free(stack);
	FUNC_END();
	return 1;
}


int
test_queue(void)
{
	squeue_t *queue;

	FUNC_START();

	queue = s_queue_alloc();

	s_queue_enqueue(queue, "A");
	s_queue_enqueue(queue, "B");
	s_queue_enqueue(queue, "C");

	assert(s_queue_size(queue) == 3);
	assert(strcmp(s_queue_dequeue(queue), "A") == 0);
	assert(strcmp(s_queue_dequeue(queue), "B") == 0);
	assert(strcmp(s_queue_dequeue(queue), "C") == 0);
	assert(s_queue_size(queue) == 0);

	
	s_queue_enqueue(queue, "a");
	s_queue_enqueue(queue, "b");
	assert(s_queue_size(queue) == 2);
	assert(strcmp(s_queue_front(queue), "a") == 0);
	assert(s_queue_size(queue) == 2);
	assert(strcmp(s_queue_dequeue(queue), "a") == 0);
	assert(strcmp(s_queue_dequeue(queue), "b") == 0);
	assert(s_queue_size(queue) == 0);

	s_queue_free(queue);
	FUNC_END();
	return 1;
}


int
test_heap()
{
	sheap_t *heap;

	FUNC_START();

	heap = s_heap_allocn(s_string_compare, 128);

	assert(s_heap_size(heap) == 0);

	s_heap_push(heap, "A"); /* pushing ABCDEF */
	s_heap_push(heap, "C");
	s_heap_push(heap, "B");
	s_heap_push(heap, "D");
	s_heap_push(heap, "F");
	s_heap_push(heap, "E");

	assert(s_heap_size(heap) == 6);

/* 	sbuffer_t *gbuf = s_heap_graphviz(heap); */
/* 	printf("%s\n", gbuf->string); */
/* 	s_buffer_free(gbuf); */

	assert(strcmp(s_heap_pop(heap), "F") == 0);
	assert(strcmp(s_heap_pop(heap), "E") == 0);
	assert(strcmp(s_heap_pop(heap), "D") == 0);
	assert(strcmp(s_heap_pop(heap), "C") == 0);
	assert(strcmp(s_heap_pop(heap), "B") == 0);
	
	assert(strcmp(s_heap_peek(heap), "A") == 0);
	assert(s_heap_size(heap) == 1);
	assert(strcmp(s_heap_pop(heap), "A") == 0);
	assert(s_heap_size(heap) == 0);
	
	s_heap_free(heap);

	FUNC_END();
	return 1;
}


int
test_btree(void)
{
	sbtree_t *btree;

	FUNC_START();	
	btree = s_btree_alloc(s_string_compare);

	assert(s_btree_size(btree) == 0);
	s_btree_add(btree, "c", "C");
	s_btree_add(btree, "a", "A");
	s_btree_add(btree, "b", "B");
	s_btree_add(btree, "d", "D");
	s_btree_add(btree, "e", "E");
	s_btree_add(btree, "f", "E");
	s_btree_add(btree, "g", "E");
	s_btree_add(btree, "h", "E");
	s_btree_add(btree, "i", "E");
	s_btree_add(btree, "j", "E");
	s_btree_add(btree, "k", "E");
	assert(s_btree_size(btree) == 11);

	assert(strcmp(s_btree_get(btree, "a"), "A") == 0);
	assert(strcmp(s_btree_get(btree, "b"), "B") == 0);
	assert(strcmp(s_btree_get(btree, "c"), "C") == 0);
	assert(strcmp(s_btree_get(btree, "d"), "D") == 0);
	assert(strcmp(s_btree_get(btree, "e"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "f"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "g"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "h"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "i"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "j"), "E") == 0);
	assert(strcmp(s_btree_get(btree, "k"), "E") == 0);

	assert(strcmp(s_btree_remove(btree, "a"), "A") == 0);
	assert(strcmp(s_btree_remove(btree, "b"), "B") == 0);
	assert(strcmp(s_btree_remove(btree, "c"), "C") == 0);
	assert(strcmp(s_btree_remove(btree, "d"), "D") == 0);
	assert(strcmp(s_btree_remove(btree, "e"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "f"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "g"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "h"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "i"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "j"), "E") == 0);
	assert(strcmp(s_btree_remove(btree, "k"), "E") == 0);
	assert(s_btree_size(btree) == 0);

	{
		siterator_t *itr;

		s_btree_add(btree, "A", "a");
		s_btree_add(btree, "B", "b");
		s_btree_add(btree, "C", "c");
		assert(s_btree_size(btree) == 3);

		itr = s_btree_lfs_iterator(btree);
		iterator_tester(itr);

		itr->free(itr);

		assert(strcmp(s_btree_remove(btree, "A"), "a") == 0);
		assert(strcmp(s_btree_remove(btree, "B"), "b") == 0);
		assert(strcmp(s_btree_remove(btree, "C"), "c") == 0);
	}

	{
		sxtable_t *xt = s_btree_attach_xtable(btree);
		sxtable_test(xt);
		s_xtable_free(xt);
	}

	s_btree_free(btree);

	FUNC_END();
	return 1;
}


int
main(int argc, char *argv[])
{
	int kflg = 0;
	char c;
	int has_error = 0;
	testfunc funcs[] = {
		test_btree,
		test_heap,
		test_queue,
		test_stack,
		test_hash,
		test_buffer,
		test_vector,
		test_list,
		NULL
	};
	testfunc *f;

	if ((c = getopt(argc, argv, "k"))) {
		switch (c) {
		case 'k':
			kflg++;
			break;
		}
	}

	for (f=funcs; *f; f++) {
		if (!(*f)()) {
			if (! kflg) {
				printf("Test failed\n");
				return EXIT_FAILURE;
			}
			else {
				printf("contains error\n");
				has_error = 1;
			}
		}
	}

	if (has_error) {
		fprintf(stderr, "\n");
		fprintf(stderr, "Test program contains ERROR.\n");
		fprintf(stderr, "May be you want to take off '-k' option.\n");
		return EXIT_FAILURE;
	}

	return EXIT_SUCCESS;
}
