/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.range;

import java.util.Collection;
import java.util.SortedSet;
import java.util.TreeSet;

import net.morilib.util.Iterators;
import net.morilib.util.Sets;

/**
 * This class represents a merged range.<br>
 * Υ饹ʣζ֤ʤ½ɽ.
 * 
 * @author MORIGUCHI, Yuichiro 2008/01/01
 */
public final class MergedRange extends Range {
	
	//
	/*package*/ SortedSet<Interval> sets;
	
	//
	/*package*/ MergedRange() {
		this.sets = new TreeSet<Interval>();
	}
	
	//
	private MergedRange(SortedSet<Interval> sets) {
		this.sets = sets;
	}
	
	//
	/*package*/ MergedRange(Range t) {
		if(t instanceof MergedRange) {
			this.sets = new TreeSet<Interval>(
					((MergedRange)t).sets);
		} else {
			this.sets = new TreeSet<Interval>();
			sets.add((Interval)t);
		}
		//addAll(t.intervals());
	}
	
	//
	/*package*/ void addAll(Collection<Interval> intervals) {
		for(Interval o : intervals) {
			add(o);
		}
	}
	
	// 20080524 add
	//
	private SortedSet<Interval> optimizeInterval(
			SortedSet<Interval> s) {
		SortedSet<Interval> r = new TreeSet<Interval>();
		Interval  k = null;
		
		for(Interval j : s) {
			if(k == null) {
				k = j;
			} else if(k.contactSupremumBound(j)) {
				if(k.isSupremumOpen() && j.isInfimumClosed()) {
					k = k.cloneInterval(
							Interval.FROM_INFIMUM, j);
				} else if(k.isSupremumClosed() && j.isInfimumOpen()) {
					k = k.cloneInterval(
							Interval.FROM_INFIMUM, j);
				} else {
					r.add(k);
					k = j;
				}
			} else {
				r.add(k);
				k = j;
			}
		}
		
		if(k != null) {
			r.add(k);
		}
		return r;
	}
	// 20080524 add end
	
	//
	/*package*/ void add(Interval t) {
		SortedSet<Interval> r = new TreeSet<Interval>();
		Interval  k = null, l = t;
		
		for(Interval j : sets) {
			if(j.independentOf(l)) {
				r.add((k == null) ? j : k);
				k = null;
			} else {
				l = k = l.coveredInterval(j);
			}
		}
		
		r.add(l);
		sets.clear();
		sets = optimizeInterval(r);
	}

	/**
	 * returns true if the specified object is below the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤꾮Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the infimum
	 * @see net.morilib.range.Range#isInfimumAbove(java.lang.Object)
	 */
	/*package*/ boolean isInfimumAbove(Object o) {
		for(Range m : sets) {
			if(!m.isInfimumAbove(o)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the specified object is above the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤礭Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the infimum
	 * @see net.morilib.range.Range#isInfimumBelow(java.lang.Object)
	 */
	/*package*/ boolean isInfimumBelow(Object o) {
		for(Range m : sets) {
			if(m.isInfimumBelow(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the specified object is equal to the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤβ¤Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the infimum
	 * @see net.morilib.range.Range#isInfimumEqualTo(java.lang.Object)
	 */
	/*package*/ boolean isInfimumEqualTo(Object o) {
		return !isInfimumAbove(o) && !isInfimumBelow(o);
	}

	/**
	 * returns true if the specified object is below the supremum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤꾮Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the supremum
	 * @see net.morilib.range.Range#isSupremumAbove(java.lang.Object)
	 */
	/*package*/ boolean isSupremumAbove(Object o) {
		for(Range m : sets) {
			if(m.isSupremumAbove(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the specified object is above the supremum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤礭Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the supremum
	 * @see net.morilib.range.Range#isSupremumBelow(java.lang.Object)
	 */
	/*package*/ boolean isSupremumBelow(Object o) {
		for(Range m : sets) {
			if(!m.isSupremumBelow(o)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the specified object is equal to the infimum
	 * of this range.<br>
	 * <p>ꤵ줿֥ȤϰϤξ¤Ȥtrue.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the supremum
	 * @see net.morilib.range.Range#isSupremumEqualTo(java.lang.Object)
	 */
	/*package*/ boolean isSupremumEqualTo(Object o) {
		return !isSupremumAbove(o) && !isSupremumBelow(o);
	}

	/**
	 * returns true if the specified object is below the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζ꾮Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the infimum
	 * @see net.morilib.range.Range#isInfimumBoundAbove(java.lang.Object)
	 */
	/*package*/ boolean isInfimumBoundAbove(Object o) {
		for(Range m : sets) {
			if(!m.isInfimumBoundAbove(o)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the specified object is above the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζ礭Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the infimum
	 * @see net.morilib.range.Range#isInfimumBoundBelow(java.lang.Object)
	 */
	/*package*/ boolean isInfimumBoundBelow(Object o) {
		for(Range m : sets) {
			if(m.isInfimumBoundBelow(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the specified object is equal to the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤβ¤ζȤ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the infimum
	 * @see net.morilib.range.Range#isInfimumBoundEqualTo(java.lang.Object)
	 */
	/*package*/ boolean isInfimumBoundEqualTo(Object o) {
		return !isInfimumBoundAbove(o) && !isInfimumBoundBelow(o);
	}

	/**
	 * returns true if the specified object is below the supremum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζ꾮Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object below the supremum
	 * @see net.morilib.range.Range#isSupremumBoundAbove(java.lang.Object)
	 */
	/*package*/ boolean isSupremumBoundAbove(Object o) {
		for(Range m : sets) {
			if(m.isSupremumBoundAbove(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the specified object is above the supremum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζ礭Ȥ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object above the supremum
	 * @see net.morilib.range.Range#isSupremumBoundBelow(java.lang.Object)
	 */
	/*package*/ boolean isSupremumBoundBelow(Object o) {
		for(Range m : sets) {
			if(!m.isSupremumBoundBelow(o)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the specified object is equal to the infimum
	 * bound of this range.<br>
	 * ꤵ줿֥ȤϰϤξ¤ζȤ
	 * true.
	 * 
	 * @param o  the object to be tested
	 * @return  true is the object equals the supremum
	 * @see net.morilib.range.Range#isSupremumBoundEqualTo(java.lang.Object)
	 */
	/*package*/ boolean isSupremumBoundEqualTo(Object o) {
		return !isSupremumBoundAbove(o) && !isSupremumBoundBelow(o);
	}

	/**
	 * returns true if all points of the specified range is below
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the infimum
	 * @see net.morilib.range.Range#isInfimumBoundAboveAllClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isInfimumBoundAboveAllClosureOf(Range r) {
		for(Range m : sets) {
			if(!m.isInfimumBoundAboveAllClosureOf(r)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if all points of the specified range is above
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤβ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the infimum
	 * @see net.morilib.range.Range#isInfimumBoundBelowAllClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isInfimumBoundBelowAllClosureOf(Range r) {
		for(Range m : sets) {
			if(m.isInfimumBoundBelowAllClosureOf(r)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if any points of the specified range is below
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the infimum
	 * @see net.morilib.range.Range#isInfimumBoundAboveAnyClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isInfimumBoundAboveAnyClosureOf(Range r) {
		for(Range m : sets) {
			if(!m.isInfimumBoundAboveAnyClosureOf(r)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if any points of the specified range is above
	 * the infimum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤβ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the infimum
	 * @see net.morilib.range.Range#isInfimumBoundBelowAnyClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isInfimumBoundBelowAnyClosureOf(Range r) {
		for(Range m : sets) {
			if(m.isInfimumBoundBelowAnyClosureOf(r)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the infimum bound of this range contacts
	 * the supremum bound of the specified range.<br>
	 * ϰϤβ¤ζͿ줿ϰϤξ¤ζȰפ
	 * (ܿ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum contacts the supremum the range
	 * @see net.morilib.range.Range#contactInfimumBound(net.morilib.range.Range)
	 */
	/*package*/ boolean contactInfimumBound(Range r) {
		return (!isInfimumBoundAboveAllClosureOf(r) &&
				!isInfimumBoundBelowAnyClosureOf(r));
	}

	/**
	 * returns true if the infimum bound of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤβ¤ζͿ줿ϰϤβ¤ζȰפ
	 * (̤Ǥ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum bound is the same as one
	 * @see net.morilib.range.Range#commonInfimumBoundTo(net.morilib.range.Range)
	 */
	/*package*/ boolean commonInfimumBoundTo(Range r) {
		return (!isInfimumBoundAboveAnyClosureOf(r) &&
				!isInfimumBoundBelowAllClosureOf(r));
	}

	/**
	 * returns true if all points of the specified range is below
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range below the supremum
	 * @see net.morilib.range.Range#isSupremumBoundAboveAllClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isSupremumBoundAboveAllClosureOf(Range r) {
		for(Range m : sets) {
			if(m.isSupremumBoundAboveAllClosureOf(r)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if all points of the specified range is above
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤƤϰϤξ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is all points of the range above the supremum
	 * @see net.morilib.range.Range#isSupremumBoundBelowAllClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isSupremumBoundBelowAllClosureOf(Range r) {
		for(Range m : sets) {
			if(!m.isSupremumBoundBelowAllClosureOf(r)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if any points of the specified range is below
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤξ¤ζ꾮Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range below the supremum
	 * @see net.morilib.range.Range#isSupremumBoundAboveAnyClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isSupremumBoundAboveAnyClosureOf(Range r) {
		for(Range m : sets) {
			if(m.isSupremumBoundAboveAnyClosureOf(r)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if any points of the specified range is above
	 * the supremum bound of this range.<br>
	 * ꤵ줿ϰϤΤϰϤξ¤ζ礭Ȥ
	 * ¸ߤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true is any points of the range above the supremum
	 * @see net.morilib.range.Range#isSupremumBoundBelowAnyClosureOf(net.morilib.range.Range)
	 */
	/*package*/ boolean isSupremumBoundBelowAnyClosureOf(Range r) {
		for(Range m : sets) {
			if(!m.isSupremumBoundBelowAnyClosureOf(r)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the supremum bound of this range contacts
	 * the infimum bound of the specified range.<br>
	 * ϰϤξ¤ζͿ줿ϰϤβ¤ζȰפ
	 * (ܿ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum contacts the infimum the range
	 * @see net.morilib.range.Range#contactSupremumBound(net.morilib.range.Range)
	 */
	/*package*/ boolean contactSupremumBound(Range r) {
		return (!isSupremumBoundAboveAnyClosureOf(r) &&
				!isSupremumBoundBelowAllClosureOf(r));
	}

	/**
	 * returns true if the supremum bound of this range is the same
	 * as one of the specified range.<br>
	 * ϰϤξ¤ζͿ줿ϰϤξ¤ζȰפ
	 * (̤Ǥ)Ȥtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum bound is the same as one
	 * @see net.morilib.range.Range#commonSupremumBoundTo(net.morilib.range.Range)
	 */
	/*package*/ boolean commonSupremumBoundTo(Range r) {
		return (!isSupremumBoundAboveAllClosureOf(r) &&
				!isSupremumBoundBelowAnyClosureOf(r));
	}

	/**
	 * Returns true if this set contains the given object.<br>
	 * ν礬Ϳ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is contained in this
	 * @see net.morilib.range.Range#contains(java.lang.Object)
	 */
	public boolean contains(Object o) {
		for(Range m : sets) {
			if(m.contains(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns true if this set is in the specified set.<br>
	 * ν礬Ϳ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 * @see net.morilib.range.Range#in(net.morilib.range.Range)
	 */
	public boolean in(Range t) {
		for(Range r : sets) {
			if(!r.in(t)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns true if this set is independent of
	 * the specified set.<br>
	 * ν礬Ϳ줿ΩƤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is independent of the topology t
	 * @see net.morilib.range.Range#independentOf(net.morilib.range.Range)
	 */
	public boolean independentOf(Range t) {
		for(Range r : sets) {
			if(!r.independentOf(t)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns true if the interior of this set is independent of
	 * the specified set.<br>
	 * νͿ줿ΩƤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is independent of the topology t
	 * @see net.morilib.range.Range#interiorIndependentOf(net.morilib.range.Range)
	 */
	public boolean interiorIndependentOf(Range t) {
		for(Range r : sets) {
			if(!r.interiorIndependentOf(t)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns true if this is empty.<br>
	 * ν礬ΤȤtrue.
	 * 
	 * @return  true if this is empty
	 * @see net.morilib.range.Range#isEmpty()
	 */
	public boolean isEmpty() {
		for(Range r : sets) {
			if(!r.isEmpty()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * return true if the smallest edge of this range is finite.
	 * <p>ϰϤξ¦üͭ¤ΤȤtrue.
	 * 
	 * @return  true if the smallest edge of this range is finite
	 * @see net.morilib.range.Range#isInfimumFinite()
	 */
	public boolean isInfimumFinite() {
		for(Range r : sets) {
			if(!r.isInfimumFinite()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * return true if the largest edge of this range is finite.
	 * <p>ϰϤ礭¦üͭ¤ΤȤtrue.
	 * 
	 * @return  true if the largest edge of this range is finite
	 * @see net.morilib.range.Range#isSupremumFinite()
	 */
	public boolean isSupremumFinite() {
		for(Range r : sets) {
			if(!r.isSupremumFinite()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the smallest edge of this range is open.
	 * <p>ϰϤξ¦üƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isInfimumClosed()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the smallest edge of this range is open
	 * @see net.morilib.range.Range#isInfimumOpen()
	 */
	public boolean isInfimumOpen() {
		return getInfimumRange().isInfimumOpen();
	}

	/**
	 * returns true if the largest edge of this range is open.
	 * <p>ϰϤ礭¦üƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isSupremumClosed()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the largest edge of this range is open
	 * @see net.morilib.range.Range#isSupremumOpen()
	 */
	public boolean isSupremumOpen() {
		return getSupremumRange().isSupremumOpen();
	}

	/**
	 * returns true if the smallest edge of this range is closed.
	 * <p>ϰϤξ¦üĤƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isInfimumOpen()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the smallest edge of this range is closed
	 * @see net.morilib.range.Range#isInfimumClosed()
	 */
	public boolean isInfimumClosed() {
		return getInfimumRange().isInfimumClosed();
	}

	/**
	 * returns true if the largest edge of this range is closed.
	 * <p>ϰϤ礭¦üĤƤȤtrue.<br />
	 * Υ᥽åɤtrueΤȤ{@link #isSupremumOpen()}
	 * falseǤȤϸ¤ʤ(㤨Сü̵ΤȤ).
	 * 
	 * @return  true if the largest edge of this range is closed
	 * @see net.morilib.range.Range#isSupremumClosed()
	 */
	public boolean isSupremumClosed() {
		return getSupremumRange().isSupremumClosed();
	}

	/**
	 * Returns the boundary set of this topological set.<br>
	 * ϰϤΤ٤Ƥζޤླྀ.
	 * 
	 * @see net.morilib.range.Range#bound()
	 */
	public Range bound() {
		RangeAdder res = new RangeAdder();
		
		for(Interval b : sets) {
			for(Object s : b.boundElements()) {
				if(!isNeighborhoodOf(s)) {
					res.add(Interval.newPoint(s));
				}
			}
		}
		return res.toRange();
	}

	/**
	 * Returns true if the given object is neighborhood of
	 * this set.<br>
	 * νζ˵ĤޤͿ줿֥Ȥ򤳤νγ
	 * ޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is neighborhood of this
	 * @see net.morilib.range.Range#isNeighborhoodOf(java.lang.Object)
	 */
	public boolean isNeighborhoodOf(Object o) {
		for(Range t : sets) {
			if(t.isNeighborhoodOf(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns true if the closure of this set contains the given
	 * object.<br>
	 * νͿ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is neighborhood of this
	 * @see net.morilib.range.Range#containsClosure(java.lang.Object)
	 */
	public boolean containsClosure(Object o) {
		for(Range t : sets) {
			if(t.containsClosure(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns true if this set contains the given object.<br>
	 * νζͿ줿֥ȤޤȤtrue.
	 * 
	 * @param o  the object to be tested
	 * @return   true if the object is contained in this
	 * @see net.morilib.range.Range#containsBound(java.lang.Object)
	 */
	public boolean containsBound(Object o) {
		for(Range t : sets) {
			if(t.containsBound(o)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the bound of this set is a subset of the
	 * specified set.<br>
	 * νζͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if the bound of this set is a subset
	 * @see net.morilib.range.Range#boundsIn(net.morilib.range.Range)
	 */
	public boolean boundsIn(Range t) {
		for(Range s : sets) {
			if(!s.boundsIn(t)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Returns true if the closure of this set is in
	 * the specified set.<br>
	 * νͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 * @see net.morilib.range.Range#closureContains(net.morilib.range.Range)
	 */
	public boolean closureContains(Range t) {
		for(Range s : sets) {
			if(s.closureContains(t)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Returns true if the interior of this set is in
	 * the specified set.<br>
	 * νͿ줿ʬΤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return   true if this is in the topology t
	 * @see net.morilib.range.Range#interiorContains(net.morilib.range.Range)
	 */
	public boolean interiorContains(Range t) {
		for(Range s : sets) {
			if(s.interiorContains(t)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * returns true if the set contains all points of
	 * the specified set.<br>
	 * ν礬Ϳ줿ƴޤ(Ϳ줿礬ν
	 * ʬǤ)Ȥtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if this contains all points of the specified set
	 * @see net.morilib.range.Range#containsAll(net.morilib.range.Range)
	 */
	public boolean containsAll(Range t) {
		for(Range s : sets) {
			if(s.containsAll(t)) {
				return true;
			}
		}
		return false;
		//return t.in(this);
	}

	/**
	 * returns the bounds of this range by SortedSet.<br>
	 * ϰϤζSortedSet. 
	 * 
	 * @return  the SortedSet which includes all bounds of this range
	 * @see net.morilib.range.Range#boundElements()
	 */
	public SortedSet<?> boundElements() {
		SortedSet<Object> res = new TreeSet<Object>();
		
		for(Range s : sets) {
			res.addAll(s.boundElements());
		}
		return Sets.sortedUnmodifiable(res);
	}

	/**
	 * returns true if the bound of this set is independent of
	 * the specified set.<br>
	 * νζͿ줿ΩǤȤtrue.
	 * 
	 * @param t  the topology to be tested
	 * @return  true if the bound of this set is independent
	 * @see net.morilib.range.Range#boundsIndependentOf(net.morilib.range.Range)
	 */
	public boolean boundsIndependentOf(Range t) {
		for(Range s : sets) {
			if(!s.boundsIndependentOf(t)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if this range is a open set.<br>
	 * ϰϤǤtrue.
	 * 
	 * @return  true if this range is open
	 * @see net.morilib.range.Range#isOpen()
	 */
	public boolean isOpen() {
		for(Range s : sets) {
			if(!s.isOpen()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if this range is a closed set.<br>
	 * ϰϤĽǤtrue.
	 * 
	 * @return  true if this range is closed
	 * @see net.morilib.range.Range#isClosed()
	 */
	public boolean isClosed() {
		for(Range s : sets) {
			if(!s.isClosed()) {
				return false;
			}
		}
		return true;
	}

	/**
	 * returns true if the infimum bound of this range is contained
	 * in the specified range.<br>
	 * Ϳ줿ϰϤϰϤβ¤ζޤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the infimum bound is contained in the range 
	 * @see net.morilib.range.Range#infimumBoundIn(net.morilib.range.Range)
	 */
	public boolean infimumBoundIn(Range r) {
		return getInfimumRange().infimumBoundIn(r);
	}

	/**
	 * returns true if the supremum bound of this range is contained
	 * in the specified range.<br>
	 * Ϳ줿ϰϤϰϤξ¤ζޤȤtrue.
	 * 
	 * @param r  the range to be tested
	 * @return  true if the supremum bound is contained in the range 
	 * @see net.morilib.range.Range#supremumBoundIn(net.morilib.range.Range)
	 */
	public boolean supremumBoundIn(Range r) {
		return getSupremumRange().supremumBoundIn(r);
	}

	/**
	 * returns the closure of this range.<br>
	 * ϰϤ֤.
	 * 
	 * @return  the closure
	 * @see net.morilib.range.Range#closure()
	 */
	public Range closure() {
		SortedSet<Interval> s = new TreeSet<Interval>();
		
		for(Interval t : sets) {
			s.add(t.closureInterval());
		}
		return new MergedRange(s);
	}

	/**
	 * returns the interior of this range.<br>
	 * ϰϤ֤.
	 * 
	 * @return  the interior
	 * @see net.morilib.range.Range#interior()
	 */
	public Range interior() {
		SortedSet<Interval> s = new TreeSet<Interval>();
		
		for(Interval t : sets) {
			s.add(t.interiorInterval());
		}
		return new MergedRange(s);
	}

	/**
	 * Returns the intervals which consists this range.<br>
	 * ϰϤƤζ֤.
	 * 
	 * @return  the intervals which consists this range
	 * @see net.morilib.range.Range#intervals()
	 */
	public SortedSet<Interval> intervals() {
		return Sets.sortedUnmodifiable(sets);
	}

	/**
	 * returns the hash code of this object.
	 * <p>Υ֥ȤΥϥåͤ.
	 * 
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return sets.hashCode();
	}

	/**
	 * returns the string representation of this object.
	 * <p>Υ֥Ȥʸɽ.
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return Iterators.toString(sets.iterator(), "+");
	}

}
