/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.sss.servlet.http;

import java.io.UnsupportedEncodingException;
import java.math.BigInteger;

import javax.servlet.http.HttpServletRequest;

import net.morilib.lisp.Datum;
import net.morilib.lisp.Environment;
import net.morilib.lisp.LispBoolean;
import net.morilib.lisp.LispInteger;
import net.morilib.lisp.LispMessage;
import net.morilib.lisp.LispString;
import net.morilib.lisp.Nil;
import net.morilib.lisp.Symbol;
import net.morilib.lisp.Undef;
import net.morilib.lisp.subr.BinaryArgs;
import net.morilib.lisp.subr.UnaryArgs;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/02/19
 */
public class LispHttpRequest extends Datum implements HasAttribute {

	//
	/*package*/ HttpServletRequest req;

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/02/19
	 */
	public static class GetParameter extends BinaryArgs {

		/**
		 * 
		 * @param str
		 * @return
		 */
		protected Datum toDatum(String str) {
			return (str == null) ? Nil.NIL : new LispString(str);
		}

		/* (non-Javadoc)
		 * @see net.morilib.lisp.subr.BinaryArgs#execute(net.morilib.lisp.Datum, net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		protected Datum execute(
				Datum c1a, Datum c2a, Environment env,
				LispMessage mesg) {
			HttpServletRequest r;

			if(!(c1a instanceof LispHttpRequest)) {
				throw mesg.getError(
						"err.sss.require.httprequest", c1a);
			} else if(!(c2a instanceof LispString)) {
				throw mesg.getError("err.require.string", c2a);
			}

			r = ((LispHttpRequest)c1a).req;
			return toDatum(r.getParameter(c2a.getString()));
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/02/19
	 */
	public static class GetParameterInteger extends GetParameter {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.sss.servlet.http.LispHttpRequest.GetParameter#toDatum(java.lang.String)
		 */
		@Override
		protected Datum toDatum(String str) {
			try {
				if(str == null || str.equals("")) {
					return Nil.NIL;
				}
				return LispInteger.valueOf(Long.parseLong(str));
			} catch(NumberFormatException e) {
				// next
			}

			try {
				return LispInteger.valueOf(new BigInteger(str));
			} catch(NumberFormatException e) {
				return Symbol.getSymbol("type-mismatch");
			}
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/02/19
	 */
	public static class SetCharacterEncoding extends BinaryArgs {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.subr.BinaryArgs#execute(net.morilib.lisp.Datum, net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		protected Datum execute(
				Datum c1a, Datum c2a, Environment env,
				LispMessage mesg) {
			HttpServletRequest r;

			if(!(c1a instanceof LispHttpRequest)) {
				throw mesg.getError(
						"err.sss.require.httprequest", c1a);
			} else if(!(c2a instanceof LispString)) {
				throw mesg.getError("err.require.string", c2a);
			}

			r = ((LispHttpRequest)c1a).req;
			try {
				r.setCharacterEncoding(c2a.getString());
			} catch (UnsupportedEncodingException e) {
				throw mesg.getError("err.unsupportedencoding", c2a);
			}
			return Undef.UNDEF;
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/02/19
	 */
	public static class IsSessionValid extends UnaryArgs {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.subr.UnaryArgs#execute(net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		protected Datum execute(
				Datum c1a, Environment env, LispMessage mesg) {
			if(c1a instanceof LispHttpRequest) {
				HttpServletRequest r = ((LispHttpRequest)c1a).req;

				return LispBoolean.getInstance(
						r.isRequestedSessionIdValid());
			} else {
				throw mesg.getError(
						"err.sss.require.httprequest", c1a);
			}
		}

	}

	/**
	 * 
	 *
	 *
	 * @author MORIGUCHI, Yuichiro 2011/02/19
	 */
	public static class GetSession extends UnaryArgs {

		/* (non-Javadoc)
		 * @see net.morilib.lisp.subr.UnaryArgs#execute(net.morilib.lisp.Datum, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
		 */
		@Override
		protected Datum execute(
				Datum c1a, Environment env, LispMessage mesg) {
			if(c1a instanceof LispHttpRequest) {
				HttpServletRequest r = ((LispHttpRequest)c1a).req;

				return new LispHttpSession(r.getSession());
			} else {
				throw mesg.getError(
						"err.sss.require.httprequest", c1a);
			}
		}

	}

	/**
	 * @param req
	 */
	/*package*/ LispHttpRequest(HttpServletRequest req) {
		this.req = req;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.sss.servlet.http.HasAttribute#getAttribute(java.lang.String)
	 */
	@Override
	public Object getAttribute(String name) {
		return req.getAttribute(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.sss.servlet.http.HasAttribute#setAttribute(java.lang.String, java.lang.String)
	 */
	@Override
	public void setAttribute(String name, Object value) {
		req.setAttribute(name, value);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.sss.servlet.http.HasAttribute#removeAttribute(java.lang.String)
	 */
	@Override
	public void removeAttribute(String name) {
		req.removeAttribute(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum#toDisplayString(java.lang.StringBuilder)
	 */
	@Override
	public void toDisplayString(StringBuilder buf) {
		buf.append("#<http-request>");
	}

}
