/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.sss.servlet.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import net.morilib.lisp.Datum;
import net.morilib.lisp.Scheme;
import net.morilib.lisp.sss.Loggable;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/02/19
 */
public class SchlushHttpServlet extends HttpServlet {

	//
	private static final long serialVersionUID = -6114578505278185400L;

	//
	private static class LispHttpServlet extends Datum
	implements Loggable {

		//
		private HttpServlet servlet;

		//
		private LispHttpServlet(HttpServlet servlet) {
			this.servlet = servlet;
		}

		/* (non-Javadoc)
		 * @see net.morilib.lisp.sss.Loggable#log(java.lang.String)
		 */
		@Override
		public void log(String str) {
			servlet.log(str);
		}

		/* (non-Javadoc)
		 * @see net.morilib.lisp.Datum#toDisplayString(java.lang.StringBuilder)
		 */
		@Override
		public void toDisplayString(StringBuilder buf) {
			buf.append("#<http-servlet>");
		}

	}

	//
	private static final String INIT_P     = "scheme-root";
	private static final String SCHEME_ENC = "scheme-encoding";
	private static final String DO_DELETE  = "do-delete";
	private static final String DO_GET     = "do-get";
	private static final String DO_HEAD    = "do-head";
	private static final String DO_OPTIONS = "do-options";
	private static final String DO_POST    = "do-post";
	private static final String DO_PUT     = "do-put";
	private static final String DO_TRACE   = "do-trace";

	//
	private Map<String, Scheme> schemes =
		new HashMap<String, Scheme>();

	//
	private void initScheme(Scheme s) {
		s.set("*servlet*", new LispHttpServlet(this));
		s.set("get-parameter", new LispHttpRequest.GetParameter());
		s.set("get-parameter-integer",
				new LispHttpRequest.GetParameterInteger());
		s.set("set-character-encoding",
				new LispHttpRequest.SetCharacterEncoding());
		s.set("session-valid?",
				new LispHttpRequest.IsSessionValid());
		s.set("get-session",
				new LispHttpRequest.GetSession());
		s.set("set-content-type",
				new LispHttpResponse.SetContentType());
		s.set("get-output-port",
				new LispHttpResponse.GetOutputPort());
		s.set("invalidate",
				new LispHttpSession.Invalidate());
		s.set("get-attribute", new GetAttribute());
		s.set("set-attribute!", new SetAttributeS());
	}

	//
	private Scheme getScheme(String s) {
		String s2 = s.replaceFirst("^/[^/]+", "");
		Scheme res;

		if((res = schemes.get(s2)) == null) {
			synchronized(this) {
				String enc = getInitParameter(SCHEME_ENC);
				String sh = getInitParameter(INIT_P);
				InputStream ins = null;

				if(enc == null) {
					enc = "UTF-8";
				}
				sh = sh.replace('.', '/');
				sh = "/" + sh.replaceFirst("^/", "");
				sh = sh.replaceFirst("/$", "") + "/";

				try {
					ins = SchlushHttpServlet.class.getResourceAsStream(
							sh + s2);
					if(ins == null) {
						throw new SchlushServletException(
								"not found:" + sh + s2);
					}
					res = Scheme.newInstance();
					res.readFile(new InputStreamReader(ins, enc));
					initScheme(res);
					schemes.put(s2, res);
				} catch (UnsupportedEncodingException e) {
					throw new SchlushServletException(e);
				} catch (IOException e) {
					throw new SchlushServletException(e);
				} finally {
					if(ins != null) {
						try {
							ins.close();
						} catch (IOException e) {
							throw new SchlushServletException(e);
						}
					}
				}
			}
		}
		return res;
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doDelete(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doDelete(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_DELETE,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doGet(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_GET,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doHead(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doHead(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_HEAD,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doOptions(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doOptions(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_OPTIONS,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doPost(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doPost(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_POST,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doPut(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doPut(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_PUT,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

	/* (non-Javadoc)
	 * @see javax.servlet.http.HttpServlet#doTrace(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
	 */
	@Override
	protected void doTrace(
			HttpServletRequest req, HttpServletResponse resp)
			throws ServletException, IOException {
		Scheme s = getScheme(req.getRequestURI());

		s.call(DO_TRACE,
				new LispHttpRequest(req),
				new LispHttpResponse(resp));
	}

}
