/*
 * Copyright 2009 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * 
 *
 *
 * @author MORIGUCHI, Yuichiro 2009
 */
public final class LispMessage {

	//
	private static final String CLSNAME =
		"net.morilib.lisp.init.messages";
	private static Map<Locale, LispMessage> instances =
		new HashMap<Locale, LispMessage>();

	//
	private ResourceBundle messages;

	//
	private LispMessage(Locale lc) {
		messages = ResourceBundle.getBundle(CLSNAME, lc);
	}

	/**
	 * 
	 * @param lc
	 * @return
	 */
	public static LispMessage getInstance(Locale lc) {
		LispMessage res = instances.get(lc);
		if(res == null) {
			synchronized(LispMessage.class) {
				res = new LispMessage(lc);
				instances.put(lc, res);
			}
		}
		return res;
	}

	/**
	 * 
	 * @return
	 */
	public static LispMessage getInstance() {
		return getInstance(Locale.getDefault());
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public String get(String prop) {
		try {
			return messages.getString(prop);
		} catch(MissingResourceException e) {
			return (messages.getString("err.unknown") +
					"(" + e.getKey() + ")");
		}
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public String get(String prop, String msg2) {
		return get(prop) + ":" + msg2;
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public String get(String prop, Datum d) {
		return get(prop) + ":" + LispUtils.getResult(d);
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public LispException getError(String prop) {
		return new LispException(prop, get(prop));
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public LispException getError(String prop, String msg2) {
		return new LispException(prop, get(prop, msg2));
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public LispException getError(String prop, Datum d) {
		return new LispException(prop, get(prop, d));
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public LispException getError(String prop, Throwable th) {
		return new LispException(prop, get(prop), th);
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public LispException getError(
			String prop, String msg2, Throwable th) {
		return new LispException(prop, get(prop, msg2), th);
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public LispException getError(
			String prop, Datum d, Throwable th) {
		return new LispException(prop, get(prop, d), th);
	}

	/**
	 * 
	 * @param prop
	 * @return
	 */
	public ReadException getReadError(String prop) {
		return new ReadException(prop, get(prop));
	}

	/**
	 * 
	 * @param prop
	 * @param msg2
	 * @return
	 */
	public ReadException getReadError(String prop, String msg2) {
		return new ReadException(prop, get(prop, msg2));
	}

	/**
	 * 
	 * @param prop
	 * @param d
	 * @return
	 */
	public ReadException getReadError(String prop, Datum d) {
		return new ReadException(prop, get(prop, d));
	}

	/**
	 * 
	 * @param e
	 */
	public LispException getUncaughtException(LispException e) {
		return new LispException(
				"err.srfi18.uncaught",
				get("err.srfi18.uncaught"), e);
	}

}
