/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.lisp.collection.hash;

import java.util.Iterator;
import java.util.Map;

import net.morilib.lisp.Datum;
import net.morilib.lisp.Datum2;
import net.morilib.lisp.Environment;
import net.morilib.lisp.LispMessage;
import net.morilib.lisp.LispUtils;
import net.morilib.lisp.Nil;
import net.morilib.lisp.Procedure;
import net.morilib.lisp.Scheme;
import net.morilib.lisp.Symbol;
import net.morilib.lisp.collection.ImmutableException;
import net.morilib.lisp.collection.LispBag;
import net.morilib.lisp.collection.LispCollection;
import net.morilib.lisp.collection.LispSet;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2011/05/21
 */
public class LispHashSet extends Datum2 implements LispSet {

	//
	private LispHash hash;

	/**
	 * @param hash
	 */
	public LispHashSet(LispHashSet hash) {
		this(hash.hash.duplicate());
	}

	//
	private LispHashSet(LispHash hash) {
		this.hash = hash;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#getCollectionName()
	 */
	public Symbol getCollectionName() {
		return Symbol.getSymbol("hash-set");
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#toList()
	 */
	public Datum toList() {
		return hash.keysToList();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#count(net.morilib.lisp.Datum)
	 */
	public int count(Datum c2a) {
		return hash.count(c2a);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#size()
	 */
	public int size() {
		return hash.size();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#prototype()
	 */
	public LispHashSet prototype() {
		return new LispHashSet((LispHash)hash.prototype());
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#clear()
	 */
	public Datum clear() throws ImmutableException {
		hash = (LispHash)hash.clear();
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#equalTo(net.morilib.lisp.collection.LispCollection)
	 */
	public boolean equalTo(LispCollection col) {
		for(Datum x : col) {
			if(!contains(x)) {
				return false;
			}
		}

		for(Datum x : this) {
			if(!col.contains(x)) {
				return false;
			}
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#equalTo(net.morilib.lisp.collection.LispCollection, net.morilib.lisp.Procedure, net.morilib.lisp.Environment, net.morilib.lisp.LispMessage)
	 */
	public boolean equalTo(LispCollection col, Procedure p,
			Environment env, LispMessage mesg) {
		outer: for(Datum x : col) {
			for(Datum y : this) {
				if(Scheme.callva(p, env, mesg, x, y).isTrue()) {
					continue outer;
				}
			}
			return false;
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#duplicate()
	 */
	public LispHashSet duplicate() {
		return new LispHashSet(this);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispCollection#contains(net.morilib.lisp.Datum)
	 */
	public boolean contains(Datum d) {
		return hash.containsKey(d);
	}

	/* (non-Javadoc)
	 * @see java.lang.Iterable#iterator()
	 */
	public Iterator<Datum> iterator() {
		final Iterator<Map.Entry<Datum, Datum>> ei;

		ei = hash.entryIterator();
		return new Iterator<Datum>() {

			public boolean hasNext() {
				return ei.hasNext();
			}

			public Datum next() {
				return ei.next().getKey();
			}

			public void remove() {
				ei.remove();
			}

		};
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#equivalence()
	 */
	public Procedure equivalence() {
		return hash.keyEquivalence();
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#equivalence(net.morilib.lisp.Datum, net.morilib.lisp.Datum)
	 */
	public boolean equivalence(Datum a, Datum b) {
		return hash.equivalenceKey(a, b);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#subset(net.morilib.lisp.collection.LispSet)
	 */
	public boolean subset(LispSet set) {
		for(Datum x : set) {
			if(!contains(x)) {
				return false;
			}
		}
		return true;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#copyAdd(net.morilib.lisp.Datum)
	 */
	public Datum copyAdd(Datum d) {
		return duplicate().add(d);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#add(net.morilib.lisp.Datum)
	 */
	public Datum add(Datum d) {
		hash.put0(d, Nil.NIL);
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#copyDelete(net.morilib.lisp.Datum)
	 */
	public Datum copyDelete(Datum d) {
		return duplicate().delete(d);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#delete(net.morilib.lisp.Datum)
	 */
	public Datum delete(Datum d) {
		hash.remove(d);
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#union(net.morilib.lisp.collection.LispSet)
	 */
	public Datum union(LispSet s) {
		return duplicate().addAll(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#addAll(net.morilib.lisp.collection.LispSet)
	 */
	public Datum addAll(LispSet s) {
		for(Datum x : s) {
			add(x);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#intersection(net.morilib.lisp.collection.LispSet)
	 */
	public Datum intersection(LispSet s) {
		LispHashSet r = prototype();

		for(Datum x : this) {
			if(s.contains(x)) {
				r.add(x);
			}
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#retainAll(net.morilib.lisp.collection.LispSet)
	 */
	public Datum retainAll(LispSet s) {
		Iterator<Datum> i = iterator();

		while(i.hasNext()) {
			if(!s.contains(i.next())) {
				i.remove();
			}
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#difference(net.morilib.lisp.collection.LispSet)
	 */
	public Datum difference(LispSet s) {
		LispHashSet r = prototype();

		for(Datum x : this) {
			if(!s.contains(x)) {
				r.add(x);
			}
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#removeAll(net.morilib.lisp.collection.LispSet)
	 */
	public Datum removeAll(LispSet s) {
		Iterator<Datum> i = iterator();

		while(i.hasNext()) {
			if(s.contains(i.next())) {
				i.remove();
			}
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#symmetricDifference(net.morilib.lisp.collection.LispSet)
	 */
	public Datum symmetricDifference(LispSet s) {
		LispHashSet r = prototype();

		for(Datum x : this) {
			if(!s.contains(x)) {
				r.add(x);
			}
		}

		for(Datum x : s) {
			if(!contains(x)) {
				r.add(x);
			}
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#symmetricDifferenceModify(net.morilib.lisp.collection.LispSet)
	 */
	public Datum symmetricDifferenceModify(LispSet s) {
		return symmetricDifference(s);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#copyAddFrom(net.morilib.lisp.collection.LispBag)
	 */
	public Datum copyAddFrom(LispBag d) {
		return duplicate().addFrom(d);
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#addFrom(net.morilib.lisp.collection.LispBag)
	 */
	public Datum addFrom(LispBag d) {
		for(Datum x : d) {
			add(x);
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#copyDeleteFrom(net.morilib.lisp.collection.LispBag)
	 */
	public Datum copyDeleteFrom(LispBag d) {
		LispHashSet r = prototype();

		for(Datum x : this) {
			if(!d.contains(x)) {
				r.add(x);
			}
		}
		return r;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.collection.LispSet#deleteFrom(net.morilib.lisp.collection.LispBag)
	 */
	public Datum deleteFrom(LispBag d) {
		Iterator<Datum> i = iterator();

		while(i.hasNext()) {
			if(d.contains(i.next())) {
				i.remove();
			}
		}
		return this;
	}

	/* (non-Javadoc)
	 * @see net.morilib.lisp.Datum2#toDisplayString(java.lang.StringBuilder)
	 */
	@Override
	public void toDisplayString(StringBuilder buf) {
		buf.append("#<hash-set>" + LispUtils.print(toList()));
	}

}
