/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "Sensors"

#include <hardware/sensors.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include <math.h>
#include <poll.h>
#include <pthread.h>
#include <stdlib.h>
#include <assert.h>

#include <linux/input.h>

#include <utils/Atomic.h>
#include <utils/Log.h>

#include "common.h"
#include "SensorBase.h"

#include "SensorConfig.h"
#include "BoardConfig.h"

#include "RotVecSensor.h"
#include "SynthCompassSensor.h"

/*****************************************************************************/

/* The SENSORS Module */
static int open_sensors(const struct hw_module_t* module, const char* id,
                        struct hw_device_t** device);


static int sensors__get_sensors_list(struct sensors_module_t*,
                                     struct sensor_t const** list)
{
        *list = BoardConfig::sensorList();
        return BoardConfig::sensorListSize();
}

static struct hw_module_methods_t sensors_module_methods = {
        open: open_sensors
};

struct sensors_module_t HAL_MODULE_INFO_SYM = {
        common: {
                tag: HARDWARE_MODULE_TAG,
                version_major: 1,
                version_minor: 0,
                id: SENSORS_HARDWARE_MODULE_ID,
                name: "Samsung Sensor module",
                author: "Samsung Electronic Company",
                methods: &sensors_module_methods,
                dso: NULL,
                reserved: {}
        },
        get_sensors_list: sensors__get_sensors_list,
};

struct sensors_poll_context_t {
    struct sensors_poll_device_t device; // must be first

        sensors_poll_context_t();
        ~sensors_poll_context_t();
    int activate(int handle, int enabled);
    int setDelay(int handle, int64_t ns);
    int pollEvents(sensors_event_t* data, int count);

private:
    static const size_t wake = numFds - 1;
    static const char WAKE_MESSAGE = 'W';
    struct pollfd mPollFds[numFds];
    int mWritePipeFd;
    SensorBase* mSensors[numSensorDrivers];
};

/*****************************************************************************/

sensors_poll_context_t::sensors_poll_context_t()
                       :mSensors()
{
    BoardConfig::initSensors(mSensors);

    for (int i = 0; i < numSensorDrivers; i++) {
        assert(mSensors[i]);
        if (mSensors[i]){
            mPollFds[i].fd = mSensors[i]->fd();
            mPollFds[i].events = POLLIN;
            mPollFds[i].revents = 0;
        }
    }

    // Mild hack: the synthetic compass is implemented as a slave of
    // the rotation vector, so we have to tell them about each other
    // explicitly in lieu of an architecture that lets them probe.
    SensorBase *rv = mSensors[rotvec], *sc = mSensors[syncompass];
    if (rv && sc) {
        ((RotVecSensor*)rv)->setSynthCompass((SynthCompassSensor*)sc);
        ((SynthCompassSensor*)sc)->setRotVecSensor((RotVecSensor*)rv);
    }

    int wakeFds[2];
    int result = pipe(wakeFds);
    ALOGE_IF(result<0, "error creating wake pipe (%s)", strerror(errno));
    fcntl(wakeFds[0], F_SETFL, O_NONBLOCK);
    fcntl(wakeFds[1], F_SETFL, O_NONBLOCK);
    mWritePipeFd = wakeFds[1];

    mPollFds[wake].fd = wakeFds[0];
    mPollFds[wake].events = POLLIN;
    mPollFds[wake].revents = 0;
}

sensors_poll_context_t::~sensors_poll_context_t() {
    for (int i=0 ; i<numSensorDrivers ; i++) {
        delete mSensors[i];
    }
    close(mPollFds[wake].fd);
    close(mWritePipeFd);
}

int sensors_poll_context_t::activate(int handle, int enabled) {
    int index = BoardConfig::handleToDriver(handle);
    if (index < 0) return index;

    ALOGD("%s: %s sensor %d", __func__, enabled?"enable":"disable", index);
    SensorBase* const s(mSensors[index]);
    int err = 0;

    if (enabled) {
        if ((err = s->enable(1)) < 0)
            s->close();
    } else {
        s->enable(0);
        s->close();
    }

    /* If the file descriptor is closed or opened the stored desciptor
       should be updated. Even when not opened/closed the fd should
       return correct value */
    if (mPollFds[index].fd != s->fd()) {
        const char wakeMessage(WAKE_MESSAGE);
        int result = write(mWritePipeFd, &wakeMessage, 1);
        ALOGE_IF(result<0, "error sending wake message (%s)", strerror(errno));
        mPollFds[index].fd = s->fd();
    }
    return err;
}

int sensors_poll_context_t::setDelay(int handle, int64_t ns) {

    int index = BoardConfig::handleToDriver(handle);
    if (index < 0) return index;
    return mSensors[index]->setDelay(ns);
}

int sensors_poll_context_t::pollEvents(sensors_event_t* data, int count)
{
    int nbEvents = 0;
    int n = 0;

    do {
        // see if we have some leftover from the last poll()
        for (int i=0 ; count && i<numSensorDrivers ; i++) {
            SensorBase* const sensor(mSensors[i]);
            mPollFds[i].fd = sensor->fd();
            if (sensor && ((mPollFds[i].revents & POLLIN) || (sensor->hasPendingEvents()))) {
                int nb = sensor->readEvents(data, count);
                if (nb < count) {
                    // no more data for this sensor
                    mPollFds[i].revents = 0;

                    if (nb < 0) {
                        ALOGD("%s: handle:%d error:%d", __func__, i, nb);
                        continue;
                    }
                }
                count -= nb;
                nbEvents += nb;
                data += nb;
            }
        }

        // Some sensors ("slaves") may have generated events based on
        // the input to the sensors above.  So check again.
        for (int i=0 ; count && i<numSensorDrivers ; i++) {
            SensorBase *sensor = mSensors[i];
            if (sensor->hasPendingEvents()) {
                int nb = sensor->readEvents(data, count);
                if (nb > 0) {
                    count -= nb;
                    nbEvents += nb;
                    data += nb;
                }
            }
        }

        if (count) {
            // we still have some room, so try to see if we can get
            // some events immediately or just wait if we don't have
            // anything to return
            n = poll(mPollFds, numFds, nbEvents ? 0 : POLL_TIMEOUT_MSEC);
            if (n<0) {
                ALOGE("poll() failed (%s)", strerror(errno));
                return -errno;
            }
            if (mPollFds[wake].revents & POLLIN) {
                char msg;
                int result = read(mPollFds[wake].fd, &msg, 1);
                ALOGE_IF(result<0, "error reading from wake pipe (%s)", strerror(errno));
                ALOGE_IF(msg != WAKE_MESSAGE, "unknown message on wake queue (0x%02x)", int(msg));
                mPollFds[wake].revents = 0;
            }
        }
        // if we have events and space, go read them
    } while (n && count);

    return nbEvents;
}

/*****************************************************************************/

static int poll__close(struct hw_device_t *dev)
{
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    if (ctx) {
        delete ctx;
    }
    return 0;
}

static int poll__activate(struct sensors_poll_device_t *dev,
        int handle, int enabled) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->activate(handle, enabled);
}

static int poll__setDelay(struct sensors_poll_device_t *dev,
        int handle, int64_t ns) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->setDelay(handle, ns);
}

static int poll__poll(struct sensors_poll_device_t *dev,
        sensors_event_t* data, int count) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->pollEvents(data, count);
}

/*****************************************************************************/

/** Open a new instance of a sensor device using name */
static int open_sensors(const struct hw_module_t* module, const char*,
                        struct hw_device_t** device)
{
    int status = -EINVAL;
    sensors_poll_context_t *dev = new sensors_poll_context_t();

    memset(&dev->device, 0, sizeof(sensors_poll_device_t));

    dev->device.common.tag = HARDWARE_DEVICE_TAG;
    dev->device.common.version = 0;
    dev->device.common.module = const_cast < hw_module_t * > (module);
    dev->device.common.close = poll__close;
    dev->device.activate = poll__activate;
    dev->device.setDelay = poll__setDelay;
    dev->device.poll = poll__poll;

    *device = &dev->device.common;
    status = 0;
    return status;
}
