!> Floating point numbers need a special treatment due to their storage format.
!!
!! Some people already made up their minds about those things, thus that we can
!! make use of their findings.
!! See further:
!!   - http://www.floating-point-gui.de/errors/comparison/
!!   - https://randomascii.wordpress.com/2012/02/25/comparing-floating-point-numbers-2012-edition/
!!   - http://www.ssec.wisc.edu/~paulv/Fortran90/Utility/Compare_Float_Numbers.f90.html
!!
!! Most of this modules code traces back to the work of Paul van Delst,
!! CIMSS/SSEC, paul.vandelst@ssec.wisc.edu
module tem_float_module

  use env_module,       only: rk

  implicit none

  private

  public :: operator(.feq.)
  public :: operator(.fne.)
  public :: operator(.fgt.)
  public :: operator(.fge.)
  public :: operator(.flt.)
  public :: operator(.fle.)

  interface operator (.feq.)
    module procedure equal
  end interface
  interface operator (.fne.)
    module procedure notEqual
  end interface
  interface operator (.fgt.)
    module procedure greaterThan
  end interface
  interface operator (.fge.)
    module procedure greaterThanOrEqual
  end interface
  interface operator (.flt.)
    module procedure lessThan
  end interface
  interface operator (.fle.)
    module procedure lessThanOrEqual
  end interface

contains

  !> Relational operator to test the equality of floating point numbers.
  pure function equal(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operands are equal
    !! to within numerical precision.
    logical :: res
    res = abs( a - b ) < spacing( max( abs( a ), abs( b ) ) )
  end function

  !> Relational operator to test the not-equality of floating point numbers.
  pure function notEqual(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operands are not
    !! equal to within numerical precision.
    logical :: res
    res = .not. equal(a,b)
  end function

  !> Relational operator to test if one operand is greater than another.
  pure function greaterThan(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operand a is greater
    !! than b by more than the spacing between representable floating point
    !! numbers.
    logical :: res
    res = ( a - b ) >= spacing( max( abs( a ), abs( b ) ) )
  end function

  !> Relational operator to test if one operand is greater than or equal to
  !! another.
  pure function greaterThanOrEqual(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operand a is greater
    !! than or equal to b by more than the spacing between representable
    !! floating point numbers.
    logical :: res
    res = greaterThan(a,b) .or. equal(a,b)
  end function

  !> Relational operator to test if one operand is less than another.
  pure function lessThan(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operand a is less
    !! than b by more than the spacing between representable floating point
    !! numbers.
    logical :: res
    res = ( b - a ) >= spacing( max( abs( a ), abs( b ) ) )
  end function

  !> Relational operator to test if one operand is less than or equal to
  !! another.
  pure function lessThanOrEqual(a,b) result(res)
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: a
    !> Floating point value to be compared.
    real(kind=rk), intent(in) :: b
    !> The result is a logical value indicating whether the operand a is less
    !! than or equal to b by more than the spacing between representable
    !! floating point numbers.
    logical :: res
    res = lessThan(a,b) .or. equal(a,b)
  end function

end module
