#include "memregions.h"

/****************************************************************************
 * regions table
 ****************************************************************************/
struct region_table regions[MAX_REGION_NUM] = {
	STATIC_REGION    (ALL, "all", FLASH,
			  FLASH_START, FLASH_SIZE, "4x32K/l,127x128K/l"),

	RESIZABLE_REGION (BOOTLOADER, "bootloader", FLASH,
			  FLASH_BOOTLOADER_START, 0x00020000, "4x32K/l"),

	RESIZABLE_REGION (KERNEL, "kernel", FLASH,
			  FLASH_START + 0x00020000, 0x00200000, "16x128K"),

	RESIZABLE_REGION (USERLAND, "userland", FLASH,
			  FLASH_START + 0x00220000, 0x00dc0000, "110x128K"),

	RESIZABLE_REGION (CONFIG, "config", FLASH,
			  FLASH_START + 0x00fe0000, 0x00020000, "1x128K"),

	DYNAMIC_REGION   (5, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (6, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (7, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (8, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (9, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (10, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (11, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (12, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (13, "", UNKNOWN, 0, 0, ""),
	DYNAMIC_REGION   (14, "", UNKNOWN, 0, 0, ""),

	STATIC_REGION    (DRAM1, "dram-1", RAM,
			  DRAM1_START, DRAM1_SIZE, ""),
};

#if defined(EXPERIMENTAL) && defined(ENABLE_DYNAMIC_REGION)
#include <target/command.h>
#include <target/scan.h>
#include <target/io.h>
#include <target/herrno.h>
#include <target/str.h>
/****************************************************************************
 * 
 ****************************************************************************/
static int
region_print_cmdfunc(int argc, char *argv[])
{
	int i;
	for (i=0; i<MAX_REGION_NUM; i++) {
		hprintf("%s%d: ", (i < 10) ? " " : "", i);

		switch (regions[i].mode) {
		case MAP_MODE_RESIZABLE: hprintf("[RESIZABLE], "); break;
		case MAP_MODE_DYNAMIC:   hprintf("[DYNAMIC  ], "); break;
		case MAP_MODE_STATIC:    hprintf("[STATIC   ], "); break;
		default:                 hprintf("[UNKNOWN  ], "); break;
		}
		if (regions[i].name[0])
			hprintf("0x%x:0x%x ", MAP_START(i), MAP_SIZE(i));

		switch (regions[i].type) {
		case MAP_TYPE_FLASH: hprintf("FLA "); break;
		case MAP_TYPE_RAM:   hprintf("RAM "); break;
		default:             break;
		}
		if (regions[i].name[0]) {
			hprintf("%s ", MAP_NAME(i));
			if(regions[i].type == MAP_TYPE_FLASH)
				hprintf("%s ", MAP_BLOCK(i));
		}

		hprintf("\n");
	}
	return 0;
}

/****************************************************************************
 * 
 ****************************************************************************/
static int
region_add_cmdfunc(int argc, char *argv[])
{
	int index, type, start, size;
	char *name, *block;

	if (argc < 6)
		return -H_EUSAGE;

	if (scan(argv[1], &index))
		return -H_EINVAL;

	if (index < 0 || MAX_REGION_NUM - 1 < index)
		return -H_EINVAL;

	if (scan(argv[3], &type))
		return -H_EINVAL;

	if (type < 1 || 2 < type)
		return -H_EINVAL;

	if (scan(argv[4], &start))
		return -H_EINVAL;

	if (scan(argv[5], &size))
		return -H_EINVAL;

	name = argv[2];

	if (argc == 7)
		block = argv[6];
	else
		block = "";

	switch (regions[index].mode) {
	case MAP_MODE_DYNAMIC:
		SET_MAP_NAME(index, name);
		SET_MAP_TYPE(index, type);
	case MAP_MODE_RESIZABLE:
		SET_MAP_START(index, start);
		SET_MAP_SIZE(index, size);
		SET_MAP_BLOCK(index, block);
		break;
	case MAP_MODE_STATIC:
		hprintf("can't modefy this region: %d\n", index);
		return 0;
	default:
		break;
	}

	return 0;
}

/****************************************************************************
 * 
 ****************************************************************************/
static int
region_del_cmdfunc(int argc, char *argv[])
{
	int index;

	if (argc != 2)
		return -H_EUSAGE;

	if (scan(argv[1], &index))
		return -H_EINVAL;

	if (index < 0 || MAX_REGION_NUM - 1 < index)
		return -H_EINVAL;

	switch (regions[index].mode) {
	case MAP_MODE_DYNAMIC:
		SET_MAP_NAME(index, "");
		SET_MAP_TYPE(index, MAP_TYPE_UNKNOWN);
	case MAP_MODE_RESIZABLE:
		SET_MAP_START(index, 0);
		SET_MAP_SIZE(index, 1);
		SET_MAP_BLOCK(index, "");
		break;
	case MAP_MODE_STATIC:
		hprintf("can't delete this region: %d\n", index);
		return 0;
	default:
		break;
	}

	return 0;
}

/****************************************************************************
 * 
 ****************************************************************************/
static int
region_save_cmdfunc(int argc, char *argv[])
{
	/* this function not yet implemented. */
	return 0;
}

/****************************************************************************
 * 
 ****************************************************************************/
const command_t region_print_command =
	{ "print", 0, "display region information", &region_print_cmdfunc };
const char add_help[] = "add region\n\tindex : region index\n\tname  : region name\n\ttype  : {1:FLASH,2:RAM}\n\tstart : region start address\n\tsize  : region size\n\tblock : block information";
const command_t region_add_command =
	{ "add", "<index> <name> <type> <start> <size> [<block>]", add_help, &region_add_cmdfunc };
const command_t region_del_command =
	{ "del", "<index>", "delete region", &region_del_cmdfunc };
const command_t region_save_command =
	{ "save", 0, "save region", &region_save_cmdfunc };

/****************************************************************************
 * 
 ****************************************************************************/
static const command_t *region_commands[] = {
        &help_command,
        &quit_command,
        &region_print_command,
        &region_add_command,
        &region_del_command,
        &region_save_command,
        0
};

/****************************************************************************
 * 
 ****************************************************************************/
ABBR(help_command, '?')
ABBR(quit_command, 'q')
ABBR(region_print_command, 'p')
ABBR(region_add_command, 'a')
ABBR(region_del_command, 'd')
ABBR(region_save_command, 's')

/****************************************************************************
 * 
 ****************************************************************************/
static const abbrev_t *region_abbrevs[] = {
	&help_command_abbrev,
	&quit_command_abbrev,
	&region_print_command_abbrev,
	&region_add_command_abbrev,
	&region_del_command_abbrev,
	&region_save_command_abbrev,
	0
};

/****************************************************************************
 * 
 ****************************************************************************/
static command_list_t region_command_list =
	{ region_commands, region_abbrevs, "region", 0 };

/****************************************************************************
 * 
 ****************************************************************************/
static int 
region_cmdfunc(int argc, char *argv[])
{
	do_command_loop(&region_command_list);
	return 0;
}

/****************************************************************************
 * 
 ****************************************************************************/
const command_t region_command =
	{ "region", 0, "", &region_cmdfunc };

COMMAND(region_command);
#endif





