/**
 * A command to calculate md5 hash for given area.
 */
#include <target/buffer.h>
#include <target/io.h>
#include <target/herrno.h>
#include <target/scan.h>
#include <target/command.h>
#include <target/flash.h>
#include <target/mmu.h>
#include <target/mem.h>
#include "common/md5.h"
#include "arch/memregions.h"

static int in_flash_region(addr_t addr)
{
	return (FLASH_START <= addr && addr < FLASH_START + FLASH_SIZE) ? 1 : 0;
}

static size_t flash_copy_to_dram_wrap(const void * from, void * to, size_t size)
{
	flash_copy_to_dram((const u32)from, (const u32)to, (size % 2) ? size + 1 : size);

	return size;
}

static size_t dram_copy_to_dram(const void * from, void * to, size_t size)
{
	memcpy(to, from, size);

	return size;
}

static int calc_md5(addr_t base, size_t len, void *resblock)
{
	struct md5_ctx ctx;
	char buffer[BLOCKSIZE];
	addr_t p;
	addr_t end = base + len;
	size_t remain;
	size_t (*copy_to_dram)(const void *, void *, size_t);

	copy_to_dram = in_flash_region(base) ? flash_copy_to_dram_wrap : dram_copy_to_dram;

	md5_init_ctx(&ctx);

	for (p = base; p + BLOCKSIZE <= end; p += BLOCKSIZE) {
		if (copy_to_dram((const char *)p, buffer, BLOCKSIZE) != BLOCKSIZE) {
			return -H_EIO;
		}

		md5_process_block(buffer, BLOCKSIZE, &ctx);
	}

	remain = end - p;
	if (remain > 0) {
		/* process partial block */
		if (copy_to_dram((const char *)p, buffer, remain) != remain) {
			return -H_EIO;
		}

		md5_process_bytes(buffer, remain, &ctx);
	}

	md5_finish_ctx(&ctx, resblock);

	return 0;
}

static int md5sum_cmdfunc(int argc, char *argv[])
{
	addr_t addr;
	size_t size;
	unsigned char val[16];
	int ret;

	if (argc != 3)
		return -H_EUSAGE;
	if (scan(*++argv, &addr))
		return -H_EADDR;
	if (scan(*++argv, &size))
		return -H_EINVAL;
	if (in_flash_region(addr) && (addr & 1))
		return -H_EALIGN;

	boost_on(BOOST_LINUX_MODE);
	ret = calc_md5(addr, size, val);
	boost_off();

	if (ret != 0)
		return ret;

	hprintf("%b%b%b%b%b%b%b%b%b%b%b%b%b%b%b%b\n",
		val[0], val[1], val[2], val[3],
		val[4], val[5], val[6], val[7],
		val[8], val[9], val[10], val[11],
		val[12], val[13], val[14], val[15]);

	return 0;
}

const command_t md5sum_command = { "md5sum", "<addr> <size>",
	"calculate md5 hash",
	&md5sum_cmdfunc
};

COMMAND(md5sum_command);
