#include <target/io.h>
#include <target/net/eth.h>
#include <target/net/eth_util.h>
#include <target/net/udp.h>
#include <target/net/tftp.h>

#ifdef DEBUG_ETH
#define DEBUG_INFO(args...) hprintf(args)
#else
#define DEBUG_INFO(args...)
#endif

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// 
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
static int create_read_packet(const char *filename, const char *mode,
			      unsigned char *pbuf, unsigned int pbuflen){
  int ptr = 0;
  int filename_len = safe_strlen(filename);
  int mode_len = safe_strlen(mode);
  unsigned short opcode = htons(TFTP_OP_READ);

  if(pbuflen < (2 + filename_len + 1 + mode_len + 1)){
    return -1;
  }

  safe_memcpy(&pbuf[ptr], &opcode, 2);
  ptr += 2;

  safe_memcpy(&pbuf[ptr], filename, filename_len);
  ptr += filename_len;

  pbuf[ptr] = 0;
  ptr += 1;

  safe_memcpy(&pbuf[ptr], mode, mode_len);
  ptr += mode_len;

  pbuf[ptr] = 0;
  ptr += 1;

  return ptr;
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// 
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
static int create_ack_packet(const unsigned short __block,
			     unsigned char *pbuf, unsigned int pbuflen){
  int ptr = 0;
  unsigned short opcode = htons(TFTP_OP_ACK);
  unsigned short block = htons(__block);

  if(pbuflen < 4){
    return -1;
  }

  safe_memcpy(&pbuf[ptr], &opcode, 2);
  ptr += 2;

  safe_memcpy(&pbuf[ptr], &block, 2);
  ptr += 2;

  return ptr;
}

//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// 
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
int tftp_get(const unsigned char *server_ipaddr,
	     const char *filename,
	     unsigned int *__filesize,
	     void *__buf, unsigned int buflen){
  int ret;
  int flag = 0;
  unsigned char send_pbuf[1024];
  int send_pbuflen;
  unsigned char *recv_pbuf;
  int recv_pbuflen;
  unsigned short port;
  
  unsigned short opecode;
  unsigned short block;
  unsigned char *rcv_data;
  unsigned int rcv_data_len;
  unsigned char *filebuf = (unsigned char *)__buf;
  unsigned int filesize = 0;
  int progress = 0;
  int req_block = 1;

  hprintf("Filename : %s\n", filename);
  send_pbuflen = create_read_packet(filename, MODE_BIN, send_pbuf, 1024);
  if(send_pbuflen == -1){
    return -1;
  }

  ret = udp_send(server_ipaddr, TFTP_PORT_NO, 30000, send_pbuf, send_pbuflen);
  if(ret == -1){
    return -1;
  }

  do{
    if((progress++ % 10) == 0){
      hprintf(".");//progress
    }

    ret = udp_proc(server_ipaddr, 30000, &port,
		   (void *)&recv_pbuf, &recv_pbuflen);
    if(ret == -1){
      return -1;
    }
    
    opecode = ntohs(*(unsigned short *)(recv_pbuf + 0));
    switch(opecode){
    case TFTP_OP_DATA:
      block = ntohs(*(unsigned short *)(recv_pbuf + 2));

      if(req_block != block){
	DEBUG_INFO("receive unknown block: %d (request: %d)\n",
		   req_block, block);
	continue;
      }else{
	req_block++;
      }

      rcv_data_len = recv_pbuflen - 4;
      rcv_data = recv_pbuf + 4;
      
      //Ack
      send_pbuflen = create_ack_packet(block, send_pbuf, 1024);
      if(send_pbuflen == -1){
	return -1;
      }
      ret = udp_send(server_ipaddr, port, 30000, send_pbuf, send_pbuflen);
      
      if(rcv_data_len < 512){
	flag = 1;
      }
      
      if(rcv_data_len == 0){
	break;
      }
      
      safe_memcpy(&filebuf[filesize], rcv_data, rcv_data_len);

      filesize += rcv_data_len;
      break;
    case TFTP_OP_ERROR:
      {
	unsigned short errcode;
	unsigned char *errmsg;
	errcode = ntohs(*(unsigned short *)(recv_pbuf + 2));
	errmsg = recv_pbuf + 4;
	hprintf("\ntftp: [%d]: %s\n", errcode, errmsg);
	return -2;
      }
    default:
      break;//continue;
    }
  }while(flag == 0);
  
  hprintf("\n");
  hprintf("Filesize : %d\n", filesize);
  hprintf("\n");
  *__filesize = filesize;

  return 0;
}

