﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Net;
using System.IO;
using Codeplex.Data;

namespace AssistRB
{
    /// <summary>
    /// 定期的に通信し、転送のステータスを監視します
    /// </summary>
    class TransferStatusChecker
    {
        private static TransferStatusChecker instance = null;

        public delegate void TransferStatusChangedDelegate(bool processing);

        /// <summary>
        /// 処理中と完了が切り替わったときのイベントです
        /// </summary>
        public event TransferStatusChangedDelegate TransferStatusChanged;

        private volatile bool stop = false;
        private volatile bool wait = true;
        private string host_url = "";
        private bool isProcessing = true;

        /// <summary>
        /// インスタンスを取得します
        /// </summary>
        /// <returns>TransferStatusCheckerを返します</returns>
        public static TransferStatusChecker GetInstance()
        {
            if (instance == null)
            {
                instance = new TransferStatusChecker();
            }

            return instance;
        }

        private TransferStatusChecker()
        {
            // GetInstance()以外での生成を抑制
        }

        /// <summary>
        /// ホストを設定します
        /// </summary>
        /// <param name="url">http://xxxxx:xxxxの形式で指定します。末尾に/は不要です。</param>
        public void Initializer(string url)
        {
            this.host_url = url;
        }

        /// <summary>
        /// スレッドで起動するメソッドです
        /// </summary>
        public void worker()
        {
            while (!this.stop)
            {
                if (host_url.Length > 0 && !wait)
                {
                    this.Update();
                }

                Thread.Sleep(1 * 1000); // 1s
            }
        }

        /// <summary>
        /// 処理中かどうかのフラグです
        /// </summary>
        public bool IsProcessing
        {
            get
            {
                return this.isProcessing;
            }

            private set
            {
                if (this.isProcessing != value)
                {
                    this.isProcessing = value;

                    if (this.isProcessing)
                    {
                        ToolLog.GetInstance().Add("処理中です");
                    }
                    else
                    {
                        ToolLog.GetInstance().Add("処理を完了しました");
                    }

                    if (this.TransferStatusChanged != null)
                    {
                        this.TransferStatusChanged(this.isProcessing);
                    }
                }
            }
        }

        /// <summary>
        /// 現在の状態を取得します
        /// </summary>
        private void Update()
        {
            string tool_api_url = host_url + "/dms/transfer_tool/api";
            string uri = tool_api_url + "/transfer_status";

            WebRequest request = WebRequest.Create(uri);
            WebResponse responce;
            try
            {
                responce = request.GetResponse();
            }
            catch (Exception)
            {
                return;
            }
            Stream stream = responce.GetResponseStream();
            if (stream == null)
            {
                responce.Close();
                return;
            }

            var json = DynamicJson.Parse(stream);
            stream.Close();
            responce.Close();

            bool error = false;
            bool processing = false;

            if (json != null)
            {
                var result = json.results;
                if (result != null && result.IsArray)
                {
                    foreach (var item in result)
                    {
                        if (item.IsDefined("status"))
                        {
                            switch (item.status as string)
                            {
                                case "waiting":
                                    processing = true;
                                    break;
                                case "transferring":
                                    processing = true;
                                    break;
                                case "cancelled":
                                    break;
                                case "stopped":
                                    break;
                                case "error":
                                    error = true;
                                    break;
                                case "unsupported":
                                    error = true;
                                    break;
                                case "done":
                                    break;
                                default:
                                    break;
                            }
                        }
                    }
                }
            }

            if (processing)
            {
                this.IsProcessing = true;
            }
            else
            {
                this.IsProcessing = false;
            }
        }

        /// <summary>
        /// 処理の終了を要求します
        /// </summary>
        public void Terminate()
        {
            this.stop = true;
        }

        /// <summary>
        /// 監視を開始します。
        /// </summary>
        public void Start()
        {
            this.wait = false;
        }

        /// <summary>
        /// 監視を中断します。
        /// </summary>
        public void Stop()
        {
            this.wait = true;
        }
    }
}
