/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.sh;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

public class ShRootShellEnvironment implements ShEnvironment {

	private ShEnvironment env;
	private Map<String, String> map;
	private Map<String, ShTree> functions;
	private Map<String, Void> flags;
	private Map<String, Attributes> attrs;
	private EnumMap<ShSignal, String> traps;

	public ShRootShellEnvironment(ShEnvironment env) {
		if(env == null)  throw new NullPointerException();
		this.env   = env;
		this.map   = new HashMap<String, String>();
		functions  = new HashMap<String, ShTree>();
		this.flags = new HashMap<String, Void>();
		this.attrs = new HashMap<String, Attributes>();
		this.traps = new EnumMap<ShSignal, String>(ShSignal.class);
	}

	public String find(String name) {
		String v;

//		return (v = map.get(name)) != null ? v : env.find(name);
		if(name == null) {
			throw new NullPointerException();
		} else if(name.equals("RANDOM")) {
			return ((int)(Math.random() * 32768)) + "";
		} else if(name.equals("$") && !contains("$")) {
			v = ShProcessNumber.getNumber() + "";
			put("$", v);
			return v;
		} else if((v = map.get(name)) != null) {
			return v;
		} else if(env.contains(name)) {
			return env.find(name);
		} else if(isSet("nounset")) {
			throw new ShRuntimeException();
		} else {
			return "";
		}
	}

	public void bind(String name, String value) {
		if(isReadonly(name)) {
			throw new ShRuntimeException();
		} else if(map.containsKey(name) || !env.contains(name)) {
			map.put(name, value);
		} else {
			env.put(name, value);
		}
	}

	public void put(String name, String value) {
		map.put(name, value);
	}

	public ShEnvironment getEnvironment() {
		return env;
	}

	public boolean contains(String name) {
		return map.containsKey(name) || env.contains(name);
	}

	public boolean isEnvironment() {
		return false;
	}

	public List<String> getPath() {
		return env.getPath();
	}

	public void bind(String name, ShTree function) {
		functions.put(name, function);
	}

	public void put(String name, ShTree function) {
		functions.put(name, function);
	}

	public ShTree findFunction(String name) {
		return functions.get(name);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#export(java.lang.String)
	 */
	public void export(String name) {
		if(map.containsKey(name)) {
			env.bind(name, map.get(name));
			map.remove(name);
		} else if(!env.contains(name)) {
			env.bind(name, "");
		}
	}

	public boolean isSet(String name) {
		return flags.containsKey(name);
	}

	public void set(String name) {
		flags.put(name, null);
	}

	public void reset(String name) {
		flags.remove(name);
	}

	public void set(String name, boolean value) {
		if(value) {
			flags.put(name, null);
		} else {
			flags.remove(name);
		}
	}

	public void unbind(String name) {
		if(map.containsKey(name)) {
			map.remove(name);
		} else {
			env.unbind(name);
		}
	}

	public boolean isReadonly(String name) {
		Attributes a;

		return (isSystem(name) ||
				(map.containsKey(name) &&
						(a = attrs.get(name)) != null &&
						a.isReadonly()) ||
				(!map.containsKey(name) && env.isReadonly(name)));
	}

	public boolean isSystem(String name) {
		return (name.matches("[0-9]+") || name.equals("@") ||
				name.equals("*")       || name.equals("$"));
	}

	public void setReadonly(String name) {
		if(map.containsKey(name)) {
			attrs.put(name, READONLY);
		} else {
			env.setReadonly(name);
		}
	}

	public Properties toProperties() {
		Properties p = env.toProperties();

		for(Map.Entry<String, String> e : map.entrySet()) {
			p.setProperty(e.getKey(), e.getValue());
		}
		return p;
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#getTrap(net.morilib.sh.ShSignal)
	 */
	public String getTrap(ShSignal signal) {
		return traps.get(signal);
	}

	/* (non-Javadoc)
	 * @see net.morilib.sh.ShEnvironment#setTrap(net.morilib.sh.ShSignal, java.lang.String)
	 */
	public void setTrap(ShSignal signal, String cmd) {
		traps.put(signal, cmd);
	}

}
