/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.value;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Collection;
import java.util.regex.Pattern;

import net.morilib.awk.expr.AwkCastException;

public class AwkFloat implements AwkValue {

	//
	private double value;

	private AwkFloat(double d) {
		value = d;
	}

	public static AwkFloat valueOf(double d) {
		return new AwkFloat(d);
	}

	public double toFloat() {
		return value;
	}

	public BigInteger toInteger() {
		return new BigDecimal(value).toBigInteger();
	}

	public boolean toBoolean() {
		return value != 0.0;
	}

	public Number toReal() {
		return new Double(value);
	}

	public Pattern toRegex() {
		throw new AwkCastException();
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.value.AwkValue#toObject()
	 */
	public Object toObject() {
		return new Double(value);
	}

	public AwkValue referArray(AwkValue v) {
		throw new AwkCastException();
	}

	public AwkValue putArray(AwkValue k, AwkValue v) {
		throw new AwkCastException();
	}

	public AwkValue deleteArray(AwkValue v) {
		throw new AwkCastException();
	}

	public Collection<AwkValue> values() {
		throw new AwkCastException();
	}

	public boolean isFloat() {
		return true;
	}

	public boolean isInteger() {
		return false;
	}

	public boolean isReal() {
		return true;
	}

	public boolean isFloatValue() {
		return true;
	}

	public boolean isIntegerValue() {
		try {
			new BigDecimal(value).toBigIntegerExact();
			return true;
		} catch(ArithmeticException e) {
			return false;
		}
	}

	public boolean isRealValue() {
		return true;
	}

	public boolean isString() {
		return false;
	}

	public boolean isRegex() {
		return false;
	}

	public boolean isFunction() {
		return false;
	}

	public int hashCode() {
		return (int)Double.doubleToRawLongBits(value);
	}

	public boolean equals(Object o) {
		if(o instanceof AwkFloat) {
			return value == ((AwkFloat)o).value;
		}
		return false;
	}

	public String toString() {
		return Double.toString(value);
	}

	public int compareTo(AwkValue o) {
		if(o instanceof AwkUndefined) {
			return 1;
		} else if(o.isFloat() || o.isInteger()) {
			return value < o.toFloat() ?
					-1 : value > o.toFloat() ? 1 : 0;
		} else {
			return toString().compareTo(o.toString());
		}
	}

}
