/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.awk.expr;

import net.morilib.awk.AwkLocation;
import net.morilib.awk.io.AwkFiles;
import net.morilib.awk.namespace.AwkNamespace;
import net.morilib.awk.value.AwkUndefined;
import net.morilib.awk.value.AwkValue;

/**
 * 変数参照を表す中間表現です。
 * 
 * @author MORIGUCHI, Yuichiro 2013/03
 */
public class AwkReferVariable extends AwkExpression {

	private AwkLocation name;

	/**
	 * 中間表現を生成します。
	 * 
	 * @param name 変数の場所
	 */
	public AwkReferVariable(AwkLocation name) {
		this.name = name;
	}

	/**
	 * 変数の場所を得ます。
	 */
	public AwkLocation getLocation() {
		return name;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#eval(net.morilib.awk.namespace.AwkNamespace, net.morilib.awk.io.AwkFiles)
	 */
	public AwkValue eval(AwkNamespace ns, AwkFiles o) {
		AwkNamespace s2 = ns, t;
		AwkValue v, w;

		for(int i = 0; i < name.sizeOfPath() - 1; i++) {
			if((t = s2.referNamespace(name.getPathOf(i))) != null) {
				s2 = t;
			} else if((w = s2.find(name.getPathOf(i))) != null &&
					w.getNamespace() != null) {
				s2 = w.getNamespace();
			} else {
				return AwkUndefined.UNDEF;
			}
		}
		v = s2.find(name.getName());
		return v != null ? v : AwkUndefined.UNDEF;
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#assign(net.morilib.awk.namespace.AwkNamespace, net.morilib.awk.value.AwkValue, net.morilib.awk.io.AwkFiles)
	 */
	@Override
	public void assign(AwkNamespace ns, AwkValue v, AwkFiles o) {
		AwkNamespace s2 = ns;

		for(int i = 0; i < name.sizeOfPath() - 1; i++) {
			s2 = s2.subnamespace(name.getPathOf(i));
		}
		s2.assign(name.getName(), v);
	}

	/* (non-Javadoc)
	 * @see net.morilib.awk.expr.AwkExpression#isLvalue()
	 */
	@Override
	public boolean isLvalue() {
		return true;
	}

}
