﻿using System;
using System.IO;
using System.Collections.Generic;
using System.Windows.Forms;
using Microsoft.VisualStudio.Tools.Applications.Runtime;
using Excel = Microsoft.Office.Interop.Excel;
using Office = Microsoft.Office.Core;
using ImageConverter;
using BTS.Backlog;
using BTS.Backlog.Forms;

namespace ExcelBacklogAddIn
{
    public partial class ThisAddIn
    {
        // メニューバー
        private const string commandBarName = "ExcelBacklogAddIn";

        private Office.CommandBarButton btnCreateNewWookbook = null;
        private Office.CommandBarButton btnDownload = null;
        private Office.CommandBarButton btnUpload = null;
        private Office.CommandBarButton btnOption = null;
        private Office.CommandBar menubar = null;

        // シート
        private Excel.Worksheet TracData = null;
        private Excel.Worksheet ImportData = null;

        // オプション
        private ExcelBacklogAddInOption Option = new ExcelBacklogAddInOption();
        private string OptionPath;

        // Trac I/F
        //private XmlRpcStruct[] FieldNames = null;

        private bool EnableID;              // IDの有効/無効状態
        private bool IsDownload = false;    // ダウンロード状態

        private Project project = null;     // 対象プロジェクト

        private static Dictionary<string, string> items = new Dictionary<string, string>
        {
            { "ID", Issue.AttributeNames.ID },
            { "件名", Issue.AttributeNames.SUMMARY },
            { "詳細", Issue.AttributeNames.DESCRIPTION },
            { "種別", Issue.AttributeNames.ISSUE_TYPE },
            { "カテゴリー", Issue.AttributeNames.COMPONENT },
            { "発生バージョン", Issue.AttributeNames.VERSION },
            { "マイルストーン", Issue.AttributeNames.MILESTONE },
            { "優先度", Issue.AttributeNames.PRIORITY },
            { "状態", Issue.AttributeNames.STATUS },
            { "完了理由", Issue.AttributeNames.STATUS },
            { "期限日", Issue.AttributeNames.DUE_DATE },
        };

        private void ThisAddIn_Startup( object sender, System.EventArgs e )
        {
            #region VSTO で生成されたコード

            this.Application = (Excel.Application)Microsoft.Office.Tools.Excel.ExcelLocale1033Proxy.Wrap(typeof(Excel.Application), this.Application);

            #endregion

            try {
                // メニューバーを追加
                menubar = Application.CommandBars.Add( commandBarName, Microsoft.Office.Core.MsoBarPosition.msoBarTop, null, false );

                // 作成ボタン
                btnCreateNewWookbook = (Office.CommandBarButton)menubar.Controls.Add( 1, missing, missing, missing, missing );
                btnCreateNewWookbook.Style = Office.MsoButtonStyle.msoButtonIcon;
                btnCreateNewWookbook.FaceId = 1;
                btnCreateNewWookbook.Tag = "ExcelBacklogAddIn1";
                btnCreateNewWookbook.Picture = ImageConverter.ImageConverter.getImage( Properties.Resources.file_add );
                btnCreateNewWookbook.TooltipText = "雛形を作成";
                btnCreateNewWookbook.Click += new Office._CommandBarButtonEvents_ClickEventHandler( CommandBarButton_CreateClick );


                // ダウンロードボタン
                btnDownload = (Office.CommandBarButton)menubar.Controls.Add( 1, missing, missing, missing, missing );
                btnDownload.Style = Office.MsoButtonStyle.msoButtonIcon;
                btnDownload.FaceId = 3;
                btnDownload.Tag = "ExcelBacklogAddIn3";
                btnDownload.Picture = ImageConverter.ImageConverter.getImage( Properties.Resources.download );
                btnDownload.TooltipText = "チケットの一覧を Backlog から取得";
                btnDownload.Click += new Office._CommandBarButtonEvents_ClickEventHandler( CommandBarButton_DownloadClick );

                // アップロードボタン
                btnUpload = (Office.CommandBarButton)menubar.Controls.Add( 1, missing, missing, missing, missing );
                btnUpload.Style = Office.MsoButtonStyle.msoButtonIcon;
                btnUpload.FaceId = 4;
                btnUpload.Tag = "ExcelBacklogAddIn4";
                btnUpload.Picture = ImageConverter.ImageConverter.getImage( Properties.Resources.upload );
                btnUpload.TooltipText = "チケットの一覧を Backlog へ送信";
                btnUpload.Click += new Office._CommandBarButtonEvents_ClickEventHandler( CommandBarButton_UploadClick );

                // 設定ボタン
                btnOption = (Office.CommandBarButton)menubar.Controls.Add( 1, missing, missing, missing, missing );
                btnOption.Style = Office.MsoButtonStyle.msoButtonIcon;
                btnOption.FaceId = 2;
                btnOption.Tag = "ExcelBacklogAddIn2";
                btnOption.Picture = ImageConverter.ImageConverter.getImage( Properties.Resources.options );
                btnOption.TooltipText = "設定";
                btnOption.Click += new Office._CommandBarButtonEvents_ClickEventHandler( CommandBarButton_OptionClick );

                OptionPath = Environment.GetFolderPath( Environment.SpecialFolder.ApplicationData ) + "\\ExcelBacklogAddInOption.xml";
                if ( File.Exists( OptionPath ) ) {
                    Option.Load( OptionPath, ref menubar );
                }
            }
            catch ( Exception ex ) {
                MessageBox.Show( ex.Message );
            }
        }

        private void ThisAddIn_Shutdown(object sender, System.EventArgs e)
        {
            // メニューを保存
            Option.Save( OptionPath, menubar );

            // メニューバーを削除
            menubar.Delete();
            menubar = null;
        }

        /// <summary>
        /// 設定ボタン
        /// </summary>
        /// <param name="Ctrl"></param>
        /// <param name="CancelDefault"></param>
        void CommandBarButton_OptionClick( Office.CommandBarButton Ctrl, ref bool CancelDefault )
        {
            try {
                FormOption form = new FormOption( Option );
                DialogResult ret = form.ShowDialog();
                if ( ret == DialogResult.OK ) {
                    // Backlog インタフェースを閉じる
                    Backlog.Close();

                    // オプションの保存
                    Option = form.Option;
                    Option.Save( OptionPath, menubar );
                }
            }
            catch ( Exception ex ) {
                MessageBox.Show( ex.Message );
            }
        }

        /// <summary>
        /// ひな形を作成ボタン
        /// </summary>
        /// <param name="Ctrl"></param>
        /// <param name="CancelDefault"></param>
        void CommandBarButton_CreateClick( Office.CommandBarButton Ctrl, ref bool CancelDefault )
        {
            try {
                // サーバに接続
                Login();

                // 新規ワークブックを作成
                CreateNewWookbooks( false );
            }
            catch ( Exception ex ) {
                Backlog.Close();
                MessageBox.Show( ex.Message );
            }
        }

        /// <summary>
        /// Tracからダウンロードボタン
        /// </summary>
        /// <param name="Ctrl"></param>
        /// <param name="CancelDefault"></param>
        void CommandBarButton_DownloadClick( Office.CommandBarButton Ctrl, ref bool CancelDefault )
        {
            try {
                // サーバに接続
                Login();

                // 新規ワークブックを作成し、チケット一覧を取得
                IsDownload = true;
                CreateNewWookbooks( true );
                DownloadTicket();
            }
            catch ( Exception ex ) {
                Backlog.Close();
                MessageBox.Show( ex.Message );
            }
            finally {
                // ダウンロード完了
                IsDownload = false;
            }
        }

        /// <summary>
        /// Tracへ送信ボタン
        /// </summary>
        /// <param name="Ctrl"></param>
        /// <param name="CancelDefault"></param>
        void CommandBarButton_UploadClick( Office.CommandBarButton Ctrl, ref bool CancelDefault )
        {
            try {
                // 確認
                if ( Option.UploadVerify ) {
                    DialogResult result = MessageBox.Show( Option.SelectedAccount.Url + " へチケットを登録します。よろしいですか？",
                                                        "", MessageBoxButtons.YesNo, MessageBoxIcon.Question,
                                                        MessageBoxDefaultButton.Button2 );
                    if ( result != DialogResult.Yes ) {
                        return;
                    }
                }

                // サーバに接続
                Login();

                // チケット一覧を Trac へ送信 - ID あり -> 更新
                if ( EnableID ) {
                    UpdateTicket();
                }
                // チケット一覧を Trac へ送信 - ID なし -> 新規作成
                else {
                    CreateTicket();
                }
            }
            catch ( Exception ex ) {
                Backlog.Close();
                MessageBox.Show( ex.Message );
            }
        }

        /// <summary>
        /// ログイン
        /// </summary>
        /// <returns></returns>
        private void Login()
        {
            if ( Backlog.IsConnected ) {
                return;
            }

            // サーバー未設定
            if ( Option.Selected == -1 ) {
                throw new Exception( "サーバ情報を設定してください" );
            }

            // Backlog へのアクセス I/F を作成
            Backlog.Connect( Option.SelectedAccount.Url,
                Option.SelectedAccount.UserName, Option.SelectedAccount.Password );
        }

        /// <summary>
        /// ひな形を作成
        /// </summary>
        /// <param name="enableid">IDを有効にするかどうか</param>
        private void CreateNewWookbooks( bool enableid )
        {
            // ワークブックの作成
            Excel.Workbook wb = this.Application.Workbooks.Add( Excel.XlWBATemplate.xlWBATWorksheet );
            wb.Sheets.Add( missing, missing, 1, Excel.XlWBATemplate.xlWBATWorksheet );


            TracData = (Excel.Worksheet)wb.Sheets[1];
            TracData.Name = "インポートデータ";
            TracData.Cells.Clear();

            ImportData = (Excel.Worksheet)wb.Sheets[2];
            ImportData.Name = "Backlogデータ";
            ImportData.Cells.Clear();

            // 日付用ダブルクリックイベント設定
            TracData.BeforeDoubleClick += new Excel.DocEvents_BeforeDoubleClickEventHandler( Sheet1_BeforeDoubleClick );

            // デフォルト値入力用セルチェンジイベント設定
            TracData.Change += new Excel.DocEvents_ChangeEventHandler( Sheet1_ChangeEventHandler );

            // Ticket I/F を作成
            //TracProxy.ITicket ticket = trac.Ticket;
            //object[] obj = ticket.GetTicketFields();
            //FieldNames = new XmlRpcStruct[obj.Length];
            //for ( int i = 0; i < FieldNames.Length; ++i ) {
            //    FieldNames[i] = (XmlRpcStruct)obj[i];
            //}

            // ID のスペースを作成
            this.EnableID = enableid;

            // カラムは1から開始
            int column = 0;
            foreach ( string key in items.Keys ) {
            //for ( int i = 0; i < items.Keys.Count; ++i ) {
                // IDが無効の場合は飛ばす
                if ( !this.EnableID && (key == "ID") ) {
                    continue;
                }

                ++column;

                // 表示名を設定
                TracData.Cells[1, column] = ImportData.Cells[1, column] = key;

                // コメント設定
                //Excel.Range cell = (Excel.Range)TracData.Cells[1, column];
                //cell.AddComment( (string)st["label"] );

                //// 表示名の要素を取得
                //GetAllitem getAllItem = null;
                //if ( name == TracProxy.TicketAttributes.Milestone ) {
                //    getAllItem = new GetAllitem( ticket.GetAllMilestones );
                //}
                //else if ( name == TracProxy.TicketAttributes.Component ) {
                //    getAllItem = new GetAllitem( ticket.GetAllComponents );
                //}
                //else if ( name == TracProxy.TicketAttributes.Priority ) {
                //    getAllItem = new GetAllitem( ticket.GetAllPriorities );
                //}
                //else if ( name == TracProxy.TicketAttributes.Resolution ) {
                //    getAllItem = new GetAllitem( ticket.GetAllResolutions );
                //}
                //else if ( name == TracProxy.TicketAttributes.Type ) {
                //    getAllItem = new GetAllitem( ticket.GetAllTypes );
                //}
                //else if ( name == TracProxy.TicketAttributes.Version ) {
                //    getAllItem = new GetAllitem( ticket.GetAllVersions );
                //}
                //else if ( name == TracProxy.TicketAttributes.Status ) {
                //    getAllItem = new GetAllitem( ticket.GetAllStatus );
                //}

                //// 幅を文字数（半角？）で指定
                //if ( name == TracProxy.TicketAttributes.Summary ) {
                //    cell.ColumnWidth = 30;
                //}
                //else if ( name == TracProxy.TicketAttributes.Description ) {
                //    cell.ColumnWidth = 50;
                //}
                //else if ( name == TracProxy.TicketAttributes.DueAssign ) {
                //    cell.ColumnWidth = 12;
                //}
                //else if ( name == TracProxy.TicketAttributes.DueClose ) {
                //    cell.ColumnWidth = 12;
                //}

                //// 入力規則の作成
                //if ( getAllItem != null ) {
                //    string[] items = getAllItem();
                //    for ( int j = 0; j < items.Length; ++j ) {
                //        ImportData.Cells[j + 2, column] = items[j];
                //    }

                //    // シートをまたぐ入力規則用に複数セルに名前をつける
                //    string dataName = "データ" + columnBase.ToString();
                //    ImportData.get_Range( ImportData.Cells[2, column], ImportData.Cells[(items.Length + 1).ToString(), column] ).Name = dataName;

                //    // コンボボックスの作成
                //    char cellName = (char)('A' + columnBase);
                //    string start = cellName + "2";
                //    string end = cellName + "100";

                //    //// 別シートの一覧を元に入力規則を作る
                //    TracData.get_Range( start, end ).Validation.Add( Excel.XlDVType.xlValidateList, Excel.XlDVAlertStyle.xlValidAlertStop,
                //            Excel.XlFormatConditionOperator.xlEqual, "=" + dataName, 0 );
                //}
            }
        }

        /// <summary>
        /// 指定されたカラムを探す
        /// </summary>
        /// <param name="columnName">探すカラム名</param>
        /// <returns>カラムのインデックス</returns>
        private int FindColumn( string columnName )
        {
            string name;

            int column = 1;
            do {
                name = (string)((Excel.Range)TracData.Cells[1, column]).Value2;
                if ( name == columnName ) {
                    break;
                }

                ++column;
            } while ( name != null );
            if ( name == null ) {
                throw new Exception( columnName + "が見つかりません" );
            }

            return column;
        }

        /// <summary>
        /// セルの内容を課題に変換する
        /// </summary>
        /// <param name="issue"></param>
        /// <param name="cells"></param>
        /// <param name="raw"></param>
        public void CellToIssue( ref Issue issue, Excel.Range cells, int raw )
        {
            foreach ( string key in items.Keys ) {
                // ID が無効で、キーがIDの場合は次へ
                if ( !EnableID &&  (key == "ID") ) {
                    continue;
                }

                string value = ((Excel.Range)TracData.Cells[raw, FindColumn( key )]).Text.ToString();
                switch ( key ) {
                case "ID":
                    issue.Id = int.Parse( value );
                    break;
                case "件名":
                    issue.Summary = value;
                    break;
                case "詳細":
                    issue.Description = value;
                    break;
                case "種別":
                    break;
                case "カテゴリー":
                    //issue.Component = new string[] { value };
                    break;
                case "発生バージョン":
                    //issue.Version = new string[] { value };
                    break;
                case "マイルストーン":
                    //issue.Milestone = new string[] { value };
                    break;
                case "優先度":
                    break;
                case "状態":
                    break;
                case "完了理由":
                    break;
                case "期限日":
                    break;
                }
            }
        }

        /// <summary>
        /// チケットの新規登録
        /// </summary>
        /// <param name="cells">セル</param>
        /// <param name="raw">登録する行</param>
        private void CreateTicket( Project project, Excel.Range cells, int raw )
        {
            // セルの内容を課題に変換して登録
            Issue issue = new Issue();
            CellToIssue( ref issue, cells, raw );
            project.CreateIssue( issue );
        }

        /// <summary>
        /// チケットの更新
        /// </summary>
        /// <param name="cells">セル</param>
        /// <param name="raw">登録する行</param>
        private void UpdateTicket( Project project, Excel.Range cells, int raw, string comment )
        {
            // ID から現在登録されているチケットを取得
            int id = int.Parse( ((Excel.Range)TracData.Cells[raw, FindColumn( "ID" )]).Text.ToString() );
            Issue baseIssue = new Issue( id );

            /// 簡単にディープコピーができないので、もう一度サーバーから引いてくる
            //Issue workIssue = DeepCopyClone.DeepCopy<Issue>( baseIssue );
            Issue workIssue = new Issue( id );

            // セルを課題に変更して更新
            CellToIssue( ref workIssue, cells, raw );

            // 更新があれば反映
            if ( !baseIssue.Equals( workIssue ) ) {
                workIssue.Update( comment );
            }
        }

        /// <summary>
        /// プロジェクトの選択
        /// </summary>
        /// <returns></returns>
        private Project SelectProject()
        {
            List<Project> projects = Project.GetAll();
            FormSelectProject selectProject = new FormSelectProject( new SelectProject( projects, -1 ) );
            DialogResult ret = selectProject.ShowDialog();
            if ( ret != DialogResult.OK ) {
                throw new Exception( "プロジェクトが選択されませんでした" );
            }

            return projects[selectProject.SelectedIndex];
        }

        /// <summary>
        /// チケットの新規登録
        /// </summary>
        private void CreateTicket()
        {
            // プロジェクトの選択
            Project project = SelectProject();

            // 行のインデックスは2から（1originで1は項目名称）
            //  登録する個数を数える
            int count = 0;
            for ( int raw = 2; true; ++raw, ++count ) {
                // 概要が空白になるまで有効なチケットとみなす
                Excel.Range cell = (Excel.Range)TracData.Cells[raw, FindColumn( "件名" )];
                if ( cell.Text.ToString() == "" ) {
                    break;
                }
            }

            // チケットを登録する
            using ( ProgressDlg progress = new ProgressDlg( 0, count ) ) {
                progress.Show();
                for ( int i = 0, raw = 2; i < count; ++i, ++raw ) {
                    ++progress.Value;
                    progress.Text = "Backlog へアップロード中 (" + progress.Value + "/" + progress.Maximum + ")";

                    // チケットを新規に登録して、次のチケットへ
                    CreateTicket( project, TracData.Cells, raw );
                }
            }
        }

        /// <summary>
        /// チケットの更新登録
        /// </summary>
        private void UpdateTicket()
        {
            // 行のインデックスは2から（1originで1は項目名称）
            //  登録する個数を数える
            int count = 0;
            for ( int raw = 2; true; ++raw, ++count ) {
                // 概要が空白になるまで有効なチケットとみなす
                Excel.Range cell = (Excel.Range)TracData.Cells[raw, FindColumn( "件名" )];
                if ( cell.Text.ToString() == "" ) {
                    break;
                }
            }

            // チケットを登録する
            using ( ProgressDlg progress = new ProgressDlg( 0, count ) ) {
                progress.Show();
                for ( int i = 0, raw = 2; i < count; ++i, ++raw ) {
                    ++progress.Value;
                    progress.Text = "Backlog へアップロード中 (" + progress.Value + "/" + progress.Maximum + ")";

                    // ID が空のチケットは新規作成
                    Excel.Range cell = (Excel.Range)TracData.Cells[raw, FindColumn( "ID" )];
                    if ( cell.Text.ToString() == "" ) {
                        CreateTicket( project, TracData.Cells, raw );
                    }
                    // ID が設定されているチケットは更新
                    else {
                        UpdateTicket( project, TracData.Cells, raw, "ExcelBacklogAddIn から更新" );
                    }
                }
            }
        }

        /// <summary>
        /// セルの取得
        /// </summary>
        /// <param name="raw"></param>
        /// <param name="column"></param>
        /// <returns></returns>
        private Excel.Range GetCell( int raw, int column )
        {
            return (Excel.Range)TracData.Cells[raw, column];
        }
        
        /// <summary>
        /// チケットのダウンロード
        /// </summary>
        private void DownloadTicket()
        {
            // プロジェクトの選択(更新用に取っておく)
            project = SelectProject();

            string name;

            // 概要を探す(列は1から開始)
            int column = FindColumn( "件名" );

            // 概要の空白（挿入位置）を探す(行は2から開始)
            int raw = 2;
            while ( true ) {
                name = (string)((Excel.Range)TracData.Cells[raw, column]).Value2;
                if ( name == null ) {
                    break;
                }

                ++raw;
            }

            // 課題一覧を取得して表示
            List<Issue> issues = project.GetIssues();
            using ( ProgressDlg progress = new ProgressDlg( 0, issues.Count ) ) {
                progress.Show();
                progress.Maximum = issues.Count;
                progress.Minimum = 0;

                foreach ( Issue issue in issues ) {
                    ++progress.Value;
                    progress.Text = "Backlog からダウンロード中 (" + progress.Value + "/" + progress.Maximum + ")";
                    progress.Update();

                    // ID を設定
                    if ( EnableID ) {
                        Excel.Range cell = (Excel.Range)TracData.Cells[raw, FindColumn( "ID" )];
                        cell.set_Value( null, issue.Id );
                    }

                    GetCell( raw, FindColumn( "件名" ) ).set_Value( null, issue.Summary );
                    GetCell( raw, FindColumn( "詳細" ) ).set_Value( null, issue.Description );

                    //// その他の項目を設定
                    //foreach ( DictionaryEntry obj in attributes ) {
                    //    // セルの書式を"文字列"にし、取得した情報を設定
                    //    Excel.Range cell = (Excel.Range)TracData.Cells[raw, FindColumn( (string)obj.Key )];
                    //    cell.NumberFormatLocal = "@";
                    //    cell.set_Value( null, obj.Value );
                    //}

                    // 次の行
                    ++raw;
                }
            }
        }

        /// <summary>
        /// セルのダブルクリックイベント
        /// </summary>
        /// <param name="Target"></param>
        /// <param name="Cancel"></param>
        private void Sheet1_BeforeDoubleClick( Excel.Range Target, ref bool Cancel )
        {
            //Excel.Range title = (Excel.Range)ImportData.Cells[1, Target.Column];
            //// "開始日"と"終了日"の場合はカレンダーを表示する
            //if ( (title.Text.ToString() == TracProxy.TicketAttributes.DueAssign) ||
            //     (title.Text.ToString() == TracProxy.TicketAttributes.DueClose) ) {
            //    Calendar cal = new Calendar();
            //    cal.StartPosition = FormStartPosition.Manual;
            //    cal.Location = Cursor.Position;
            //    DialogResult ret = cal.ShowDialog( null );
            //    if ( ret == DialogResult.OK ) {
            //        // セルの書式を"文字列"にし、選択された日付を設定
            //        Target.NumberFormatLocal = "@";
            //        Target.set_Value( null, cal.SelectDate.ToShortDateString() );
            //    }

            //    // セルの入力はさせたくないので、キャンセル扱いにする
            //    Cancel = true;
            //}
        }

        /// <summary>
        /// セルが変更されたときに呼ばれる
        /// </summary>
        /// <param name="Target"></param>
        private void Sheet1_ChangeEventHandler( Excel.Range Target )
        {
            //// 項目名に対しては行わない
            //if ( Target.Row == 1 ) {
            //    return;
            //}

            //// ダウンロード中は行わない
            //if ( IsDownload ) {
            //    return;
            //}

            //try {
            //    // 全要素のについて検索
            //    for ( int colomun = 1; true; ++colomun ) {
            //        Excel.Range title = (Excel.Range)TracData.Cells[1, colomun];
            //        string s = title.Text.ToString();
            //        // タイトルが設定されていなければ終了
            //        if ( title.Text.ToString() == "" ) {
            //            break;
            //        }

            //        // id にデフォルト値は不要なので無視する
            //        // resolution に fixed が入っているのでこの項目は無視する
            //        if ( (title.Text.ToString() == "resolution") ||
            //             (title.Text.ToString() == "id") ) {
            //            continue;
            //        }

            //        // このカラムのデータを取得
            //        XmlRpcStruct Field = null;
            //        for ( int i = 0; i < FieldNames.Length; ++i ) {
            //            if ( FieldNames[i]["name"].ToString() == title.Text.ToString() ) {
            //                Field = FieldNames[i];
            //                break;
            //            }
            //        }
            //        if ( Field == null ) {
            //            Trace.WriteLine( title.Text.ToString() );
            //            throw new Exception( "フィールドの項目が不足しています" );
            //        }

            //        // 報告者は選択ユーザ名を入れる
            //        if ( (title.Text.ToString() == "reporter") || (title.Text.ToString() == "owner") ) {
            //            Excel.Range cell = (Excel.Range)TracData.Cells[Target.Row, colomun];
            //            if ( cell.Text.ToString() == "" ) {
            //                cell.set_Value( null, Option.SelectedAccount.UserName );
            //            }
            //        }
            //        // その他はデフォルト値を設定
            //        else {
            //            // デフォルト値があり、セルにテキストが
            //            // 設定されていなければ、デフォルト値を設定する
            //            if ( Field["value"] != null ) {
            //                Excel.Range cell = (Excel.Range)TracData.Cells[Target.Row, colomun];
            //                if ( cell.Text.ToString() == "" ) {
            //                    if ( Field["value"].ToString() != "" ) {
            //                        cell.set_Value( null, Field["value"].ToString() );
            //                    }
            //                }
            //            }
            //        }
            //    }
            //}
            //catch ( Exception ex ) {
            //    MessageBox.Show( ex.Message );
            //}
        }

        #region VSTO で生成されたコード

        /// <summary>
        /// デザイナ サポートに必要なメソッドです。このメソッドの内容を
        /// コード エディタで変更しないでください。
        /// </summary>
        private void InternalStartup()
        {
            this.Startup += new System.EventHandler(ThisAddIn_Startup);
            this.Shutdown += new System.EventHandler(ThisAddIn_Shutdown);
        }
        
        #endregion
    }
}
