﻿using System;
using System.Globalization;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CookComputing.XmlRpc;

namespace BTS.Backlog
{
    /// <summary>
    /// 課題検索データ
    /// </summary>
    public class FindIssueData
    {
        /// <summary>
        /// チケット属性
        /// </summary>
        class AttributeNames
        {
            public const string PROJECT_ID = "projectId";       // プロジェクトID

            public const string ISSUE_TYPE_ID = "issueTypeId";  // 課題種別ID
            public const string COMPONENT_ID = "componentId";   // コンポーネントID
            public const string VERSION_ID = "versionId";       // バージョンID
            public const string MILESTONE_ID = "milestoneId";   // マイルストーンID
            public const string STATUS_ID = "statusId";         // 状態ID
            public const string PRIORITY_ID = "priorityId";     // 優先度ID
            public const string ASSIGNER_ID = "assignerId";     // 担当者ID
            public const string RESOLUTION_ID = "resolutionId"; // 完了理由ID

            public const string CREATED_ON_MIN = "created_on_min";  // 課題の登録日 範囲開始日(YYYYMMDD)
            public const string CREATED_ON_MAX = "created_on_max";  // 課題の登録日 範囲終了日(YYYYMMDD)
            public const string UPDATED_ON_MIN = "updated_on_min";  // 課題の更新日 範囲開始日(YYYYMMDD)
            public const string UPDATED_ON_MAX = "updated_on_max";  // 課題の更新日 範囲終了日(YYYYMMDD)
            public const string DUE_DATE_MIN = "due_date_min";      // 課題の期限日 範囲開始日(YYYYMMDD)
            public const string DUE_DATE_MAX = "due_date_max";      // 課題の期限日 範囲終了日(YYYYMMDD)

            public const string QUERY = "query";                // 検索キーワード

            public const string ORDER = "order";                // 昇順/降順
            public const string OFFSET = "offset";              // 取得する検索結果のオフセット値 
            public const string LIMIT = "limit";                // 取得する検索結果の課題数(0〜100)
        }

        /// <summary>
        /// DateTime のフォーマット（年月日）
        /// </summary>
        private static string DateFormat = "yyyyMMdd";

        /// <summary>
        /// DateTime のフォーマット（年月日自分秒）
        /// </summary>
        private static string DateTimeFormat = "yyyyMMddHHmmss";

        /// <summary>
        /// 属性
        /// </summary>
        public XmlRpcAttributes Attributes
        {
            get;
            private set;
        }

        /// <summary>
        /// プロジェクトID
        /// </summary>
        public int ProjetcId
        {
            get
            {
                return Attributes.Get<int>( AttributeNames.PROJECT_ID );
            }

            set
            {
                Attributes.Set<int>( AttributeNames.PROJECT_ID, value );
            }
        }

        /// <summary>
        /// 課題種別ID
        /// </summary>
        public int[] IssueTypeId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.ISSUE_TYPE_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.ISSUE_TYPE_ID, value );
            }
        }

        /// <summary>
        /// コンポーネントID
        /// </summary>
        public int[] ComponentId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.COMPONENT_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.COMPONENT_ID, value );
            }
        }

        /// <summary>
        /// バージョンID
        /// </summary>
        public int[] VersionId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.VERSION_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.VERSION_ID, value );
            }
        }

        /// <summary>
        /// マイルストーンID
        /// </summary>
        public int[] MilestoneId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.MILESTONE_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.MILESTONE_ID, value );
            }
        }

        /// <summary>
        /// 状態ID
        /// </summary>
        public int[] StatusId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.STATUS_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.STATUS_ID, value );
            }
        }

        /// <summary>
        /// 優先度ID
        /// </summary>
        public int[] PriorityId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.PRIORITY_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.PRIORITY_ID, value );
            }
        }

        /// <summary>
        /// 担当者ID
        /// </summary>
        public int[] AssignerId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.ASSIGNER_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.ASSIGNER_ID, value );
            }
        }

        /// <summary>
        /// 完了理由ID
        /// </summary>
        public int[] ResolutionId
        {
            get
            {
                return Attributes.Get<int[]>( AttributeNames.RESOLUTION_ID );
            }

            set
            {
                Attributes.Set<int[]>( AttributeNames.RESOLUTION_ID, value );
            }
        }

        /// <summary>
        /// DateTime のパース
        /// </summary>
        /// <param name="key"></param>
        /// <returns></returns>
        private DateTime ParseDateTime( string key )
        {
            string str = Attributes.Get<string>( key );
            if ( string.IsNullOrEmpty( str ) ) {
                return new DateTime();
            }
            else {
                return DateTime.ParseExact( str, DateTimeFormat, DateTimeFormatInfo.InvariantInfo );
            }
        }

        /// <summary>
        /// 課題の登録日 範囲開始日(YYYYMMDD)
        /// </summary>
        public DateTime CreatedOnMin
        {
            get
            {
                return ParseDateTime( AttributeNames.CREATED_ON_MIN );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.CREATED_ON_MIN, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 課題の登録日 範囲終了日(YYYYMMDD)
        /// </summary>
        public DateTime CreatedOnMax
        {
            get
            {
                return ParseDateTime( AttributeNames.CREATED_ON_MAX );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.CREATED_ON_MAX, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 課題の更新日 範囲開始日(YYYYMMDD)
        /// </summary>
        public DateTime UpdatedOnMin
        {
            get
            {
                return ParseDateTime( AttributeNames.UPDATED_ON_MIN );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.UPDATED_ON_MIN, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 課題の更新日 範囲終了日(YYYYMMDD)
        /// </summary>
        public DateTime UpdatedOnMax
        {
            get
            {
                return ParseDateTime( AttributeNames.UPDATED_ON_MAX );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.UPDATED_ON_MAX, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 課題の期限日 範囲開始日(YYYYMMDD)
        /// </summary>
        public DateTime DueDateMin
        {
            get
            {
                return ParseDateTime( AttributeNames.DUE_DATE_MIN );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.DUE_DATE_MIN, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// 課題の期限日 範囲終了日(YYYYMMDD)
        /// </summary>
        public DateTime DueDateMax
        {
            get
            {
                return ParseDateTime( AttributeNames.DUE_DATE_MAX );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.DUE_DATE_MAX, value.ToString( DateFormat ) );
            }
        }

        /// <summary>
        /// チケットの並び順
        /// </summary>
        public string Query
        {
            get
            {
                return Attributes.Get<string>( AttributeNames.QUERY );
            }

            set
            {
                Attributes.Set<string>( AttributeNames.QUERY, value );
            }
        }

        /// <summary>
        /// チケットの並び順
        /// </summary>
        public bool Order
        {
            get
            {
                return Attributes.Get<bool>( AttributeNames.ORDER );
            }

            set
            {
                Attributes.Set<bool>( AttributeNames.ORDER, value );
            }
        }

        /// <summary>
        /// 取得する検索結果のオフセット値 
        /// </summary>
        public int Offset
        {
            get
            {
                return Attributes.Get<int>( AttributeNames.OFFSET );
            }

            set
            {
                Attributes.Set<int>( AttributeNames.OFFSET, value );
            }
        }

        /// <summary>
        /// 取得する検索結果の課題数(0〜100)
        /// </summary>
        public int Limit
        {
            get
            {
                return Attributes.Get<int>( AttributeNames.LIMIT );
            }

            set
            {
                if ( (value < 0) || (100 < value) ) {
                    throw new ArgumentException();
                }

                Attributes.Set<int>( AttributeNames.LIMIT, value );
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public FindIssueData()
        {
            Attributes = new XmlRpcAttributes();
        }

        /// <summary>
        /// XmlRpcStruct に変換
        /// </summary>
        /// <returns></returns>
        public XmlRpcStruct ToXmlRpcStruct()
        {
            return Attributes.Value;
        }
    }
}
