/*
Copyright 2009 senju@users.sourceforge.jp

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 */

package net.excentrics.bandWidth;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;

/**
 * <p>
 * java.nio.SocketChannelの送信帯域を制限するためのクラスです。
 * <p>
 * SocketChannelを生成する際に非ブロッキングモードで生成した場合、このクラスのすべてのメソッド呼び出しはブロックしません。
 * <p>
 * 全てのメソッドはスレッドセーフでは<b>ありません。</b>
 * <p>
 * 受信の帯域を制限する機能は現在実装されていません。
 * <p>
 * 使い方
 * 
 * <pre>
 * 		ByteBuffer buf =  ByteBuffer.allocate(BUFSIZE);
 * 		SocketChannel sc = serverSocketChannel.accept();
 * 		//制限速度をbandWidthに設定
 * 		SocketChannelWriter	scw = new SocketChannelWriter(sc,bandWidth);
 * 		while(//バッファーにデータを読み込む){
 * 			long nextTime = scw.write(buf);
 * 			sleep(nextTime);
 * 		}
 * </pre>
 * <p>
 * 詳しい使い方についてはソース付属のサンプルおよびjava.nioパッケージのドキュメントを参照してください。
 * 
 */

public class SocketChannelWriter {

	private final SocketChannel channel;
	private final BandWidthController c;

	/**
	 * SocketChannelWriterを生成します。
	 * 
	 * @param channel
	 *            データの送信を行うチャネル
	 * @param bandWidth
	 *            制限速度(bit per second)
	 */
	public SocketChannelWriter(SocketChannel channel, double bandWidth) {
		this.channel = channel;
		this.c = new BandWidthController();
		this.c.setBandWidth(bandWidth);
	}

	/**
	 * 制限速度を返します。
	 * 
	 * @return 現在設定されている制限速度(bit per second)
	 */
	public double getBandWidth() {
		return this.c.getBandWidth();
	}

	/**
	 * 制限速度を変更します。
	 * <p>
	 * 既に送出が開始されているSocketChannelWriterの制限速度を変更することも可能です。
	 * 
	 * @param bandWidth
	 *            新しい制限速度(bit per second)
	 */
	public void setBandWidth(double bandWidth) {
		this.c.setBandWidth(bandWidth);
	}

	/**
	 * <p>
	 * 帯域制限をかけた送出を行います。
	 * <p>
	 * 指定されたSocketChannelが非ブロッキングモードの場合、このメソッド呼び出しがブロックされることはありません。
	 * <p>
	 * 戻り値は無視しても帯域制限は行われますが、不要なＣＰＵリソースの消費を 避けるためには戻り値を利用してください。
	 * 
	 * @param buffer
	 *            送出するデータが入ったバッファ。
	 *            <p>
	 *            制限速度をbyte単位に変更した時の10分の１以上のサイズが推奨されます。
	 *            <p>
	 *            つまりgetBandWidth() / 8 / 10 以上です。
	 * 
	 * @return 待機時間をミリ秒で返します。
	 *         <p>
	 *         次回のメソッド呼び出しは、待機時間ミリ秒経過した後に行うことが推奨されます。
	 *         つまりXミリ秒経過してから続きのデータを送出してください。
	 *         <p>
	 *         待機時間以内にこのメソッドを呼び出した場合、0バイト送出を行った後、すぐに戻ります。
	 *         よって適切に帯域制限は行われますが、CPUリソースを無駄に消費します。
	 *         <p>
	 *         待機時間以内に次回送出するべきデータをバッファに追加しておき、
	 *         かつ残った待機時間はThread.sleep()し他のスレッドにCPUリソースを明け渡すことを推奨します。
	 * 
	 * 
	 * @throws IOException
	 *             　SocketChannelが例外を発生させた場合。
	 */
	public long write(ByteBuffer buffer) throws IOException {
		int writtenSize = 0;
		final int size = buffer.remaining();
		final SizeAndInterval si = this.c.preWrite(buffer.capacity());

		try {
			if (si.getSizeInByte() == 0) {
				this.c.postWrite(0);
			} else if (si.getSizeInByte() >= size || si.getSizeInByte() == -1) {
				// 制限不要の場合
				writtenSize = this.channel.write(buffer);
				this.c.postWrite(writtenSize);
			} else {
				// バッファの中身の途中まで送出する場合
				// limitを退避
				final int limit = buffer.limit();
				buffer.limit(buffer.position() + si.getSizeInByte());

				writtenSize = this.channel.write(buffer);
				this.c.postWrite(writtenSize);
				// 退避したlimitを元に戻す
				buffer.limit(limit);
			}
		} catch (final AsymmetricalCallException e) {
			throw new IOException(e);
		}

		return si.getIntervalInMils();
	}

}
