/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.viewers;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.net.URL;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import junit.framework.AssertionFailedError;
import junit.framework.TestCase;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Path;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.ide.IDE;

import benten.cat.ui.internal.tabletree.XLIFFMultiPageEditorPart;
import benten.core.CorePlugin;
import benten.core.dom.AltTransDelegate;
import benten.core.dom.NoteDelegate;
import benten.core.io.Files;
import benten.ui.UiPlugin;

public class TransUnitViewerTest extends TestCase {

	private IEditorPart openEditor(final IWorkbenchPage page, final String tempFileName) throws Exception {

		// テスト用の xlf ロード
		final String path = "/src/" + getClass().getPackage().getName().replace('.', '/') + "/test.xlf";
		final URL url = getClass().getClassLoader().getResource(path);
		final String dataPath = FileLocator.resolve(url).getPath();
		final byte[] bytes = Files.readFileToByteArray(new File(dataPath));

		// テスト用の xlf を作成
		final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		final IProject project = root.getProject("test.project");
		if (!project.exists()) {
			project.create(null);
		}
		project.open(null);
		final IFile xliffFile = project.getFile(new Path(tempFileName));
		xliffFile.delete(true, null);
		xliffFile.create(new ByteArrayInputStream(bytes), true, null);

		return IDE.openEditor(page, xliffFile);
	}

	private void scrollTransUnitSelection(final TransUnitViewer viewer, final int move) {
		viewer.scrollTransUnitSelection(move);
		viewer.sourceCompareViewer.fireContentChange();
		viewer.targetCompareViewer.fireContentChange();
	}

	//-------------------------------------------------------------------------

	public void test() throws Exception {
		try {
			final IWorkbenchPage page = UiPlugin.getActivePage();

			// 読み込み
			final IEditorPart editor0001 = openEditor(page, "0001.xlf");
			final TransUnitViewer viewer0001 = ((XLIFFMultiPageEditorPart) editor0001).getTransUnitViewer();

			Display.getCurrent().asyncExec(new Runnable() {
				public void run() {
					try {

						doTest0001(viewer0001);

					} catch (final AssertionFailedError e) {
						e.printStackTrace();
					} catch (final Throwable e) {
						System.out.println(e.getMessage());
						if (CorePlugin.IS_DEBUG) {
							e.printStackTrace();
						}
					} finally {
						page.closeEditor(editor0001, false);
					}
				}
			});

			// 次の翻訳単位
			final IEditorPart editor0002 = openEditor(page, "0002.xlf");
			final TransUnitViewer viewer0002 = ((XLIFFMultiPageEditorPart) editor0002).getTransUnitViewer();
			scrollTransUnitSelection(viewer0002, 1);

			Display.getCurrent().asyncExec(new Runnable() {
				public void run() {
					try {

						doTest0002(viewer0002);

					} catch (final AssertionFailedError e) {
						e.printStackTrace();
					} catch (final Throwable e) {
						System.out.println(e.getMessage());
						if (CorePlugin.IS_DEBUG) {
							e.printStackTrace();
						}
					} finally {
						page.closeEditor(editor0002, false);
					}
				}
			});

			// state 条件変更
			final IEditorPart editor0003 = openEditor(page, "0003.xlf");
			final TransUnitViewer viewer0003 = ((XLIFFMultiPageEditorPart) editor0003).getTransUnitViewer();
			for (final Button check : viewer0003.stateChecks) {
				final String state = (String) check.getData();
				if (state.equals("") || state.equals("x-reviewed")) {
					check.setSelection(false);

				}
			}
			viewer0003.refresh();
			viewer0003.changeSelectionTransUnit();
			scrollTransUnitSelection(viewer0003, 0);

			Display.getCurrent().asyncExec(new Runnable() {
				public void run() {
					try {

						doTest0003(viewer0003);

					} catch (final AssertionFailedError e) {
						e.printStackTrace();
					} catch (final Throwable e) {
						System.out.println(e.getMessage());
						if (CorePlugin.IS_DEBUG) {
							e.printStackTrace();
						}
					} finally {
						page.closeEditor(editor0003, false);
					}
				}
			});

		} catch (final Throwable e) {
			System.out.println(e.getMessage());
			if (CorePlugin.IS_DEBUG) {
				e.printStackTrace();
			}
		}
	}

	private void doTest0001(final TransUnitViewer viewer) throws Exception {

		assertEquals(1905, viewer.transUnitTableViewer.getTable().getItemCount());
		assertEquals(0, viewer.transUnitTableViewer.getTable().getSelectionIndex());

		assertEquals("1 / 1905", viewer.transUnitCountLabel.getText());
		assertEquals(true, viewer.transUnitCountLabel.getEnabled());
		assertEquals(false, viewer.joinTransUnitButton.getEnabled());
		assertEquals(false, viewer.prevTransUnitButton.getEnabled());
		assertEquals(true, viewer.nextTransUnitButton.getEnabled());

		assertEquals(
				"babel://org.eclipse.team.svn.ui/org/eclipse/team/svn/ui/messages.properties?key=SetPropertyAction_label",
				viewer.idText.getText());
		assertEquals(true, viewer.idText.getEnabled());

		assertEquals(12, viewer.stateCombo.getItemCount());
		assertEquals(true, viewer.stateCombo.getEnabled());
		final List<String> stateList = new LinkedList<String>(Arrays.asList(viewer.stateCombo.getItems()));
		assertEquals("", stateList.remove(0));
		assertEquals("(1) x-reviewed", stateList.remove(0));
		assertEquals("(2) final", stateList.remove(0));
		assertEquals("(3) needs-adaptation", stateList.remove(0));
		assertEquals("(4) needs-l10n", stateList.remove(0));
		assertEquals("(5) needs-review-adaptation", stateList.remove(0));
		assertEquals("(6) needs-review-l10n", stateList.remove(0));
		assertEquals("(7) needs-review-translation", stateList.remove(0));
		assertEquals("(8) needs-translation", stateList.remove(0));
		assertEquals("(9) new", stateList.remove(0));
		assertEquals("(b) signed-off", stateList.remove(0));
		assertEquals("(k) translated", stateList.remove(0));

		assertEquals(false, viewer.notTranslateCheck.getSelection());
		assertEquals(true, viewer.notTranslateCheck.getEnabled());
		assertEquals(true, viewer.updateTmCheck.getSelection());
		assertEquals(true, viewer.updateTmCheck.getEnabled());

		assertEquals("Set Property...", viewer.sourceCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("プロパティーの設定...", viewer.targetCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(true, viewer.targetCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("Set Property...", viewer.sourceCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getRightSourceViewer().isEditable());
		assertEquals("プロパティーの設定...", viewer.targetCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.targetCompareViewer.getRightSourceViewer().isEditable());

		assertEquals(true, viewer.copyFromSourceButton.getEnabled());
		assertEquals(true, viewer.targetCompareViewer.getCopyButton().getEnabled());

		assertEquals(1, viewer.altTransTableViewer.getTable().getItemCount());
		assertEquals(0, viewer.altTransTableViewer.getTable().getSelectionIndex());

		assertEquals("pleiades", viewer.originLabel.getText());
		assertEquals(true, viewer.originLabel.getEnabled());
		assertEquals(0, viewer.altTransQualityBar.getSelection());
		assertEquals(false, viewer.altTransQualityBar.isVisible());
		assertEquals("1 / 1", viewer.altTransCountLabel.getText());
		assertEquals(true, viewer.altTransCountLabel.getEnabled());
		assertEquals(false, viewer.prevAltTransButton.getEnabled());
		assertEquals(false, viewer.nextAltTransButton.getEnabled());

		assertEquals(0, viewer.noteTableViewer.getTable().getItemCount());
		assertEquals(-1, viewer.noteTableViewer.getTable().getSelectionIndex());

		assertEquals(true, viewer.newNoteButton.getEnabled());
		assertEquals(false, viewer.editNoteButton.getEnabled());
		assertEquals(false, viewer.removeNoteButton.getEnabled());

		assertEquals(
				"babel://org.eclipse.team.svn.ui/org/eclipse/team/svn/ui/messages.properties?key=SetPropertyAction_label",
				viewer.selectedTransUnit.getId());
		assertEquals("Set Property...", viewer.selectedTransUnit.getSource());
		assertEquals("プロパティーの設定...", viewer.selectedTransUnit.getTarget());
		assertEquals("", viewer.selectedTransUnit.getTargetState());
		assertEquals(false, viewer.selectedTransUnit.isContextGourpTmOmit());
		assertEquals(true, viewer.selectedTransUnit.isTranslate());
		assertEquals(1, viewer.selectedTransUnit.getAltTransList().size());
		assertEquals(0, viewer.selectedTransUnit.getNoteList().size());

		assertEquals(null, viewer.selectedNote);

		final List<String> checkedStateList = getCheckedStateList(viewer);
		assertEquals(12, checkedStateList.size());
		assertEquals("", checkedStateList.remove(0));
		assertEquals("final", checkedStateList.remove(0));
		assertEquals("needs-adaptation", checkedStateList.remove(0));
		assertEquals("needs-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-adaptation", checkedStateList.remove(0));
		assertEquals("needs-review-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-translation", checkedStateList.remove(0));
		assertEquals("needs-translation", checkedStateList.remove(0));
		assertEquals("new", checkedStateList.remove(0));
		assertEquals("signed-off", checkedStateList.remove(0));
		assertEquals("translated", checkedStateList.remove(0));
		assertEquals("x-reviewed", checkedStateList.remove(0));
	}

	private void doTest0002(final TransUnitViewer viewer) throws Exception {

		assertEquals(1905, viewer.transUnitTableViewer.getTable().getItemCount());
		assertEquals(1, viewer.transUnitTableViewer.getTable().getSelectionIndex());

		assertEquals("2 / 1905", viewer.transUnitCountLabel.getText());
		assertEquals(true, viewer.transUnitCountLabel.getEnabled());
		assertEquals(true, viewer.joinTransUnitButton.getEnabled());
		assertEquals(true, viewer.prevTransUnitButton.getEnabled());
		assertEquals(true, viewer.nextTransUnitButton.getEnabled());

		assertEquals("benten://org.eclipse.mylyn/source/about.html?seq=5-1", viewer.idText.getText());
		assertEquals(true, viewer.idText.getEnabled());

		assertEquals(12, viewer.stateCombo.getItemCount());
		assertEquals(true, viewer.stateCombo.getEnabled());
		final List<String> stateList = new LinkedList<String>(Arrays.asList(viewer.stateCombo.getItems()));
		assertEquals("", stateList.remove(0));
		assertEquals("(1) x-reviewed", stateList.remove(0));
		assertEquals("(2) final", stateList.remove(0));
		assertEquals("(3) needs-adaptation", stateList.remove(0));
		assertEquals("(4) needs-l10n", stateList.remove(0));
		assertEquals("(5) needs-review-adaptation", stateList.remove(0));
		assertEquals("(6) needs-review-l10n", stateList.remove(0));
		assertEquals("(7) needs-review-translation", stateList.remove(0));
		assertEquals("(8) needs-translation", stateList.remove(0));
		assertEquals("(9) new", stateList.remove(0));
		assertEquals("(b) signed-off", stateList.remove(0));
		assertEquals("(k) translated", stateList.remove(0));

		assertEquals(false, viewer.notTranslateCheck.getSelection());
		assertEquals(true, viewer.notTranslateCheck.getEnabled());
		assertEquals(true, viewer.updateTmCheck.getSelection());
		assertEquals(true, viewer.updateTmCheck.getEnabled());

		assertEquals("Use blocks in 'if' statements.", viewer.sourceCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("'if' ステートメントでブロックを使用", viewer.targetCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(true, viewer.targetCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("Use blocks in if/while/for...", viewer.sourceCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getRightSourceViewer().isEditable());
		assertEquals("if/while/for...", viewer.targetCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.targetCompareViewer.getRightSourceViewer().isEditable());

		assertEquals(true, viewer.copyFromSourceButton.getEnabled());
		assertEquals(true, viewer.targetCompareViewer.getCopyButton().getEnabled());

		assertEquals(5, viewer.altTransTableViewer.getTable().getItemCount());
		assertEquals(0, viewer.altTransTableViewer.getTable().getSelectionIndex());

		assertEquals("pleiades long long name 1234567890 1234567890 1234567890 1234567890 1234567890: 90%",
				viewer.originLabel.getText());
		assertEquals(true, viewer.originLabel.getEnabled());
		assertEquals(90, viewer.altTransQualityBar.getSelection());
		assertEquals(true, viewer.altTransQualityBar.isVisible());
		assertEquals("1 / 5", viewer.altTransCountLabel.getText());
		assertEquals(true, viewer.altTransCountLabel.getEnabled());
		assertEquals(false, viewer.prevAltTransButton.getEnabled());
		assertEquals(true, viewer.nextAltTransButton.getEnabled());

		assertEquals(2, viewer.noteTableViewer.getTable().getItemCount());
		assertEquals(-1, viewer.noteTableViewer.getTable().getSelectionIndex());

		assertEquals(true, viewer.newNoteButton.getEnabled());
		assertEquals(false, viewer.editNoteButton.getEnabled());
		assertEquals(false, viewer.removeNoteButton.getEnabled());

		assertEquals("benten://org.eclipse.mylyn/source/about.html?seq=5-1", viewer.selectedTransUnit.getId());
		assertEquals("Use blocks in 'if' statements.", viewer.selectedTransUnit.getSource());
		assertEquals("'if' ステートメントでブロックを使用", viewer.selectedTransUnit.getTarget());
		assertEquals("x-reviewed", viewer.selectedTransUnit.getTargetState());
		assertEquals(false, viewer.selectedTransUnit.isContextGourpTmOmit());
		assertEquals(true, viewer.selectedTransUnit.isTranslate());

		assertEquals(5, viewer.selectedTransUnit.getAltTransList().size());
		AltTransDelegate altTrans = viewer.selectedTransUnit.getAltTransList().get(0);
		assertEquals("90%", altTrans.getMatchQuality());
		assertEquals("pleiades long long name 1234567890 1234567890 1234567890 1234567890 1234567890", altTrans
				.getOrigin());
		assertEquals("Use blocks in if/while/for...", altTrans.getSource());
		assertEquals("if/while/for...", altTrans.getTarget());
		altTrans = viewer.selectedTransUnit.getAltTransList().get(1);
		assertEquals("62%", altTrans.getMatchQuality());
		assertEquals("pleiades", altTrans.getOrigin());
		assertEquals("Use blocks in if/while/for/do statements", altTrans.getSource());
		assertEquals("if/while/for/do ステートメントでブロックを使用", altTrans.getTarget());
		altTrans = viewer.selectedTransUnit.getAltTransList().get(2);
		assertEquals("50%", altTrans.getMatchQuality());
		assertEquals("pleiades", altTrans.getOrigin());
		assertEquals("Allow nested blocks in case statements", altTrans.getSource());
		assertEquals("case ステートメント内のネストされたブロックを許可", altTrans.getTarget());
		altTrans = viewer.selectedTransUnit.getAltTransList().get(3);
		assertEquals("40%", altTrans.getMatchQuality());
		assertEquals("pleiades", altTrans.getOrigin());
		assertEquals("Use features in the workspace", altTrans.getSource());
		assertEquals("ワークスペースでフィーチャーを使用", altTrans.getTarget());
		altTrans = viewer.selectedTransUnit.getAltTransList().get(4);
		assertEquals("40%", altTrans.getMatchQuality());
		assertEquals("pleiades", altTrans.getOrigin());
		assertEquals("Free blocks in filesystem", altTrans.getSource());
		assertEquals("ファイル・システム内のフリー・ブロック", altTrans.getTarget());

		assertEquals(2, viewer.selectedTransUnit.getNoteList().size());
		NoteDelegate note = viewer.selectedTransUnit.getNoteList().get(0);
		assertEquals("k.ymoto@nifty.com", note.getFrom());
		assertEquals("2009.04.20 翻訳", note.getValue());
		note = viewer.selectedTransUnit.getNoteList().get(1);
		assertEquals("tosiki.iga@nifty.ne.jp", note.getFrom());
		assertEquals("2009.04.24 レビュー", note.getValue());
		assertEquals(null, viewer.selectedNote);

		final List<String> checkedStateList = getCheckedStateList(viewer);
		assertEquals(12, checkedStateList.size());
		assertEquals("", checkedStateList.remove(0));
		assertEquals("final", checkedStateList.remove(0));
		assertEquals("needs-adaptation", checkedStateList.remove(0));
		assertEquals("needs-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-adaptation", checkedStateList.remove(0));
		assertEquals("needs-review-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-translation", checkedStateList.remove(0));
		assertEquals("needs-translation", checkedStateList.remove(0));
		assertEquals("new", checkedStateList.remove(0));
		assertEquals("signed-off", checkedStateList.remove(0));
		assertEquals("translated", checkedStateList.remove(0));
		assertEquals("x-reviewed", checkedStateList.remove(0));
	}

	private void doTest0003(final TransUnitViewer viewer) throws Exception {

		assertEquals(70, viewer.transUnitTableViewer.getTable().getItemCount());
		assertEquals(0, viewer.transUnitTableViewer.getTable().getSelectionIndex());

		assertEquals("1 / 70", viewer.transUnitCountLabel.getText());
		assertEquals(true, viewer.transUnitCountLabel.getEnabled());
		assertEquals(false, viewer.joinTransUnitButton.getEnabled());
		assertEquals(false, viewer.prevTransUnitButton.getEnabled());
		assertEquals(true, viewer.nextTransUnitButton.getEnabled());

		assertEquals(
				"babel://org.eclipse.team.svn.ui/org/eclipse/team/svn/ui/messages.properties?key=RelocationChoicesPanel_Description",
				viewer.idText.getText());
		assertEquals(true, viewer.idText.getEnabled());

		assertEquals(12, viewer.stateCombo.getItemCount());
		assertEquals(true, viewer.stateCombo.getEnabled());
		final List<String> stateList = new LinkedList<String>(Arrays.asList(viewer.stateCombo.getItems()));
		assertEquals("", stateList.remove(0));
		assertEquals("(1) x-reviewed", stateList.remove(0));
		assertEquals("(2) final", stateList.remove(0));
		assertEquals("(3) needs-adaptation", stateList.remove(0));
		assertEquals("(4) needs-l10n", stateList.remove(0));
		assertEquals("(5) needs-review-adaptation", stateList.remove(0));
		assertEquals("(6) needs-review-l10n", stateList.remove(0));
		assertEquals("(7) needs-review-translation", stateList.remove(0));
		assertEquals("(8) needs-translation", stateList.remove(0));
		assertEquals("(9) new", stateList.remove(0));
		assertEquals("(b) signed-off", stateList.remove(0));
		assertEquals("(k) translated", stateList.remove(0));

		assertEquals(false, viewer.notTranslateCheck.getSelection());
		assertEquals(true, viewer.notTranslateCheck.getEnabled());
		assertEquals(true, viewer.updateTmCheck.getSelection());
		assertEquals(true, viewer.updateTmCheck.getEnabled());

		assertEquals("Project ''{0}'' is relocated", viewer.sourceCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("プロジェクト ''{0}'' は再配置されました", viewer.targetCompareViewer.getLeftSourceViewer().getTextWidget().getText());
		assertEquals(true, viewer.targetCompareViewer.getLeftSourceViewer().isEditable());
		assertEquals("", viewer.sourceCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.sourceCompareViewer.getRightSourceViewer().isEditable());
		assertEquals("", viewer.targetCompareViewer.getRightSourceViewer().getTextWidget().getText());
		assertEquals(false, viewer.targetCompareViewer.getRightSourceViewer().isEditable());

		assertEquals(true, viewer.copyFromSourceButton.getEnabled());
		assertEquals(true, viewer.targetCompareViewer.getCopyButton().getEnabled());

		assertEquals(0, viewer.altTransTableViewer.getTable().getItemCount());
		assertEquals(-1, viewer.altTransTableViewer.getTable().getSelectionIndex());

		assertEquals("", viewer.originLabel.getText());
		assertEquals(true, viewer.originLabel.getEnabled());
		assertEquals(0, viewer.altTransQualityBar.getSelection());
		assertEquals(false, viewer.altTransQualityBar.isVisible());
		assertEquals("", viewer.altTransCountLabel.getText());
		assertEquals(true, viewer.altTransCountLabel.getEnabled());
		assertEquals(false, viewer.prevAltTransButton.getEnabled());
		assertEquals(false, viewer.nextAltTransButton.getEnabled());

		assertEquals(1, viewer.noteTableViewer.getTable().getItemCount());
		assertEquals(-1, viewer.noteTableViewer.getTable().getSelectionIndex());

		assertEquals(true, viewer.newNoteButton.getEnabled());
		assertEquals(false, viewer.editNoteButton.getEnabled());
		assertEquals(false, viewer.removeNoteButton.getEnabled());

		assertEquals(
				"babel://org.eclipse.team.svn.ui/org/eclipse/team/svn/ui/messages.properties?key=RelocationChoicesPanel_Description",
				viewer.selectedTransUnit.getId());
		assertEquals("Project ''{0}'' is relocated", viewer.selectedTransUnit.getSource());
		assertEquals("プロジェクト ''{0}'' は再配置されました", viewer.selectedTransUnit.getTarget());
		assertEquals("needs-review-translation", viewer.selectedTransUnit.getTargetState());
		assertEquals(false, viewer.selectedTransUnit.isContextGourpTmOmit());
		assertEquals(true, viewer.selectedTransUnit.isTranslate());
		assertEquals(0, viewer.selectedTransUnit.getAltTransList().size());
		assertEquals(1, viewer.selectedTransUnit.getNoteList().size());

		assertEquals("xxxxx", viewer.selectedNote.getFrom());
		assertEquals("", viewer.selectedNote.getValue());

		final List<String> checkedStateList = getCheckedStateList(viewer);
		assertEquals(10, checkedStateList.size());
		assertEquals("final", checkedStateList.remove(0));
		assertEquals("needs-adaptation", checkedStateList.remove(0));
		assertEquals("needs-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-adaptation", checkedStateList.remove(0));
		assertEquals("needs-review-l10n", checkedStateList.remove(0));
		assertEquals("needs-review-translation", checkedStateList.remove(0));
		assertEquals("needs-translation", checkedStateList.remove(0));
		assertEquals("new", checkedStateList.remove(0));
		assertEquals("signed-off", checkedStateList.remove(0));
		assertEquals("translated", checkedStateList.remove(0));
	}

	private List<String> getCheckedStateList(final TransUnitViewer viewer) {
		final List<String> checkedStateList = new LinkedList<String>();
		for (final Button check : viewer.stateChecks) {
			if (check.getSelection()) {
				checkedStateList.add((String) check.getData());
			}
		}
		Collections.sort(checkedStateList);
		return checkedStateList;
	}
}
