/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.ui.viewers;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.compare.CompareConfiguration;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextListener;
import org.eclipse.jface.text.TextEvent;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.viewers.ContentViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.ProgressBar;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.events.ExpansionAdapter;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.wst.sse.core.StructuredModelManager;
import org.eclipse.wst.sse.core.internal.provisional.IStructuredModel;
import org.eclipse.wst.sse.core.internal.undo.IStructuredTextUndoManager;
import org.eclipse.wst.sse.ui.internal.contentoutline.ConfigurableContentOutlinePage;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMModel;
import org.eclipse.wst.xml.ui.internal.tabletree.IDesignViewer;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import benten.cat.glossary.ui.actions.ShowGlossaryAction;
import benten.cat.glossary.ui.editor.GlossaryConfiguration;
import benten.cat.ui.CatUiPlugin;
import benten.cat.ui.TranslationPerspectiveFactory;
import benten.cat.ui.dialogs.NoteDialog;
import benten.cat.ui.dialogs.NoteDialog.NoteDialogEntry;
import benten.cat.ui.internal.tabletree.XLIFFSelectionProvider;
import benten.cat.ui.viewers.messages.TransUnitViewerMessages;
import benten.core.dom.AltTransDelegate;
import benten.core.dom.ElementDelegate;
import benten.core.dom.NoteDelegate;
import benten.core.dom.TransUnitDelegate;
import benten.core.model.BentenXliff;
import benten.core.model.HelpTransUnitId;
import benten.ui.UiPlugin;
import benten.ui.preference.BentenProjectProperty;
import benten.ui.preference.BentenProjectProperty.ProjectProperty;
import benten.ui.views.AsyncSelectionChangeAdapter;
import benten.ui.views.AutoColumnWidthTableViewer;
import benten.ui.views.TableLabelProvider;
import benten.ui.views.UiToolkit;

/**
 * XLIFF エディターの翻訳単位タブに表示するビューアー。
 *
 * <UL>
 * <LI>XLIFF エディターの翻訳単位タブを実現します。
 * <LI>※ソースコード可読性の維持のために、いくつかの private メソッド、private フィールドおよびインナークラスの
 * public メソッドの JavaDoc について記載していません。
 * </UL>
 * <p>
 * ★基本設計「翻訳支援機能: 翻訳中間形式エディター機能: 翻訳単位編集」に対応します。<br>
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings( { "restriction" })
public class TransUnitViewer extends ContentViewer implements IDesignViewer {

	/**
	 * XLIFF エディターの翻訳単位タブに表示するビューアーのためのメッセージ。
	 */
	protected static final TransUnitViewerMessages fMsg = new TransUnitViewerMessages();

	/**
	 * 保管キー。
	 * <UL>
	 * <LI>この定義は XLIFF エディター固有のものです。
	 * </UL>
	 */
	protected static enum StoreKey {

		/** デフォルトのノート担当者 */
		DEFAULT_NOTE_FROM,

		/** state フィルター・チェックボックス展開コンポジットの展開状態 */
		EXPAND_FILTER_STATE_CHECKS,

		/** alt-trans 展開コンポジットの展開状態 */
		EXPAND_ALT_TRANS_TABLE,

		/** note リスト展開コンポジットの展開状態 */
		EXPAND_NOTE_LIST_TABLE
	}

	/**
	 * Benten が提供する基本的なウィジェットを生成する UI ツールキットのインスタンス。
	 */
	protected final UiToolkit toolkit = new UiToolkit();

	/** エディター・パート。 */
	protected final IEditorPart editorPart;

	/** コントロール。 */
	protected final Composite control;

	// 翻訳単位リストビューの構成部品。

	protected AutoColumnWidthTableViewer transUnitTableViewer;
	protected Label transUnitCountLabel;
	protected Button joinTransUnitButton;
	protected Button prevTransUnitButton;
	protected Button nextTransUnitButton;
	protected final List<Button> stateChecks = new LinkedList<Button>();

	// 翻訳単位エディターの構成部品。

	protected Text idText;
	protected Combo stateCombo;
	protected Button notTranslateCheck;
	protected Button updateTmCheck;

	protected TransCompareViewer sourceCompareViewer;
	protected TransCompareViewer targetCompareViewer;
	protected Button copyFromSourceButton;
	protected Button diffButton;

	protected AutoColumnWidthTableViewer altTransTableViewer;
	protected Label originLabel;
	protected ProgressBar altTransQualityBar;
	protected Label altTransCountLabel;
	protected Button prevAltTransButton;
	protected Button nextAltTransButton;

	protected AutoColumnWidthTableViewer noteTableViewer;
	protected Button newNoteButton;
	protected Button editNoteButton;
	protected Button removeNoteButton;

	// XML を操作するためのオブジェクト。

	protected TransUnitDelegate selectedTransUnit;
	protected NoteDelegate selectedNote;

	// 状態を記憶するためのフィールド。

	protected final Map<String, Character> stateKeyBindMap = new LinkedHashMap<String, Character>();
	protected final char[] stateKeyBindChars = new char[] { ' ', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'b', 'k' };

	/** プラグイン間連携用のリソース変更リスナー。 */
	protected final IResourceChangeListener resourceChangeListener = new IResourceChangeListener() {
		public void resourceChanged(final IResourceChangeEvent event) {
			refresh();
			transUnitTableViewer.refresh();
		}
	};

	/** 選択プロバイダー。 */
	protected final ISelectionProvider selectionProvider = new XLIFFSelectionProvider(this);

	/**
	 * コンストラクター。
	 * @param parent 親コンポジット
	 * @param editorPart エディター・パーツ
	 */
	public TransUnitViewer(final Composite parent, final IEditorPart editorPart) {

		this.editorPart = editorPart;
		this.control = toolkit.createComposite(parent, 1, 10);
		createTransUnitGroup(control);

		setContentProvider(new IContentProvider() {
			public void dispose() {
			}

			public void inputChanged(final Viewer viewer, final Object oldInput, final Object newInput) {
			}
		});

		// UiPlugin にプラグイン間連携用のリソース変更リスナーをセットします。
		// 登録したリスナーは翻訳ワークフロー各処理の終了時に呼び出されます。
		UiPlugin.getDefault().addResourceChangeListner(resourceChangeListener);
		control.addDisposeListener(new DisposeListener() {
			public void widgetDisposed(final DisposeEvent event) {
				final int size = UiPlugin.getDefault().removeResourceChangeListner(resourceChangeListener);

				// 翻訳単位エディターがすべて閉じられた場合、翻訳単位ビューではなく、
				// プロジェクト・エクスプローラーをアクティブにします。
				if (size == 0) {
					try {
						final IWorkbenchPage page = editorPart.getSite().getPage();
						page.showView(TranslationPerspectiveFactory.ID_PROJECT_EXPLORER);
					} catch (final PartInitException e) {
						throw new IllegalStateException(e);
					}
				}
			}
		});
	}

	/**
	 * 「翻訳単位」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitGroup(final Composite parent) {
		createTransUnitAttributes(parent);
		createTransUnitText(parent);
		createAltTransButtons(parent);
		createAltTransTableViewer(parent);
		createNoteTableViewer(parent);
	}

	/**
	 * 「翻訳単位リスト」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitListGroup(final Composite parent) {
		createStateItemGroup(parent);
		createTransUnitTableViewer(parent);
		createTransUnitButtons(parent);
		refresh();

		// ラベルの & 制御文字で設定できない部分のニモーニックを設定します。
		final KeyListener mnemonicKeyListener = new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.stateMask != SWT.ALT) {
					return;
				}
				final char key = (char) e.keyCode;
				final int index = Arrays.binarySearch(stateKeyBindChars, key);
				if (index >= 0) {
					if (stateCombo.isEnabled()) {
						stateCombo.select(index);
						stateCombo.setFocus();
					}
				} else {
					final String targetLeftLabel = fMsg.getLabelTarget();
					if (targetLeftLabel.contains("&")) { //$NON-NLS-1$
						final String mnemonic = targetLeftLabel.replaceFirst("^.*?&(.).*$", "$1"); //$NON-NLS-1$ //$NON-NLS-2$
						if (key == mnemonic.toLowerCase().charAt(0)) {
							targetCompareViewer.getLeftSourceViewer().getTextWidget().setFocus();
						}
					}
				}
			}
		};
		final Set<Control> allControls = new HashSet<Control>();
		assenbleChildrenRecursively(parent, allControls);
		assenbleChildrenRecursively(control, allControls);
		for (final Control c : allControls) {
			c.addKeyListener(mnemonicKeyListener);
		}

		// エディター部分以外のビューで UNDO が効かないため、翻訳単位リスト・ビューにキー・リスナーを設定します。
		final KeyListener transUnitListUndoKeyListener = new KeyAdapter() {
			@Override
			public void keyPressed(final KeyEvent e) {
				if (e.stateMask != SWT.CONTROL) {
					return;
				}
				final char key = (char) e.keyCode;
				final IDOMDocument domDoc = (IDOMDocument) getInput();
				if (key == 'z') {
					domDoc.getModel().getUndoManager().undo();
					refresh();
				} else if (key == 'y') {
					domDoc.getModel().getUndoManager().redo();
					refresh();
				}
			}
		};
		final Set<Control> transUnitListControls = new HashSet<Control>();
		assenbleChildrenRecursively(parent, transUnitListControls);
		for (final Control c : transUnitListControls) {
			c.addKeyListener(transUnitListUndoKeyListener);
		}
	}

	/**
	 * 子コントロールを再帰的に収集。
	 * @param parent 親コントロール。
	 * @param children 収集した子コントロールを格納するセット。
	 */
	private void assenbleChildrenRecursively(final Composite parent, final Set<Control> children) {
		for (final Control c : parent.getChildren()) {
			if (c instanceof Composite) {
				assenbleChildrenRecursively((Composite) c, children);
			}
			children.add(c);
		}
	}

	/**
	 * 「状態」グループの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createStateItemGroup(final Composite parent) {

		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelStateOfFilter());
		final Composite outer = new Composite(expandable, SWT.BORDER);
		outer.setLayout(new FillLayout());
		final Composite inner = toolkit.createComposite(outer, 2, 5);

		final Composite left = toolkit.createComposite(inner, 1, 5);
		final Button allCheckButton = new Button(left, SWT.CHECK);
		allCheckButton.setText(fMsg.getLabelSelectAllState());
		allCheckButton.setSelection(true);
		allCheckButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				final Button b = (Button) event.widget;
				for (final Button check : stateChecks) {
					check.setSelection(b.getSelection());
				}
				refresh();
			}
		});
		new Label(left, SWT.NONE);

		final Composite right = new Composite(inner, SWT.BORDER);
		right.setLayout(toolkit.createGridLayout(6, 5));
		right.setLayoutData(new GridData(GridData.FILL_BOTH));

		for (final String stateItem : BentenXliff.getStateItems()) {

			final Button checkButton = new Button(right, SWT.CHECK);
			checkButton.setData(stateItem);
			checkButton.setText(stateItem.equals("") ? fMsg.getLabelFilterNone() : stateItem); //$NON-NLS-1$
			checkButton.setSelection(true);
			checkButton.addSelectionListener(new SelectionAdapter() {
				@Override
				public void widgetSelected(final SelectionEvent event) {
					refresh();

					int selectionCount = 0;
					for (final Button check : stateChecks) {
						if (check.getSelection()) {
							selectionCount++;
						}
					}
					if (selectionCount == stateChecks.size()) {
						allCheckButton.setSelection(true);
					} else if (selectionCount == 0) {
						allCheckButton.setSelection(false);
					}
				}
			});
			stateChecks.add(checkButton);
		}

		expandable.loadExpandedSetting(StoreKey.EXPAND_FILTER_STATE_CHECKS.name());
		expandable.setClient(outer);
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				parent.setVisible(false);
				parent.pack();
				final Point point = parent.getParent().getSize();
				parent.setSize(point);
				parent.setVisible(true);
				expandable.setFocus();
			}
		});
	}

	/**
	 * 「翻訳単位」テーブル・ビューアーのラベル・プロバイダー・クラス。
	 */
	private class TransUnitTableColorLabelProvider extends TableLabelProvider {
		public String getColumnText(final Object element, final int columnIndex) {
			final TransUnitDelegate transUnit = new TransUnitDelegate((IDOMDocument) getInput(), (Element) element);
			switch (columnIndex) {
			case 0:
				return transUnit.getTargetState();
			case 1:
				return transUnit.getSource();
			case 2:
				return transUnit.getTarget();
			}
			return null;
		}
	}

	/**
	 * 「翻訳単位」テーブル・ビューアーの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitTableViewer(final Composite parent) {

		transUnitTableViewer = new AutoColumnWidthTableViewer(parent);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleState(), 16);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleSource(), 52);
		transUnitTableViewer.addColumn(fMsg.getLabelListTitleTarget(), 52);

		transUnitTableViewer.setLabelProvider(new TransUnitTableColorLabelProvider());
		transUnitTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				// 注意：このメソッドはテーブル選択だけでなく、ソース・タブでカーソル移動したときも呼び出されます。
				// そのため、ここでフォーカス制御などの処理を行なうと編集できなくなります。
				if (selection.isEmpty()) {
					return;
				}
				final Element element = (Element) selection.getFirstElement();
				if (element.getFirstChild() == null) {
					// ディスク上のファイルと表示が不整合の場合は中身が null となっているため refresh します。
					refresh();
					return;
				}
				selectedTransUnit = new TransUnitDelegate((IDOMDocument) getInput(), element);
				changeSelectionTransUnit();
				showGlossary();
			}
		});

		final Table table = transUnitTableViewer.getTable();
		final Listener paintListener = new Listener() {
			public void handleEvent(final Event event) {

				final GC gc = event.gc;
				final Display display = getControl().getDisplay();
				final Color oldBackground = gc.getBackground();
				final Color oldForeground = gc.getForeground();

				switch (event.type) {
				case SWT.MeasureItem: {
					final TableItem item = (TableItem) event.item;
					final String text = getText(item, event.index, event);
					final Point size = event.gc.textExtent(text);
					event.width = size.x;

					// デフォルトの高さは値から算出ではなく、2 行分にしておきます。
					event.height = table.getSize().y / 5;
					final int minHeight = event.gc.getFontMetrics().getHeight() * 2;
					if (event.height < minHeight) {
						event.height = minHeight;
					}
					// 最大の高さは 4 行分にしておきます。
					final int maxHeight = event.gc.getFontMetrics().getHeight() * 4;
					if (event.height > maxHeight) {
						event.height = maxHeight;
					}
					// 一度大きくすると行の高さが小さくなりません。
					// http://www.coderanch.com/t/455267/GUI/java/Modify-height-rows-swt-Table
					break;
				}
				case SWT.PaintItem: {
					final TableItem item = (TableItem) event.item;
					final String text = getText(item, event.index, event);
					final Point size = event.gc.textExtent(text);
					final int offset2 = event.index == 0 ? Math.max(0, (event.height - size.y) / 2) : 0;

					// 翻訳対象外：字の色をグレーにします。
					final TransUnitDelegate transUnit = new TransUnitDelegate((IDOMDocument) getInput(),
							(Element) event.item.getData());
					final int fgColor = transUnit.isTranslate() ? SWT.COLOR_BLACK : SWT.COLOR_DARK_GRAY;
					gc.setForeground(display.getSystemColor(fgColor));

					event.gc.drawText(text, event.x, event.y + offset2, true);

					gc.setForeground(oldForeground);
					break;
				}
				case SWT.EraseItem: {
					// テキスト二重表示を消去します。(上記 SWT.PaintItem イベントの drawText に対応する処理)
					event.detail &= ~SWT.FOREGROUND;

					// テーブル行の背景色を設定します。
					try {
						Color colorBackground = null;
						Color colorForeground = null;
						final TransUnitDelegate transUnit = new TransUnitDelegate((IDOMDocument) getInput(),
								(Element) event.item.getData());

						// 翻訳対象：代替翻訳の品質により背景色を設定します。
						if (transUnit.isTranslate()) {
							int quality = 0;
							if (transUnit.getTargetStateQualifier().equals("exact-match")) { //$NON-NLS-1$
								quality = 100;
							} else {
								try {
									final List<AltTransDelegate> altTransList = transUnit.getAltTransList();
									quality = Integer.parseInt(altTransList.get(0).getMatchQuality().replace("%", "")); //$NON-NLS-1$ //$NON-NLS-2$
									if (quality < 0) {
										quality = 0;
									}
								} catch (final RuntimeException e) {
									// 代替翻訳が無い場合や品質未設定の場合は 0
								}
							}
							if (quality >= 100) {
								// 緑
								colorBackground = new Color(display, 170, 255, 100);
							} else if (quality > 0) {
								// 黄 (品質が低いほど薄くします)
								colorBackground = new Color(display, 255, 255, 200 - quality * 2);
							} else {
								// 白
								colorBackground = new Color(display, 255, 255, 255);
							}
							colorForeground = display.getSystemColor(SWT.COLOR_WHITE);

						} else {
							// 翻訳対象外：背景色を茶色(焦茶こげちゃ)にします。
							colorBackground = new Color(display, 0x6f, 0x4b, 0x3e);
							colorForeground = display.getSystemColor(SWT.COLOR_WHITE);
						}

						gc.setBackground(colorBackground);
						gc.setForeground(colorForeground);
						gc.fillRectangle(0, event.y, table.getClientArea().width, event.height);

						// 色を元に戻します。
						gc.setBackground(oldBackground);
						gc.setForeground(oldForeground);

					} catch (final RuntimeException e) {

						// 1番目の品質が取得できない場合、背景はデフォルトの白のままにしておきます。
						CatUiPlugin.getDefault().log(e);
					}
					break;
				}
				}
			}

			String getText(final TableItem item, final int columnIndex, final Event event) {

				// 表示用に改行を除去します。
				String text = item.getText(columnIndex);
				text = text.replaceAll("[\r\n]+", " "); //$NON-NLS-1$ //$NON-NLS-2$

				final int columnWidth = table.getColumn(columnIndex).getWidth() - 10;

				if (event.gc.textExtent(text).x > columnWidth) {

					final StringBuilder rows = new StringBuilder();
					StringBuilder row = new StringBuilder();

					// 列幅に合わせて改行を挿入します。
					for (final char c : text.toCharArray()) {
						row.append(c);
						if (event.gc.textExtent(row.toString()).x > columnWidth) {
							rows.append(row);
							rows.append("\n"); //$NON-NLS-1$
							row = new StringBuilder();
						}
					}
					if (row.length() > 0) {
						rows.append(row);
					}
					text = rows.toString();
				}
				return text;
			}
		};
		table.addListener(SWT.MeasureItem, paintListener);
		table.addListener(SWT.PaintItem, paintListener);
		table.addListener(SWT.EraseItem, paintListener);
	}

	/**
	 * 選択翻訳単位の変更。
	 */
	protected void changeSelectionTransUnit() {

		updateTransUnitButtons();
		updateNoteButtons();

		if (selectedTransUnit == null) {
			return;
		}
		setEnabledTargetAttribute(true);
		setTranslationable(selectedTransUnit.isTranslate());

		final List<NoteDelegate> noteList = selectedTransUnit.getNoteList();
		final List<AltTransDelegate> altTransList = selectedTransUnit.getAltTransList();

		noteTableViewer.setInput(ElementDelegate.toElementList(noteList));
		idText.setText(selectedTransUnit.getId());
		stateCombo.setText(createStateLabel(selectedTransUnit.getTargetState()));
		notTranslateCheck.setSelection(!selectedTransUnit.isTranslate());
		updateTmCheck.setSelection(!selectedTransUnit.isContextGourpTmOmit());
		sourceCompareViewer.setLeftContent(selectedTransUnit.getSource());
		targetCompareViewer.setLeftContent(selectedTransUnit.getTarget());

		if (altTransList.size() == 0) {
			clearSelectedAltTrans();
		} else {
			altTransTableViewer.setInput(ElementDelegate.toElementList(altTransList));
			altTransTableViewer.getTable().setSelection(0);
			setSelectedAltTrans(altTransList.get(0));
		}
		updateAltTransButtons();
		updateOutlineView(new StructuredSelection(selectedTransUnit.getElement()));

		sourceCompareViewer.fireContentChange();
		targetCompareViewer.fireContentChange();
	}

	/**
	 * 用語集の表示。
	 */
	public void showGlossary() {
		if (selectedTransUnit == null) {
			return;
		}
		final ShowGlossaryAction action = new ShowGlossaryAction();
		action.selectionChanged(null, new TextSelection(0, 0) {
			@Override
			public String getText() {
				return selectedTransUnit.getSource();
			}
		});
		action.run(null);
	}

	/**
	 * 状態コンボの表示文字列を作成。
	 * @param label ラベル
	 * @return ラベルにキー・バインド文字列を追加文字列
	 */
	protected String createStateLabel(final String label) {
		if (label.length() > 0) {
			final char key = stateKeyBindMap.get(label);
			return "(" + key + ") " + label; //$NON-NLS-1$ //$NON-NLS-2$
		}
		return label;
	}

	/**
	 * 「翻訳単位」の属性の作成。翻訳対象外のチェックボックスなどが含まれる。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitAttributes(final Composite parent) {

		Composite c = toolkit.createComposite(parent, 2);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		toolkit.createLabel(c, fMsg.getLabelId(), 50);
		idText = new Text(c, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
		idText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		c = toolkit.createComposite(parent, 4);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		final String[] stateItems = BentenXliff.getStateItems();
		for (int i = 0; i < stateItems.length; i++) {
			final String label = stateItems[i];
			final char key = stateKeyBindChars[i];
			stateKeyBindMap.put(label, key);
			stateItems[i] = createStateLabel(label);
		}

		toolkit.createLabel(c, fMsg.getLabelState(), 50);
		stateCombo = new Combo(c, SWT.READ_ONLY);
		stateCombo.setItems(stateItems);
		stateCombo.addModifyListener(new ModifyListener() {
			public void modifyText(final ModifyEvent e) {
				if (selectedTransUnit == null || !targetCompareViewer.getLeftSourceViewer().isEditable()) {
					return;
				}
				final String state = ((Combo) e.getSource()).getText();
				if (selectedTransUnit.setTargetState(state.replaceFirst("^.+ ", ""))) { //$NON-NLS-1$ //$NON-NLS-2$
					transUnitTableViewer.update(selectedTransUnit.getElement(), null);
				}
			}
		});

		GridData gd = new GridData();
		gd.horizontalAlignment = SWT.RIGHT;
		gd.grabExcessHorizontalSpace = true;
		notTranslateCheck = new Button(c, SWT.CHECK);
		notTranslateCheck.setLayoutData(gd);
		notTranslateCheck.setText(fMsg.getLabelOutOfTranslation());
		notTranslateCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				selectedTransUnit.setTranslate(!checked);
				setTranslationable(!checked);
				updateNoteButtons();
				targetCompareViewer.fireContentChange();
				transUnitTableViewer.refresh();
			}
		});

		gd = new GridData();
		gd.horizontalAlignment = SWT.RIGHT;
		gd.horizontalIndent = 30;
		updateTmCheck = new Button(c, SWT.CHECK);
		updateTmCheck.setLayoutData(gd);
		updateTmCheck.setText(fMsg.getLabelApplyToTM());
		updateTmCheck.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final boolean checked = ((Button) e.widget).getSelection();
				selectedTransUnit.setContextGroupTmOmit(!checked);
			}
		});
	}

	/**
	 * 「翻訳単位」のテキスト部の作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitText(final Composite parent) {

		final Composite composite = toolkit.createComposite(parent, 1);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		final CompareConfiguration compareConfiguration = new CompareConfiguration();

		final TransCompareContentProvider sourceContentProvider = new TransCompareContentProvider(compareConfiguration,
				fMsg.getLabelSource(), fMsg.getLabelSelectedAltTransSource());
		sourceCompareViewer = new TransCompareViewer(composite, compareConfiguration);
		sourceCompareViewer.setContentProvider(sourceContentProvider);
		sourceCompareViewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));

		copyFromSourceButton = new Button(composite, SWT.ARROW | SWT.DOWN | SWT.BORDER);
		copyFromSourceButton.setVisible(false);
		copyFromSourceButton.setLayoutData(toolkit.createWidthGridData(30));
		copyFromSourceButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final String source = sourceCompareViewer.getLeftSourceViewer().getDocument().get();
				targetCompareViewer.getLeftSourceViewer().getDocument().set(source);
			}
		});

		final TransCompareContentProvider targetContentProvider = new TransCompareContentProvider(compareConfiguration,
				fMsg.getLabelTarget(), fMsg.getLabelSelectedAltTransTarget());
		targetContentProvider.setLeftEditable(true);
		targetCompareViewer = new TransCompareViewer(composite, compareConfiguration);
		targetCompareViewer.setContentProvider(targetContentProvider);
		targetCompareViewer.getControl().setLayoutData(new GridData(GridData.FILL_BOTH));

		final SourceViewer targetLeftViewer = targetCompareViewer.getLeftSourceViewer();
		targetLeftViewer.unconfigure();
		targetLeftViewer.configure(new GlossaryConfiguration());
		targetLeftViewer.addTextListener(new ITextListener() {
			public void textChanged(final TextEvent event) {
				Display.getCurrent().asyncExec(new Runnable() {
					public void run() {
						try {
							if (selectedTransUnit == null || !targetLeftViewer.isEditable()) {
								return;
							}
							final DocumentEvent documentEvent = event.getDocumentEvent();
							if (documentEvent != null) {
								final String target = documentEvent.getDocument().get();
								if (selectedTransUnit.setTarget(target)) {
									transUnitTableViewer.update(selectedTransUnit.getElement(), null);
								}
							}
						} catch (final RuntimeException e) {
							// ターゲットが取得できない場合は何もしません。
							CatUiPlugin.getDefault().log(e);
						}
					}
				});
			}
		});
	}

	/**
	 * ビューアーのフォントを設定。
	 * @param parent 親オブジェクト。
	 * @param viewer ビューアー。
	 */
	protected void setViewerFont(final Composite parent, final Viewer viewer) {

		final IProject project = ((IFileEditorInput) editorPart.getEditorInput()).getFile().getProject();
		final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);
		final String sourceLang = projectStore.getString(ProjectProperty.TRANS_SOURCE_LANG.name());
		final String targetLang = projectStore.getString(ProjectProperty.TRANS_TARGET_LANG.name());

		// TICKET #21759 XLIFFエディタで中国語、ベトナム語が文字化け
		// フォントが Monospaced の場合に発生するということで、プロジェクトの翻訳元、翻訳先の言語が
		// 両方とも en- または ja- で始める場合のみ Monospaced にします。
		if (sourceLang.matches("^(en|ja)-.+") && targetLang.matches("^(en|ja)-.+")) { //$NON-NLS-1$ //$NON-NLS-2$
			final Font mono = new Font(parent.getDisplay(), "Monospaced", 10, SWT.NONE); //$NON-NLS-1$
			viewer.getControl().setFont(mono);
		}
	}

	/**
	 * 「代替翻訳」テーブルの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createAltTransTableViewer(final Composite parent) {
		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelAltTransList());

		final Composite c = toolkit.createComposite(expandable, 2);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		altTransTableViewer = new AutoColumnWidthTableViewer(c);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleOrigin(), 12);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleQuality(), 6);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleSource(), 41);
		altTransTableViewer.addColumn(fMsg.getLabelAltTransTitleTarget(), 41);

		expandable.loadExpandedSetting(StoreKey.EXPAND_ALT_TRANS_TABLE.name());
		expandable.setClient(c);
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				packTable(parent, altTransTableViewer);
			}
		});

		altTransTableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(final Object element, final int columnIndex) {
				if (!(element instanceof Element)) {
					return null;
				}
				final AltTransDelegate altTrans = new AltTransDelegate((IDOMDocument) getInput(), (Element) element);
				switch (columnIndex) {
				case 0:
					return altTrans.getOrigin();
				case 1:
					return altTrans.getMatchQuality();
				case 2:
					return altTrans.getSource();
				case 3:
					return altTrans.getTarget();
				}
				return null;
			}
		});

		altTransTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				if (selection.isEmpty()) {
					return;
				}
				final Element firstElement = (Element) selection.toArray()[0];
				final AltTransDelegate altTrans = new AltTransDelegate((IDOMDocument) getInput(), firstElement);

				setSelectedAltTrans(altTrans);
				updateAltTransButtons();
				updateOutlineView(selection);

				sourceCompareViewer.fireContentChange();
				targetCompareViewer.fireContentChange();
			}
		});

		// 展開したときに中身がないと高さが 0 に近いものになってしまうのを抑止
		final Label label = toolkit.createLabel(c, ""); //$NON-NLS-1$
		final GridData gd = new GridData();
		final int lineHeight = (int) (new GC(label).getFontMetrics().getHeight() * 1.6);
		gd.heightHint = lineHeight * 5;
		label.setLayoutData(gd);
	}

	/**
	 * 「ノート」テーブル・ビューアの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createNoteTableViewer(final Composite parent) {
		final PersistenceExpandableComposite expandable = new PersistenceExpandableComposite(parent, fMsg
				.getLabelNoteList());
		final Composite c = toolkit.createComposite(expandable, 2);
		c.setLayoutData(new GridData(GridData.FILL_BOTH));

		noteTableViewer = new AutoColumnWidthTableViewer(c);
		noteTableViewer.addColumn(fMsg.getLabelNoteTitleTranslator(), 20);
		noteTableViewer.addColumn(fMsg.getLabelNoteTitleContents(), 80);

		expandable.loadExpandedSetting(StoreKey.EXPAND_NOTE_LIST_TABLE.name());
		expandable.setClient(c);
		expandable.addExpansionListener(new ExpansionAdapter() {
			@Override
			public void expansionStateChanged(final ExpansionEvent e) {
				packTable(parent, noteTableViewer);
			}
		});

		noteTableViewer.setLabelProvider(new TableLabelProvider() {
			public String getColumnText(final Object element, final int columnIndex) {
				final NoteDelegate note = new NoteDelegate((IDOMDocument) getInput(), (Element) element);
				switch (columnIndex) {
				case 0:
					return note.getFrom();
				case 1:
					return note.getValue();
				}
				return null;
			}
		});
		noteTableViewer.addPostSelectionChangedListener(new AsyncSelectionChangeAdapter() {
			@Override
			public void selectionChanged(final IStructuredSelection selection) {
				if (!selection.isEmpty()) {
					selectedNote = new NoteDelegate((IDOMDocument) getInput(), (Element) selection.getFirstElement());
				}
				updateNoteButtons();
				updateOutlineView(selection);
			}
		});
		noteTableViewer.addDoubleClickListener(new IDoubleClickListener() {
			public void doubleClick(final DoubleClickEvent event) {
				editNoteDialog(parent);
			}
		});

		final Composite buttonOuter = toolkit.createComposite(c, 1);

		newNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteNew(), 60);
		newNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				final IPreferenceStore store = CatUiPlugin.getDefault().getPreferenceStore();
				final String from = store.getString(StoreKey.DEFAULT_NOTE_FROM.name());

				final NoteDialogEntry note = new NoteDialogEntry(from, ""); //$NON-NLS-1$
				final NoteDialog dialog = new NoteDialog(parent.getShell(), fMsg.getLabelNoteNewText(), note);

				if (dialog.open() == Window.OK) {
					final Node noteElement = selectedTransUnit.appendNote(note.from, note.content);
					noteTableViewer.add(noteElement);
					store.setValue(StoreKey.DEFAULT_NOTE_FROM.name(), note.from);
				}
			}
		});
		editNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteEdit(), 60);
		editNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				editNoteDialog(parent);
			}
		});
		removeNoteButton = toolkit.createButton(buttonOuter, fMsg.getLabelNoteRemove(), 60);
		removeNoteButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent event) {
				selectedTransUnit.removeNote(selectedNote);
				final Table table = noteTableViewer.getTable();
				final int index = table.getSelectionIndex();

				table.remove(index);
				if (index > 0) {
					noteTableViewer
							.setSelection(new StructuredSelection(noteTableViewer.getElementAt(index - 1)), true);
				} else {
					noteTableViewer.setSelection(new StructuredSelection(), true);
				}
			}
		});
	}

	/**
	 * ノートをノート・ダイアログによって編集。
	 *
	 * <UL>
	 * <LI>選択されたノートをノートダイアログで編集します。
	 * <LI>万が一「selectedNote」が null のままこのメソッドが呼び出された場合には、何もせずに処理を戻します。(環境由来のバグ回避策)
	 * </UL>
	 * @param parent 親オブジェクト。
	 */
	protected void editNoteDialog(final Composite parent) {
		if (selectedNote == null) {
			// 環境によって selectedNote が null のまま このメソッドが呼び出される場合があります。
			// その場合には、何もせずに処理を終了します。
			return;
		}

		final NoteDialogEntry note = new NoteDialogEntry(selectedNote.getFrom(), selectedNote.getValue());
		note.modify = true;
		final NoteDialog dialog = new NoteDialog(parent.getShell(), fMsg.getLabelEditNote(), note);

		if (dialog.open() == Window.OK) {
			selectedNote.setFrom(note.from);
			selectedNote.setValue(note.content);
			noteTableViewer.update(selectedNote.getElement(), null);
		}
	}

	/**
	 * 翻訳単位のためのボタンの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createTransUnitButtons(final Composite parent) {

		final Composite c = toolkit.createComposite(parent, 5);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		joinTransUnitButton = toolkit.createButton(c, fMsg.getLabelJoinToNextTransUnit(), 170);

		toolkit.createHorizontalSpacer(c);

		transUnitCountLabel = toolkit.createLabel(c, "", 120); //$NON-NLS-1$
		transUnitCountLabel.setAlignment(SWT.RIGHT);

		prevTransUnitButton = toolkit.createButton(c, fMsg.getLabelPrevTransUnit(), 140);
		nextTransUnitButton = toolkit.createButton(c, fMsg.getLabelNextTransUnit(), 140);

		joinTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				selectedTransUnit.joinNextTransUnit();
				refresh();
			}
		});
		prevTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				scrollTransUnitSelection(-1);
			}
		});
		nextTransUnitButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				scrollTransUnitSelection(1);
			}
		});
	}

	/**
	 * 翻訳単位の選択を移動。
	 * @param move 移動距離。
	 */
	protected void scrollTransUnitSelection(final int move) {
		final Table table = transUnitTableViewer.getTable();
		final int index = table.getSelectionIndex();
		final Object dest = transUnitTableViewer.getElementAt(index + move);
		if (dest != null) {
			transUnitTableViewer.setSelection(new StructuredSelection(dest), true);
		} else {
			clearSelectedTransUnit();
		}
	}

	/**
	 * 「代替翻訳」のボタンの作成。
	 * @param parent 親オブジェクト。
	 */
	protected void createAltTransButtons(final Composite parent) {

		final Composite c = toolkit.createComposite(parent, 6);
		c.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		diffButton = new Button(c, SWT.TOGGLE);
		diffButton.setLayoutData(toolkit.createWidthGridData(140));
		diffButton.setText(fMsg.getLabelShowDifferences());
		diffButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Button b = (Button) e.widget;
				sourceCompareViewer.setEnabledDiff(b.getSelection());
				targetCompareViewer.setEnabledDiff(b.getSelection());
			}
		});

		originLabel = new Label(c, SWT.RIGHT);
		originLabel.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

		altTransQualityBar = new ProgressBar(c, SWT.SMOOTH);
		altTransQualityBar.setLayoutData(toolkit.createWidthGridData(80));
		altTransQualityBar.setMinimum(0);
		altTransQualityBar.setMaximum(100);

		altTransCountLabel = toolkit.createLabel(c, "", 60); //$NON-NLS-1$
		altTransCountLabel.setAlignment(SWT.RIGHT);

		prevAltTransButton = toolkit.createButton(c, fMsg.getLabelPrevAltTrans(), 140);
		nextAltTransButton = toolkit.createButton(c, fMsg.getLabelNextAltTrans(), 140);

		prevAltTransButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Table table = altTransTableViewer.getTable();
				final int index = table.getSelectionIndex();
				if (index > 0) {
					altTransTableViewer.setSelection(new StructuredSelection(altTransTableViewer
							.getElementAt(index - 1)), true);
				}
				updateAltTransButtons();
				if (!prevAltTransButton.getEnabled()) {
					nextAltTransButton.setFocus();
				}
			}
		});
		nextAltTransButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(final SelectionEvent e) {
				final Table table = altTransTableViewer.getTable();
				final int index = table.getSelectionIndex();
				if (index < table.getItemCount() - 1) {
					altTransTableViewer.setSelection(new StructuredSelection(altTransTableViewer
							.getElementAt(index + 1)), true);
				}
				updateAltTransButtons();
				if (!nextAltTransButton.getEnabled()) {
					prevAltTransButton.setFocus();
				}
			}
		});
	}

	//-------------------------------------------------------------------------
	// 可視制御

	/**
	 * テーブルをパック。
	 * @param parent 親コンポジット
	 * @param tableViewer テーブル・ビューアー
	 */
	protected void packTable(final Composite parent, final TableViewer tableViewer) {

		parent.pack();
		final Point point = parent.getParent().getSize();
		parent.setSize(point.x, point.y - 23); // 少しはみ出るため調整

		// 第一カラムを再セットすることで、リサイズ時に横スクロールバーが表示されるのを抑止
		// → TableColumnLayout のバグ扱いとし、対応しない 2009.07.10
		//final TableColumn column = tableViewer.getTable().getColumn(0);
		//column.setWidth(column.getWidth());
	}

	/**
	 * 「翻訳単位」の可視制御。
	 * @param enabled 有効化するかどうか。
	 */
	protected void setEnabledTargetAttribute(final boolean enabled) {
		notTranslateCheck.setEnabled(enabled);
		updateTmCheck.setEnabled(enabled);
	}

	/**
	 * 「翻訳」関連ボタンの更新。
	 * @param translationable 翻訳対象の場合は true
	 */
	protected void setTranslationable(final boolean translationable) {
		stateCombo.setEnabled(translationable);
		targetCompareViewer.setLeftEditable(translationable);
		copyFromSourceButton.setVisible(translationable);
	}

	/**
	 * 「翻訳単位」のボタンを更新。
	 */
	protected void updateTransUnitButtons() {
		if (joinTransUnitButton == null) {
			return;
		}
		final boolean hasContinue = selectedTransUnit != null && HelpTransUnitId.hasContinue(selectedTransUnit.getId());
		joinTransUnitButton.setEnabled(hasContinue);

		final Table table = transUnitTableViewer.getTable();
		final int index = table.getSelectionIndex();
		prevTransUnitButton.setEnabled(index > 0);
		nextTransUnitButton.setEnabled(index < table.getItemCount() - 1);

		final int row = table.getSelectionIndex() + 1;
		final String s = row + " / " + table.getItemCount(); //$NON-NLS-1$
		transUnitCountLabel.setText(s);
	}

	/**
	 * 選択した「翻訳単位」をクリアー。
	 */
	protected void clearSelectedTransUnit() {

		selectedTransUnit = null;

		setEnabledTargetAttribute(false);
		setTranslationable(false);

		noteTableViewer.setInput(Collections.EMPTY_LIST);
		idText.setText(""); //$NON-NLS-1$
		stateCombo.setText(""); //$NON-NLS-1$

		sourceCompareViewer.setLeftContent(""); //$NON-NLS-1$
		targetCompareViewer.setLeftContent(""); //$NON-NLS-1$

		clearSelectedAltTrans();
		updateAltTransButtons();

		sourceCompareViewer.fireContentChange();
		targetCompareViewer.fireContentChange();
	}

	/**
	 * 選択した「代替翻訳」をクリアー。
	 */
	protected void clearSelectedAltTrans() {

		sourceCompareViewer.setRightContent(""); //$NON-NLS-1$
		targetCompareViewer.setRightContent(""); //$NON-NLS-1$

		altTransTableViewer.setInput(Collections.EMPTY_LIST);
		originLabel.setText(""); //$NON-NLS-1$
		altTransQualityBar.setVisible(false);
		altTransCountLabel.setText(""); //$NON-NLS-1$
	}

	/**
	 * 選択した「代替翻訳」をセット。
	 * @param altTrans 代替翻訳。
	 */
	protected void setSelectedAltTrans(final AltTransDelegate altTrans) {

		sourceCompareViewer.setRightContent(altTrans.getSource());
		targetCompareViewer.setRightContent(altTrans.getTarget());

		final String quality = altTrans.getMatchQuality();
		try {
			final int qualityInt = Integer.parseInt(quality.replace("%", "")); //$NON-NLS-1$ //$NON-NLS-2$

			// 一旦、NORMAL に戻さないとバーの長さが正常に更新されません。
			altTransQualityBar.setState(SWT.NORMAL);
			altTransQualityBar.setSelection(qualityInt);
			altTransQualityBar.setState(qualityInt == 100 ? SWT.NORMAL : SWT.PAUSED);
			altTransQualityBar.setVisible(true);
		} catch (final RuntimeException e) {
			altTransQualityBar.setSelection(0);
			altTransQualityBar.setVisible(false);
		}

		String text = altTrans.getOrigin();
		if (quality != null && !quality.equals("")) { //$NON-NLS-1$
			text += ": " + quality; //$NON-NLS-1$
		}
		originLabel.setText(text);

		final Table table = altTransTableViewer.getTable();
		final int row = table.getSelectionIndex() + 1;
		final String s = row + " / " + table.getItemCount(); //$NON-NLS-1$
		altTransCountLabel.setText(s);
	}

	/**
	 * 「代替翻訳」のボタンを更新。
	 */
	protected void updateAltTransButtons() {
		final Table table = altTransTableViewer.getTable();
		final int index = table.getSelectionIndex();
		prevAltTransButton.setEnabled(index > 0);
		nextAltTransButton.setEnabled(index < table.getItemCount() - 1);
	}

	/**
	 * 「ノート」のボタンを更新。
	 */
	protected void updateNoteButtons() {
		final boolean canCreate = (selectedTransUnit != null) && selectedTransUnit.isTranslate();
		final boolean canEdit = canCreate && !noteTableViewer.getSelection().isEmpty();
		editNoteButton.setEnabled(canEdit);
		removeNoteButton.setEnabled(canEdit);
		newNoteButton.setEnabled(canCreate);
	}

	/**
	 * 「アウトライン」ビューの更新。
	 * @param selection 選択。
	 */
	protected void updateOutlineView(final ISelection selection) {
		final ConfigurableContentOutlinePage outline = (ConfigurableContentOutlinePage) editorPart
				.getAdapter(IContentOutlinePage.class);
		if (outline.getConfiguration().isLinkedWithEditor(null)) {
			outline.setSelection(selection);
		}
	}

	//-------------------------------------------------------------------------
	// org.eclipse.wst.xml.ui.internal.tabletree.IDesignViewer の実装

	/**
	 * CTabItem 上に表示するコントロールを取得。
	 * @return コントロール
	 */
	@Override
	public Control getControl() {
		return control;
	}

	/**
	 * 選択プロバイダーを取得。
	 * @return 選択プロバイダー
	 */
	public ISelectionProvider getSelectionProvider() {
		return selectionProvider;
	}

	/**
	 * ビューアーのタブ・タイトルを取得。
	 * @return タブ・タイトル
	 */
	public String getTitle() {
		return fMsg.getViewerTitle();
	}

	/**
	 * ドキュメントを設定。
	 * @param document ドキュメント
	 */
	public void setDocument(final IDocument document) {

		IStructuredModel model = null;
		try {
			model = StructuredModelManager.getModelManager().getExistingModelForRead(document);
			if (model == null) {
				// エディターのクローズで呼び出されたときは、null です。何も行ないません。
				return;
			}
			final IDOMDocument domDoc = ((IDOMModel) model).getDocument();

			// ファイルを開いたときにリフレッシュが必要な状態の場合、読み込みます。
			if (domDoc.getFirstChild() == null) {
				final IFile file = ((IFileEditorInput) editorPart.getEditorInput()).getFile();
				file.refreshLocal(0, null);
				model = StructuredModelManager.getModelManager().getModelForRead(file);
			}

			// 翻訳単位リスト・ビューが開かれていない場合、開きます。
			showTransUnitListView();

			setInput(domDoc);
			refresh();

			// EMF コマンド・スタックを拡張し、UNDO、REDO 後に refresh します。
			final IStructuredTextUndoManager undoManager = model.getUndoManager();
			final CommandStack commandStack = undoManager.getCommandStack();
			undoManager.setCommandStack(new CommandStackDelegate(commandStack) {
				@Override
				public void redo() {
					super.redo();
					refreshTransUnit();
				}

				@Override
				public void undo() {
					super.undo();
					if (!canUndo()) {
						// ダーティーフラグを解除します。
						editorPart.doSave(null);
					}
					refreshTransUnit();
				}

				private void refreshTransUnit() {
					changeSelectionTransUnit();
					transUnitTableViewer.refresh();
				}
			});

		} catch (final IOException e) {
			CatUiPlugin.getDefault().log(e);

		} catch (final CoreException e) {
			CatUiPlugin.getDefault().log(e);

		} finally {
			if (model != null) {
				model.releaseFromRead();
			}
		}
	}

	private void showTransUnitListView() {
		Display.getCurrent().asyncExec(new Runnable() {
			public void run() {
				final IWorkbenchPage page = editorPart.getSite().getPage();
				try {
					final IPartListener view = (IPartListener) page.showView(TransUnitListView.class.getName());
					if (transUnitTableViewer == null) {
						view.partActivated(editorPart);
					}
				} catch (final PartInitException e) {
					CatUiPlugin.getDefault().log(e);
				}
			}
		});
	}

	//-------------------------------------------------------------------------
	// org.eclipse.jface.viewers.Viewer オーバーライド

	/**
	 * このビューアーをリフレッシュ。
	 *
	 * <UL>
	 * <LI>LabelProvider や ContentProvider が設定されたときなどに呼び出されます。
	 * </UL>
	 */
	@Override
	public void refresh() {

		final IDOMDocument domDoc = (IDOMDocument) getInput();
		if (domDoc == null) {
			return;
		}

		final List<TransUnitDelegate> allTransUnitList = TransUnitDelegate.listOf(domDoc);
		final List<TransUnitDelegate> transUnitList = new LinkedList<TransUnitDelegate>();

		final Set<String> checkedStateItems = new HashSet<String>();
		for (final Button check : stateChecks) {
			if (check.isDisposed()) {
				return;
			}
			if (check.getSelection()) {
				checkedStateItems.add((String) check.getData());
			}
		}

		for (final TransUnitDelegate transUnit : allTransUnitList) {
			final String state = transUnit.getTargetState();
			if (checkedStateItems.contains(state)) {
				transUnitList.add(transUnit);
			}
		}

		if (transUnitTableViewer != null && transUnitTableViewer.getContentProvider() != null) {
			transUnitTableViewer.setInput(ElementDelegate.toElementList(transUnitList));
		}
		if (transUnitList.size() == 0) {

			clearSelectedTransUnit();

		} else {

			if (transUnitTableViewer != null && transUnitTableViewer.getSelection().isEmpty()) {

				final Object firstElement = transUnitTableViewer.getElementAt(0);
				final StructuredSelection sel = new StructuredSelection(firstElement);
				transUnitTableViewer.setSelection(sel, true);
				final IContentOutlinePage outline = (IContentOutlinePage) editorPart
						.getAdapter(IContentOutlinePage.class);
				if (outline != null) {
					outline.setSelection(sel);
				}
			}
			TransUnitDelegate newSelectedTransUnit = transUnitList.get(0);
			if (selectedTransUnit != null) {
				for (final TransUnitDelegate transUnit : transUnitList) {
					if (selectedTransUnit.getId().equals(transUnit.getId())) {
						newSelectedTransUnit = transUnit;
					}
				}
			}
			selectedTransUnit = newSelectedTransUnit;

			changeSelectionTransUnit();
			final List<NoteDelegate> noteList = selectedTransUnit.getNoteList();
			if (noteList.size() == 0) {
				selectedNote = null;
			} else {
				selectedNote = new NoteDelegate((IDOMDocument) getInput(), noteList.get(0).getElement());
			}
		}
		if (transUnitTableViewer != null) {
			transUnitTableViewer.refresh();
		}
	}

	/**
	 * このビューアーの選択を取得。
	 *
	 * <UL>
	 * <LI>このビューアーがアクティブになった時に呼び出されます。
	 * <LI>この選択はアウトライン・ビューなどに反映されます。
	 * </UL>
	 */
	@Override
	public ISelection getSelection() {

		refresh();

		CatUiPlugin.getDefault().setActiveEditorPart(editorPart);

		if (transUnitTableViewer == null) {
			return new StructuredSelection();
		}
		final ISelection selection = transUnitTableViewer.getSelection();

		// ここでは代替翻訳の選択は利用しない
		//ISelection selection = altTransTableViewer.getSelection();
		//if (selection.isEmpty()) {
		//	selection = transUnitTableViewer.getSelection();
		//}

		return selection;
	}

	/**
	 * このビューアーの選択を設定。
	 *
	 * <UL>
	 * <LI>ソース・タブやアウトライン・ビュー選択時に呼び出されます。
	 * </UL>
	 */
	@Override
	public void setSelection(final ISelection selection, final boolean reveal) {

		if (!selection.isEmpty() && selection instanceof IStructuredSelection) {

			final IStructuredSelection structuredSelection = (IStructuredSelection) selection;
			final Node firstElement = (Node) structuredSelection.getFirstElement();
			final Node node = firstElement;
			Node transUnitNode = null;
			Node altTransNode = null;
			Node noteNode = null;

			for (Node n = node; n != null; n = n.getParentNode()) {
				final String nodeName = n.getNodeName();
				if (noteNode == null && nodeName.equals("note")) { //$NON-NLS-1$
					noteNode = n;
				} else if (altTransNode == null && nodeName.equals("alt-trans")) { //$NON-NLS-1$
					altTransNode = n;
				} else if (transUnitNode == null && nodeName.equals("trans-unit")) { //$NON-NLS-1$
					transUnitNode = n;
					break;
				}
			}
			if (transUnitNode != null && transUnitTableViewer != null && !transUnitTableViewer.getTable().isDisposed()
					&& !transUnitTableViewer.getTable().isFocusControl()) {
				final ISelection sel = new StructuredSelection(transUnitNode);
				transUnitTableViewer.setSelection(sel, true);
			}
			if (altTransNode != null && altTransTableViewer != null && !altTransTableViewer.getTable().isDisposed()
					&& !altTransTableViewer.getTable().isFocusControl()) {
				final ISelection sel = new StructuredSelection(altTransNode);
				altTransTableViewer.setSelection(sel, true);
			}
			if (noteNode != null && noteTableViewer != null && !noteTableViewer.getTable().isDisposed()
					&& !noteTableViewer.getTable().isFocusControl()) {
				final ISelection sel = new StructuredSelection(noteNode);
				noteTableViewer.setSelection(sel, true);
			}
		}
	}

	/**
	 * ターゲット・ビューアーの取得。
	 * @return ターゲット・ビューアー
	 */
	public ISourceViewer getTargetViewer() {
		return targetCompareViewer.getLeftSourceViewer();
	}
}
