/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.validation.validator;

import junit.framework.TestCase;

import org.eclipse.wst.validation.internal.provisional.core.IMessage;

public class TranslationValidatorTest extends TestCase {

	private final TranslationValidator validator = new TranslationValidator();

	private TranslationValidationReporterStub reporter;

	private static class TranslationValidationReporterStub extends TranslationValidationReporter {
		public int serverity;
		public String id;
		public String[] params;

		public TranslationValidationReporterStub() {
			super(null, null, null, null);
		}

		@Override
		protected void add(final int serverity, final int targetOffset, final int length, final String id,
				final String... params) {

			this.serverity = serverity;
			this.id = id;
			this.params = params;
		}
	};

	private void validateTarget(final String source, final String target) {
		reporter = new TranslationValidationReporterStub();
		validator.validateTarget(reporter, source, target, 0, 0);
	}

	//-------------------------------------------------------------------------

	public void testValidateTarget() {
		try {
			validateTarget(null, "");
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		try {
			validateTarget("", null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}

		//-----------------------------------------------------------
		// 翻訳忘れチェック

		validateTarget("", "");
		assertEquals(IMessage.LOW_SEVERITY, reporter.serverity);
		assertEquals("NotTranslated", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test", "");
		assertEquals(IMessage.LOW_SEVERITY, reporter.serverity);
		assertEquals("NotTranslated", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test", "test");
		assertEquals(IMessage.LOW_SEVERITY, reporter.serverity);
		assertEquals("NotTranslated", reporter.id);
		assertEquals(0, reporter.params.length);

		//-----------------------------------------------------------
		// 末尾の同一性チェック

		validateTarget("test...", "テスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("ThreeFullStopEllipsisNeeded", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test...", "テスト.");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("ThreeFullStopEllipsisNeeded", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test...", "テスト..");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("ThreeFullStopEllipsisNeeded", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test...", "テスト...");
		assertNotSame("ThreeFullStopEllipsisNeeded", reporter.id);

		//-----------------------------------------------------------

		validateTarget("test text.", "テスト・テキスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("KutenNeeded", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test text.", "テスト・テキスト。");
		assertNotSame("KutenNeeded", reporter.id);

		validateTarget("test text: A.", "テスト・テキスト: あ");
		assertNotSame("KutenNeeded", reporter.id);

		validateTarget("test.", "テスト");
		assertNotSame("KutenNeeded", reporter.id);

		//-----------------------------------------------------------

		validateTarget("test:", "テスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("SemiColonNeeded", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("test:", "テスト:");
		assertNotSame("SemiColonNeeded", reporter.id);

		//-----------------------------------------------------------
		// 英数字隣接空白チェック

		validateTarget("", "あ1");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("AdjacentSpaceCheck", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("", "1あ");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("AdjacentSpaceCheck", reporter.id);
		assertEquals(0, reporter.params.length);

		validateTarget("", "（1");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "「1");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "1）");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "1」");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "1、");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "1。");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		validateTarget("", "1･･･");
		assertNotSame("AdjacentSpaceCheck", reporter.id);

		//-----------------------------------------------------------
		// 埋め込み引数の出現数チェック

		validateTarget("line {0}", "行");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("IlligalVariableParameter", reporter.id);
		assertEquals(1, reporter.params.length);
		assertEquals("{0}", reporter.params[0]);

		validateTarget("line {0}-{1}", "行 {0}");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("IlligalVariableParameter", reporter.id);
		assertEquals(1, reporter.params.length);
		assertEquals("{1}", reporter.params[0]);

		validateTarget("line {0}-{1}", "行 {0}-{2}");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("IlligalVariableParameter", reporter.id);
		assertEquals(1, reporter.params.length);
		assertEquals("{1}", reporter.params[0]);

		validateTarget("line {0}", "行 {0}");
		assertNotSame("IlligalVariableParameter", reporter.id);

		validateTarget("line {0}-{1}", "行 {0}-{1}");
		assertNotSame("IlligalVariableParameter", reporter.id);

		//-----------------------------------------------------------
		// 改行数チェック

		validateTarget("test\ntext", "テスト テキスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("UnmatchedLineDelimiter", reporter.id);
		assertEquals(2, reporter.params.length);
		assertEquals("1", reporter.params[0]);
		assertEquals("0", reporter.params[1]);

		validateTarget("test text", "テスト\nテキスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("UnmatchedLineDelimiter", reporter.id);
		assertEquals(2, reporter.params.length);
		assertEquals("0", reporter.params[0]);
		assertEquals("1", reporter.params[1]);

		validateTarget("test\ntext\n\ntest\n\n\ntext", "テスト\n\n\nテキスト");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("UnmatchedLineDelimiter", reporter.id);
		assertEquals(2, reporter.params.length);
		assertEquals("6", reporter.params[0]);
		assertEquals("3", reporter.params[1]);

		validateTarget("test\ntext", "テスト\nテキスト");
		assertNotSame("UnmatchedLineDelimiter", reporter.id);

		//-----------------------------------------------------------
		// 用語チェック

		validateTarget("", "ABC ビーン");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("TermValidation", reporter.id);
		assertEquals(1, reporter.params.length);
		assertEquals("ビーン", reporter.params[0]);

		validateTarget("", "ABC カンマ");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("TermValidation2", reporter.id);
		assertEquals(2, reporter.params.length);
		assertEquals("カンマ", reporter.params[0]);
		assertEquals("コンマ", reporter.params[1]);

		validateTarget("", "ABC コンマ");
		assertNotSame("TermValidation", reporter.id);
		assertNotSame("TermValidation2", reporter.id);

		//-----------------------------------------------------------
		// 対訳正規表現チェック

		validateTarget("super class", "スーパークラス");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("TranslationValidation", reporter.id);
		assertEquals(2, reporter.params.length);

		validateTarget("super-class", "スーパークラス");
		assertNotSame("TranslationValidation", reporter.id);

		validateTarget("super class", "スーパー・クラス");
		assertNotSame("TranslationValidation", reporter.id);

		//-----------------------------------------------------------

		validateTarget("push", "押します");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("TranslationValidation2", reporter.id);
		assertEquals(2, reporter.params.length);

		validateTarget("press", "押します");
		assertNotSame("TranslationValidation2", reporter.id);

		//-----------------------------------------------------------
		// 翻訳禁止チェック

		validateTarget("null", "ヌル");
		assertEquals(IMessage.NORMAL_SEVERITY, reporter.serverity);
		assertEquals("ForbiddenTranslation", reporter.id);
		assertEquals(1, reporter.params.length);
		assertEquals("null", reporter.params[0]);

		validateTarget("null", "null");
		assertNotSame("ForbiddenTranslation", reporter.id);
	}
}
