/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.dom;

import java.util.List;

import junit.framework.TestCase;

import org.eclipse.wst.xml.core.internal.document.DOMModelImpl;
import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.w3c.dom.Element;

@SuppressWarnings("restriction")
public class TransUnitDelegateTest extends TestCase {

	private IDOMDocument doc;
	private TransUnitDelegate delegate;

	@Override
	protected void setUp() throws Exception {
		final DOMModelImpl model = new DOMModelImpl();
		doc = model.getDocument();

		final Element xliff = doc.createElement("xliff");
		doc.appendChild(xliff);
		final Element file = doc.createElement("file");
		xliff.appendChild(file);
		final Element body = doc.createElement("body");
		file.appendChild(body);

		final Element transUnit = createTransUnit(body, "benten://test?seq=1-1");
		createTransUnit(body, "benten://test?seq=1-END");

		delegate = new TransUnitDelegate(doc, transUnit);
	}

	private Element createTransUnit(final Element body, final String id) {
		final Element transUnit = doc.createElement("trans-unit");
		transUnit.setAttribute("id", id);
		body.appendChild(transUnit);
		body.appendChild(doc.createTextNode(""));

		final Element source = doc.createElement("source");
		source.appendChild(doc.createTextNode("source text"));
		transUnit.appendChild(source);
		transUnit.appendChild(doc.createTextNode(""));

		createAltTrans(transUnit, "source text 1.");
		createAltTrans(transUnit, "source text 2.");
		return transUnit;
	}

	private void createAltTrans(final Element transUnit, final String sourceText) {
		final Element altTrans = doc.createElement("alt-trans");
		transUnit.appendChild(altTrans);
		transUnit.appendChild(doc.createTextNode(""));

		final Element altTransSource = doc.createElement("source");
		altTransSource.appendChild(doc.createTextNode(sourceText));
		altTrans.appendChild(altTransSource);
		altTrans.appendChild(doc.createTextNode(""));

		final Element altTransTarget = doc.createElement("target");
		altTransTarget.appendChild(doc.createTextNode(""));
		altTrans.appendChild(altTransTarget);
		altTrans.appendChild(doc.createTextNode(""));
	}

	//-------------------------------------------------------------------------

	public void testGetId() {
		assertEquals("benten://test?seq=1-1", delegate.getId());
	}

	public void testIsContextGourpTmOmit() {
		assertEquals(false, delegate.isContextGourpTmOmit());
		delegate.setContextGroupTmOmit(true);
		assertEquals(true, delegate.isContextGourpTmOmit());
		delegate.setContextGroupTmOmit(false);
		assertEquals(false, delegate.isContextGourpTmOmit());
	}

	public void testIsTranslate() {
		assertEquals(true, delegate.isTranslate());
		delegate.setTranslate(false);
		assertEquals(false, delegate.isTranslate());
		delegate.setTranslate(true);
		assertEquals(true, delegate.isTranslate());
	}

	public void testGetSource() {
		assertEquals("source text", delegate.getSource());
		delegate.setSource("test");
		assertEquals("test", delegate.getSource());
		delegate.setSource("");
		assertEquals("", delegate.getSource());
		delegate.setSource(null);
		assertEquals("", delegate.getSource());
	}

	public void testGetTarget() {
		try {
			delegate.setTarget(null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		assertEquals("", delegate.getTarget());
		delegate.setTarget("test");
		assertEquals("test", delegate.getTarget());
		delegate.setTarget("");
		assertEquals("", delegate.getTarget());
	}

	public void testGetTargetState() {
		try {
			delegate.setTargetState(null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		assertEquals("", delegate.getTargetState());
		delegate.setTargetState("test");
		assertEquals("test", delegate.getTargetState());
		delegate.setTargetState("");
		assertEquals("", delegate.getTargetState());
	}

	public void testGetNoteList() {
		assertEquals(0, delegate.getNoteList().size());

		delegate.appendNote("from", "text");
		List<NoteDelegate> noteList = delegate.getNoteList();
		assertEquals(1, noteList.size());
		NoteDelegate note = noteList.get(0);
		assertEquals("from", note.getFrom());
		assertEquals("text", note.getValue());

		delegate.removeNote(note);
		noteList = delegate.getNoteList();
		assertEquals(0, noteList.size());

		delegate.appendNote(null, null);
		noteList = delegate.getNoteList();
		assertEquals(1, noteList.size());
		note = noteList.get(0);
		assertEquals("", note.getFrom());
		assertEquals("", note.getValue());

		delegate.appendNote("", "");
		noteList = delegate.getNoteList();
		assertEquals(2, noteList.size());
		note = noteList.get(0);
		assertEquals("", note.getFrom());
		assertEquals("", note.getValue());
	}

	public void testGetAltTransList() {
		final List<AltTransDelegate> altTransList = delegate.getAltTransList();
		assertEquals(2, altTransList.size());

		AltTransDelegate altTrans = altTransList.get(0);
		assertEquals("", altTrans.getMatchQuality());
		assertEquals("", altTrans.getOrigin());
		assertEquals("source text 1.", altTrans.getSource());
		assertEquals("", altTrans.getTarget());

		altTrans = altTransList.get(1);
		assertEquals("", altTrans.getMatchQuality());
		assertEquals("", altTrans.getOrigin());
		assertEquals("source text 2.", altTrans.getSource());
		assertEquals("", altTrans.getTarget());
	}

	public void testJoinNextTransUnit() {
		List<TransUnitDelegate> transUnitList = TransUnitDelegate.listOf(doc);
		assertEquals(2, transUnitList.size());
		delegate.appendNote("from", "text");
		delegate.appendNote("from", "text");

		assertEquals("benten://test?seq=1-1", delegate.getId());
		assertEquals("source text", delegate.getSource());
		assertEquals("", delegate.getTarget());
		assertEquals(2, delegate.getAltTransList().size());
		assertEquals(2, delegate.getNoteList().size());

		delegate.joinNextTransUnit();
		transUnitList = TransUnitDelegate.listOf(doc);
		assertEquals(1, transUnitList.size());

		assertEquals("benten://test?seq=1-END", delegate.getId());
		assertEquals("source text source text", delegate.getSource());
		assertEquals("", delegate.getTarget());
		assertEquals(4, delegate.getAltTransList().size());
		assertEquals(2, delegate.getNoteList().size());
	}
}
