/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.dom;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.wst.xml.core.internal.provisional.document.IDOMDocument;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import benten.core.dom.messages.BentenCoreDomMessages;

/**
 * XLIFF の trans-unit 要素。
 *
 * @author KASHIHARA Shinji
 */
@SuppressWarnings("restriction")
public class TransUnitDelegate extends TransBaseDelegate {
	/**
	 * Benten コア、DOM のためのメッセージ。
	 */
	protected static final BentenCoreDomMessages fMsg = new BentenCoreDomMessages();

	/**
	 * コンストラクター。
	 * @param domDocument dom ドキュメント
	 * @param element 要素
	 */
	public TransUnitDelegate(final IDOMDocument domDocument, final Element element) {
		super(domDocument, element);
	}

	/**
	 * id 属性の取得。
	 * @return id 属性
	 */
	public String getId() {
		return getValue("@id"); //$NON-NLS-1$
	}

	/**
	 * context-group 要素の x-omit をセット。
	 * @param omit x-omit の場合は true
	 */
	public void setContextGroupTmOmit(final boolean omit) {
		final NodeList contextGroups = element.getElementsByTagName("context-group"); //$NON-NLS-1$
		if (contextGroups.getLength() > 0) {
			final Node contextGroup = contextGroups.item(0);
			element.removeChild(contextGroup.getNextSibling());
			element.removeChild(contextGroup);
		}
		if (omit) {
			final Element contextGroup = domDocument.createElement("context-group"); //$NON-NLS-1$
			contextGroup.setAttribute("purpose", "x-tm"); //$NON-NLS-1$ //$NON-NLS-2$
			final Element context = domDocument.createElement("context"); //$NON-NLS-1$
			context.setAttribute("context-type", "x-omit"); //$NON-NLS-1$ //$NON-NLS-2$

			contextGroup.appendChild(domDocument.createTextNode("\n          ")); //$NON-NLS-1$
			contextGroup.appendChild(context);
			contextGroup.appendChild(domDocument.createTextNode("\n        ")); //$NON-NLS-1$

			final Node target = getChildElement("target"); //$NON-NLS-1$
			final Node source = getChildElement("source"); //$NON-NLS-1$

			// target、source 要素のどちらも存在しない場合は作成をあきらめます。
			if (target == null && source == null) {
				throw new IllegalArgumentException(fMsg.getTransUnitDelegateMsg001());
			}
			insertAfter(contextGroup, target, source);
		}
	}

	/**
	 * context-group 要素に x-omit が存在するか判定。
	 * @return 存在する場合は true
	 */
	public boolean isContextGourpTmOmit() {
		return "x-omit".equals(getValue("context-group/context/@context-type")); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * traslate 属性のセット。
	 * @param translate true の場合は yes をセット
	 * @return 変更があった場合は true
	 */
	public boolean setTranslate(final boolean translate) {
		final String newTranslate = translate ? "yes" : "no"; //$NON-NLS-1$ //$NON-NLS-2$
		final String oldTranslate = getValue("@translate", "yes"); //$NON-NLS-1$ //$NON-NLS-2$
		if (!oldTranslate.equals(newTranslate)) {
			return setValue("@translate", newTranslate); //$NON-NLS-1$
		}
		return false;
	}

	/**
	 * translate 属性が yes か判定。
	 * @return yes の場合は true
	 */
	public boolean isTranslate() {
		return "yes".equals(getValue("@translate", "yes")); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	}

	/**
	 * target 要素のセット。
	 * @param target target 要素の値
	 * @return 変更があった場合は true
	 */
	public boolean setTarget(final String target) {
		final String old = getValue("target", null); //$NON-NLS-1$
		if (target.equals("") && old == null) { //$NON-NLS-1$
			return false;
		}
		if (!target.equals(old)) {
			createEmptyTarget();
			return setValue("target", target); //$NON-NLS-1$
		}
		return false;
	}

	/**
	 * target 要素の state 属性をセット。
	 * @param state state 属性の値
	 * @return 変更があった場合は true
	 */
	public boolean setTargetState(final String state) {
		final String oldState = getTargetState();
		if (!state.equals(oldState)) {
			createEmptyTarget();
			return setValue("target/@state", state); //$NON-NLS-1$
		}
		return false;
	}

	/**
	 * target 要素の state 属性を取得。
	 * @return target 要素の state 属性の値
	 */
	public String getTargetState() {
		return getValue("target/@state"); //$NON-NLS-1$
	}

	/**
	 * target 要素の state-qualifier 属性を取得。
	 * @return target 要素の state-qualifier 属性の値
	 */
	public String getTargetStateQualifier() {
		return getValue("target/@state-qualifier"); //$NON-NLS-1$
	}

	/**
	 * note 要素の追加。
	 * @param from from 属性の値
	 * @param text note 要素の値
	 * @return 追加した note 要素
	 */
	public Node appendNote(final String from, final String text) {
		final Element noteElement = domDocument.createElement("note"); //$NON-NLS-1$
		final org.w3c.dom.Text textNode = domDocument.createTextNode(text);
		noteElement.appendChild(textNode);
		noteElement.setAttribute("from", from); //$NON-NLS-1$
		final NodeList noteList = element.getElementsByTagName("note"); //$NON-NLS-1$

		Node beforeNode = null;
		if (noteList.getLength() > 0) {
			beforeNode = noteList.item(noteList.getLength() - 1);
		} else {
			beforeNode = getTargetNode();
			if (beforeNode == null) {
				beforeNode = getSourceNode();
			}
		}
		// 既存 note、target、source 要素のいずれも存在しない場合は作成をあきらめます。
		if (beforeNode == null) {
			throw new IllegalArgumentException(fMsg.getTransUnitDelegateMsg002());
		}
		insertAfter(noteElement, beforeNode);
		return noteElement;
	}

	/**
	 * note 要素の削除。
	 * @param note 削除する note 要素
	 */
	public void removeNote(final ElementDelegate note) {
		element.removeChild(note.getElement().getNextSibling());
		element.removeChild(note.getElement());
	}

	/**
	 * note 要素のリストを取得。
	 * @return note 要素のリスト
	 */
	public List<NoteDelegate> getNoteList() {
		return getList(NoteDelegate.class, "note"); //$NON-NLS-1$
	}

	/**
	 * alt-trans 要素のリストを取得。
	 * @return alt-trans 要素のリスト
	 */
	public List<AltTransDelegate> getAltTransList() {
		return getList(AltTransDelegate.class, "alt-trans"); //$NON-NLS-1$
	}

	/**
	 * この trans-unit を次の trans-unit と結合。
	 */
	public void joinNextTransUnit() {
		final Node nextText = element.getNextSibling();
		final Element nextElement = (Element) nextText.getNextSibling();
		final TransUnitDelegate nextUnit = new TransUnitDelegate(domDocument, nextElement);

		setId(nextUnit.getId());
		setSource(joinSentence(getSource(), nextUnit.getSource()));
		createEmptyTarget();
		setTarget(joinSentence(getTarget(), nextUnit.getTarget()));

		final Element targetNode = (Element) getTargetNode();
		targetNode.removeAttribute("state-qualifier"); //$NON-NLS-1$

		Node beforeNode = targetNode;

		for (final AltTransDelegate altTrans : nextUnit.getAltTransList()) {
			final Node cloneNode = altTrans.getElement().cloneNode(true);
			insertAfter(cloneNode, beforeNode);
			beforeNode = cloneNode;
		}
		for (final NoteDelegate note : nextUnit.getNoteList()) {
			final Node cloneNode = note.getElement().cloneNode(true);
			insertAfter(cloneNode, beforeNode);
			beforeNode = cloneNode;
		}

		element.getParentNode().removeChild(nextText);
		element.getParentNode().removeChild(nextElement);
	}

	/**
	 * 指定された DOM ドキュメントから trans-unit 要素のリストを取得。
	 * @param domDocument DOM ドキュメント
	 * @return trans-unit 要素のリスト
	 */
	public static List<TransUnitDelegate> listOf(final IDOMDocument domDocument) {
		final NodeList nodeList = domDocument.getElementsByTagName("trans-unit"); //$NON-NLS-1$
		final List<TransUnitDelegate> list = new LinkedList<TransUnitDelegate>();
		final int nodeLength = nodeList.getLength();
		for (int i = 0; i < nodeLength; i++) {
			final TransUnitDelegate ele = new TransUnitDelegate(domDocument, (Element) nodeList.item(i));
			list.add(ele);
		}
		return list;
	}

	//-------------------------------------------------------------------------
	// protected

	/**
	 * 空の target 要素作成。
	 */
	protected void createEmptyTarget() {
		Node targetElement = getChildElement("target"); //$NON-NLS-1$
		if (targetElement == null) {
			targetElement = domDocument.createElement("target"); //$NON-NLS-1$
			final Node source = getChildElement("source"); //$NON-NLS-1$

			// source 要素が存在しない場合は作成をあきらめます。
			if (source == null) {
				throw new IllegalArgumentException(fMsg.getTransUnitDelegateMsg003());
			}
			insertAfter(targetElement, source);
		}
	}

	/**
	 * 指定されたノードの後ろにノードを追加。
	 * @param insertNode 追加ノード
	 * @param beforeNodes 追加する前のノード
	 */
	protected void insertAfter(final Node insertNode, final Node... beforeNodes) {
		for (final Node beforeNode : beforeNodes) {
			if (beforeNode != null) {
				final Node afterNode = beforeNode.getNextSibling().getNextSibling();

				// 次のノードが </trans-unit> の場合
				if (afterNode == null) {
					element.insertBefore(domDocument.createTextNode("  "), null); //$NON-NLS-1$
					element.insertBefore(insertNode, null);
					element.insertBefore(domDocument.createTextNode("\n      "), null); //$NON-NLS-1$
				} else {
					element.insertBefore(insertNode, afterNode);
					element.insertBefore(domDocument.createTextNode("\n        "), afterNode); //$NON-NLS-1$
				}
				return;
			}
		}
		throw new IllegalArgumentException("TransUnitDelegate#insertAfter: Non-null beforeNode not found."); //$NON-NLS-1$
	}

	/**
	 * source 要素をセット。
	 * @param source source 要素の値
	 * @return 変更があった場合は true
	 */
	protected boolean setSource(final String source) {
		return setValue("source", source); //$NON-NLS-1$
	}

	/**
	 * id のセット。
	 * @param id id の値
	 * @return 変更があった場合は true
	 */
	protected boolean setId(final String id) {
		return setValue("@id", id); //$NON-NLS-1$
	}

	/**
	 * 文の結合。
	 *
	 * <UL>
	 * <LI>結合結果の間に空白がない場合は、空白が追加されます。
	 * </UL>
	 *
	 * @param s1 文 1
	 * @param s2 文 2
	 * @return 結合した文
	 */
	protected String joinSentence(final String s1, final String s2) {
		if (s1.equals("") || s2.equals("")) { //$NON-NLS-1$ //$NON-NLS-2$
			return s1 + s2;
		}
		if (s1.matches(".*\\s") || s2.matches("\\s.*")) { //$NON-NLS-1$ //$NON-NLS-2$
			return s1 + s2;
		}
		return s1 + " " + s2; //$NON-NLS-1$
	}
}
