/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.cat.core;

import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;
import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;

public class BentenTmUtilTest extends TestCase {

	public void testAdjustForUntrim() {
		try {
			BentenTmUtil.adjustForUntrim(null, null, null);
			fail();
		} catch (final NullPointerException e) {
			assertTrue(true);
		}
		assertEquals("前後に何も無し", "hello", BentenTmUtil.adjustForUntrim("hello", "hello", "hello"));
		assertEquals("前後に何も無し", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello", "こんにちは"));
		assertEquals("後方に半角 1 つ", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello ", "こんにちは "));
		assertEquals("後方に半角 2 つ", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello  ", "こんにちは  "));
		assertEquals("前方に半角 1 つ", "こんにちは", BentenTmUtil.adjustForUntrim("hello", " hello", " こんにちは"));
		assertEquals("前方に半角 2 つ", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "  hello", "  こんにちは"));

		assertEquals("後方に半角 1 つ", "こんにちは ", BentenTmUtil.adjustForUntrim("hello ", "hello", "こんにちは"));
		assertEquals("後方に半角 2 つ", "こんにちは  ", BentenTmUtil.adjustForUntrim("hello  ", "hello", "こんにちは"));
		assertEquals("前方に半角 1 つ", " こんにちは", BentenTmUtil.adjustForUntrim(" hello", "hello", "こんにちは"));
		assertEquals("前方に半角 2 つ", "  こんにちは", BentenTmUtil.adjustForUntrim("  hello", "hello", "こんにちは"));
		assertNotSame("強制トリム", "  こんにちは", BentenTmUtil.adjustForUntrim("  hello", "hello:", "こんにちは"));

		assertEquals("バリエーション", "こんにちは ", BentenTmUtil.adjustForUntrim("hello ", "hello ", "こんにちは "));
		assertEquals("バリエーション", "こんにちは  ", BentenTmUtil.adjustForUntrim("hello  ", "hello ", "こんにちは "));
		assertEquals("バリエーション", " こんにちは", BentenTmUtil.adjustForUntrim(" hello", "hello ", "こんにちは "));
		assertEquals("バリエーション", "  こんにちは", BentenTmUtil.adjustForUntrim("  hello", "hello ", "こんにちは"));

		assertNotSame("トリム", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "(hello)", "(こんにちは)"));
		assertNotSame("トリム", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "--- hello ---", "--- こんにちは ---"));
		assertEquals("トリム", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello&nbsp;", "こんにちは&nbsp;"));

		assertNotSame("強制トリム", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello:", "こんにちは:"));
		assertNotSame("強制トリム", "こんにちは", BentenTmUtil.adjustForUntrim("hello", "hello.", "こんにちは。"));
		assertNotSame("強制トリム", "こんにちは。", BentenTmUtil.adjustForUntrim("hello.", "hello", "こんにちは"));
		assertNotSame("強制トリム", "こんにちは!", BentenTmUtil.adjustForUntrim("hello!", "hello", "こんにちは"));
	}

	public void testSearchExactMatch01() throws Exception {
		try {
			BentenTmUtil.searchExactMatch((BentenTmEngine) null, null);
			fail();
		} catch (final IllegalArgumentException e) {
			assertTrue(true);
		}
		assertEquals("1 件もない", null, BentenTmUtil.searchExactMatch((List<BentenTmSearchResult>) null, null));

		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();
		{
			assertEquals("1 件もない", null, BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello");
			res1.setTarget("こんにちは");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "こんにちは", BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			assertEquals("1 件のみヒット", "こんにちは ", BentenTmUtil.searchExactMatch(matchList, "hello "));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello");
			res1.setTarget("にいはお");
			assertEquals("2 件あたり、それらがおのおの異なる", null, BentenTmUtil.searchExactMatch(matchList, "hello"));
		}
	}

	public void testSearchExactMatch02() throws Exception {
		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();
		{
			assertEquals("1 件もない", null, BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello");
			res1.setTarget("こんにちは");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "こんにちは", BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello ");
			res1.setTarget("こんにちは ");
			assertEquals("2 件あたり、それらが一致する", "こんにちは", BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource(" hello");
			res1.setTarget(" こんにちは");
			assertEquals("3 件あたり、それらが一致する", " こんにちは  ", BentenTmUtil.searchExactMatch(matchList, " hello  "));
		}
	}

	public void testSearchExactMatch03() throws Exception {
		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello");
			res1.setTarget("こんにちは");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "こんにちは", BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello ");
			res1.setTarget("こんにちは ");
			assertEquals("2 件あたり、それらが一致する", "こんにちは", BentenTmUtil.searchExactMatch(matchList, "hello"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource(" hello");
			res1.setTarget(" こんにちは");
			assertEquals("3 件あたり、それらが一致する", " こんにちは  ", BentenTmUtil.searchExactMatch(matchList, " hello  "));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("hello");
			res1.setTarget("さようなら");
			assertEquals("4 件あたり、その 1 つは一致しない", null, BentenTmUtil.searchExactMatch(matchList, " hello  "));
		}
	}

	public void testSearchExactMatch04() throws Exception {
		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("Synchronize.");
			res1.setTarget("同期化します。");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "同期化します。", BentenTmUtil.searchExactMatch(matchList, "Synchronize."));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("Synchronize.");
			res1.setTarget("同期化します。");
			// match-quality は無視!
			assertEquals("2 件ヒット", "同期化します。", BentenTmUtil.searchExactMatch(matchList, "Synchronize."));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("Synchronize");
			res1.setTarget("同期化");
			// match-quality は無視!
			assertEquals("2 件ヒット", "同期化します。", BentenTmUtil.searchExactMatch(matchList, "Synchronize."));
		}
	}

	public void testSearchExactMatch05() throws Exception {
		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("The Workbench menu bar and toolbar will be updated with options for Microsoft Word.)");
			res1.setTarget("ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。", BentenTmUtil
					.searchExactMatch(matchList,
							"The Workbench menu bar and toolbar will be updated with options for Microsoft Word.)"));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("The Workbench menu bar and toolbar will be updated with options for Microsoft Word.) ");
			res1.setTarget("ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。", BentenTmUtil
					.searchExactMatch(matchList,
							"The Workbench menu bar and toolbar will be updated with options for Microsoft Word.)"));
		}
	}

	public void testSearchExactMatch06() throws Exception {
		final List<BentenTmSearchResult> matchList = new ArrayList<BentenTmSearchResult>();

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("The Workbench menu bar and toolbar will be updated with options for Microsoft Word.)");
			res1.setTarget("ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。");
			// match-quality は無視!
			assertEquals("1 件のみヒット", "ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。 ", BentenTmUtil
					.searchExactMatch(matchList,
							"The Workbench menu bar and toolbar will be updated with options for Microsoft Word.) "));
		}

		{
			final BentenTmSearchResult res1 = new BentenTmSearchResult();
			matchList.add(res1);
			res1.setSource("The Workbench menu bar and toolbar will be updated with options for Microsoft Word.) ");
			res1.setTarget("ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。");
			// match-quality は無視!
			// TODO このテストケースが失敗します。このような異常なデータでも適切に動作する必要があります。柏原さん、これを調べてください。
			assertEquals("1 件のみヒット", "ワークベンチのメニューバーとツールバーは、Microsoft Word のオプションを使用して更新されます。", BentenTmUtil
					.searchExactMatch(matchList,
							"The Workbench menu bar and toolbar will be updated with options for Microsoft Word.)"));
		}
	}
}
