/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.cat.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.cat.core.valueobject.BentenApplyExistentTranslationProcessInput;

/**
 * Apache Antタスク [BentenApplyExistentTranslation]のクラス。
 *
 * 過去訳適用のための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.cat.core]にBentenApplyExistentTranslationBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenapplyexistenttranslation&quot; classname=&quot;benten.twa.cat.core.BentenApplyExistentTranslationTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenApplyExistentTranslationTask extends Task {
    /**
     * 過去訳適用のための Ant タスク。
     */
    protected BentenApplyExistentTranslationProcessInput fInput = new BentenApplyExistentTranslationProcessInput();

    /**
     * フィールド [targetdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetdirProcessed = false;

    /**
     * フィールド [sourcedir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcedirProcessed = false;

    /**
     * フィールド [transsourcelang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranssourcelangProcessed = false;

    /**
     * フィールド [transtargetlang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranstargetlangProcessed = false;

    /**
     * フィールド [tmdriverclassname] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTmdriverclassnameProcessed = false;

    /**
     * フィールド [ignorewhitespacetmreference] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnorewhitespacetmreferenceProcessed = false;

    /**
     * フィールド [ignoremnemonickeytmreference] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnoremnemonickeytmreferenceProcessed = false;

    /**
     * フィールド [tmpdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTmpdirProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[targetdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 適用先 XLIFF が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetdir(final String arg) {
        fInput.setTargetdir(arg);
        fIsFieldTargetdirProcessed = true;
    }

    /**
     * Antタスクの[targetdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 適用先 XLIFF が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetdir() {
        return fInput.getTargetdir();
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 適用元(過去訳) XLIFF が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcedir(final String arg) {
        fInput.setSourcedir(arg);
        fIsFieldSourcedirProcessed = true;
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 適用元(過去訳) XLIFF が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcedir() {
        return fInput.getSourcedir();
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳元言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranssourcelang(final String arg) {
        fInput.setTranssourcelang(arg);
        fIsFieldTranssourcelangProcessed = true;
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳元言語<br>
     * デフォルト値[en-US]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranssourcelang() {
        return fInput.getTranssourcelang();
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳先言語<br>
     *
     * @param arg セットしたい値
     */
    public void setTranstargetlang(final String arg) {
        fInput.setTranstargetlang(arg);
        fIsFieldTranstargetlangProcessed = true;
    }

    /**
     * Antタスクの[transtargetlang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * 翻訳先言語<br>
     * デフォルト値[ja-JP]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranstargetlang() {
        return fInput.getTranstargetlang();
    }

    /**
     * Antタスクの[tmdriverclassname]アトリビュートのセッターメソッド。
     *
     * 項目番号: 5<br>
     * Ant タスクの場合には、TM ドライバーのクラス名を指定します。<br>
     *
     * @param arg セットしたい値
     */
    public void setTmdriverclassname(final String arg) {
        fInput.setTmdriverclassname(arg);
        fIsFieldTmdriverclassnameProcessed = true;
    }

    /**
     * Antタスクの[tmdriverclassname]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 5<br>
     * Ant タスクの場合には、TM ドライバーのクラス名を指定します。<br>
     *
     * @return このフィールドの値
     */
    public String getTmdriverclassname() {
        return fInput.getTmdriverclassname();
    }

    /**
     * Antタスクの[ignorewhitespacetmreference]アトリビュートのセッターメソッド。
     *
     * 項目番号: 6<br>
     * TMX を検索する際にホワイトスペースを無視するかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnorewhitespacetmreference(final boolean arg) {
        fInput.setIgnorewhitespacetmreference(arg);
        fIsFieldIgnorewhitespacetmreferenceProcessed = true;
    }

    /**
     * Antタスクの[ignorewhitespacetmreference]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 6<br>
     * TMX を検索する際にホワイトスペースを無視するかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnorewhitespacetmreference() {
        return fInput.getIgnorewhitespacetmreference();
    }

    /**
     * Antタスクの[ignoremnemonickeytmreference]アトリビュートのセッターメソッド。
     *
     * 項目番号: 7<br>
     * TMX を検索する際にニーモニックを無視するかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnoremnemonickeytmreference(final boolean arg) {
        fInput.setIgnoremnemonickeytmreference(arg);
        fIsFieldIgnoremnemonickeytmreferenceProcessed = true;
    }

    /**
     * Antタスクの[ignoremnemonickeytmreference]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 7<br>
     * TMX を検索する際にニーモニックを無視するかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnoremnemonickeytmreference() {
        return fInput.getIgnoremnemonickeytmreference();
    }

    /**
     * Antタスクの[tmpdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 8<br>
     * テンポラリ・ディレクトリ<br>
     *
     * @param arg セットしたい値
     */
    public void setTmpdir(final String arg) {
        fInput.setTmpdir(arg);
        fIsFieldTmpdirProcessed = true;
    }

    /**
     * Antタスクの[tmpdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 8<br>
     * テンポラリ・ディレクトリ<br>
     * デフォルト値[&quot;./tmp&quot;]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTmpdir() {
        return fInput.getTmpdir();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenApplyExistentTranslationTask begin.");

        // 項目番号[1]、アトリビュート[targetdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetdir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[sourcedir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcedirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcedir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- targetdir:[" + getTargetdir() + "]");
            System.out.println("- sourcedir:[" + getSourcedir() + "]");
            System.out.println("- transsourcelang:[" + getTranssourcelang() + "]");
            System.out.println("- transtargetlang:[" + getTranstargetlang() + "]");
            System.out.println("- tmdriverclassname:[" + getTmdriverclassname() + "]");
            System.out.println("- ignorewhitespacetmreference:[" + getIgnorewhitespacetmreference() + "]");
            System.out.println("- ignoremnemonickeytmreference:[" + getIgnoremnemonickeytmreference() + "]");
            System.out.println("- tmpdir:[" + getTmpdir() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenApplyExistentTranslationProcessインタフェースを実装して benten.twa.cat.coreパッケージに BentenApplyExistentTranslationProcessImplクラスを作成することにより解決できる場合があります。
            final BentenApplyExistentTranslationProcess proc = new BentenApplyExistentTranslationProcessImpl();
            if (proc.execute(fInput) != BentenApplyExistentTranslationBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
