/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.converter.tmx;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.converter.tmx.valueobject.BentenConverterTmxAdjusterProcessInput;

/**
 * Apache Antタスク [BentenConverterTmxAdjuster]のクラス。
 *
 * TMX を Benten 形式に変換するための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.converter.tmx]にBentenConverterTmxAdjusterBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenconvertertmxadjuster&quot; classname=&quot;benten.twa.converter.tmx.BentenConverterTmxAdjusterTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenConverterTmxAdjusterTask extends Task {
    /**
     * TMX を Benten 形式に変換するための Ant タスク。
     */
    protected BentenConverterTmxAdjusterProcessInput fInput = new BentenConverterTmxAdjusterProcessInput();

    /**
     * フィールド [sourcedir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldSourcedirProcessed = false;

    /**
     * フィールド [targetdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTargetdirProcessed = false;

    /**
     * フィールド [ignorewhitespacetmxconvert] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnorewhitespacetmxconvertProcessed = false;

    /**
     * フィールド [convertcharnbsp] に値がセットされたかどうか。
     */
    protected boolean fIsFieldConvertcharnbspProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 入力となる TMX が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setSourcedir(final String arg) {
        fInput.setSourcedir(arg);
        fIsFieldSourcedirProcessed = true;
    }

    /**
     * Antタスクの[sourcedir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 入力となる TMX が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getSourcedir() {
        return fInput.getSourcedir();
    }

    /**
     * Antタスクの[targetdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 出力となる TMX が格納されているディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTargetdir(final String arg) {
        fInput.setTargetdir(arg);
        fIsFieldTargetdirProcessed = true;
    }

    /**
     * Antタスクの[targetdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 出力となる TMX が格納されているディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTargetdir() {
        return fInput.getTargetdir();
    }

    /**
     * Antタスクの[ignorewhitespacetmxconvert]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * ホワイトスペースを無視した内容の TMX を抽出するかどうか。HTML の場合には true を推奨。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnorewhitespacetmxconvert(final boolean arg) {
        fInput.setIgnorewhitespacetmxconvert(arg);
        fIsFieldIgnorewhitespacetmxconvertProcessed = true;
    }

    /**
     * Antタスクの[ignorewhitespacetmxconvert]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * ホワイトスペースを無視した内容の TMX を抽出するかどうか。HTML の場合には true を推奨。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnorewhitespacetmxconvert() {
        return fInput.getIgnorewhitespacetmxconvert();
    }

    /**
     * Antタスクの[convertcharnbsp]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * Unicode の a0 を &amp;nbsp; に変換するかどうか。HTML の場合には true を推奨。<br>
     *
     * @param arg セットしたい値
     */
    public void setConvertcharnbsp(final boolean arg) {
        fInput.setConvertcharnbsp(arg);
        fIsFieldConvertcharnbspProcessed = true;
    }

    /**
     * Antタスクの[convertcharnbsp]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * Unicode の a0 を &amp;nbsp; に変換するかどうか。HTML の場合には true を推奨。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getConvertcharnbsp() {
        return fInput.getConvertcharnbsp();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenConverterTmxAdjusterTask begin.");

        // 項目番号[1]、アトリビュート[sourcedir]は必須入力です。入力チェックを行います。
        if (fIsFieldSourcedirProcessed == false) {
            throw new BuildException("必須アトリビュート[sourcedir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[targetdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTargetdirProcessed == false) {
            throw new BuildException("必須アトリビュート[targetdir]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- sourcedir:[" + getSourcedir() + "]");
            System.out.println("- targetdir:[" + getTargetdir() + "]");
            System.out.println("- ignorewhitespacetmxconvert:[" + getIgnorewhitespacetmxconvert() + "]");
            System.out.println("- convertcharnbsp:[" + getConvertcharnbsp() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenConverterTmxAdjusterProcessインタフェースを実装して benten.twa.converter.tmxパッケージに BentenConverterTmxAdjusterProcessImplクラスを作成することにより解決できる場合があります。
            final BentenConverterTmxAdjusterProcess proc = new BentenConverterTmxAdjusterProcessImpl();
            if (proc.execute(fInput) != BentenConverterTmxAdjusterBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
