/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.converter.tmx;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.StringWriter;

import benten.core.text.Strings;
import benten.twa.converter.tmx.valueobject.BentenConverterTmxAdjusterProcessInput;
import benten.twa.io.AbstractTraverseDir;
import blanco.commons.util.BlancoStringUtil;
import blanco.tmx.BlancoTmxParser;
import blanco.tmx.BlancoTmxSerializer;
import blanco.tmx.valueobject.BlancoTmx;
import blanco.tmx.valueobject.BlancoTmxTu;
import blanco.tmx.valueobject.BlancoTmxTuv;
import blanco.xml.bind.BlancoXmlMarshaller;
import blanco.xml.bind.BlancoXmlUnmarshaller;
import blanco.xml.bind.valueobject.BlancoXmlAttribute;
import blanco.xml.bind.valueobject.BlancoXmlCharacters;
import blanco.xml.bind.valueobject.BlancoXmlDocument;
import blanco.xml.bind.valueobject.BlancoXmlElement;
import blanco.xml.bind.valueobject.BlancoXmlNode;

/**
 * Benten 以外のシステムが出力した TMX を Benten で扱うことができるようにするためのコンバーターです。
 * 
 * @author IGA Tosiki
 */
public class BentenConverterTmxAdjusterProcessImpl extends AbstractTraverseDir implements
		BentenConverterTmxAdjusterProcess {
	BentenConverterTmxAdjusterProcessInput fInput = null;

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenConverterTmxAdjusterProcessInput input) throws IOException, IllegalArgumentException {
		fInput = input;
		processDir(new File(input.getSourcedir()));
		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		System.out.println(argProgressMessage);
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		return file.getName().toLowerCase().endsWith(".tmx"); //$NON-NLS-1$
	}

	@Override
	protected void processFile(final File file, final String baseDir) throws IOException {
		progress("Processing file [" + file.toString() + "] ..."); //$NON-NLS-1$ //$NON-NLS-2$

		final File output = new File(fInput.getTargetdir() + "/" + baseDir + "/" + file.getName()); //$NON-NLS-1$ //$NON-NLS-2$

		{
			progress("phase1: convert character encoding."); //$NON-NLS-1$
			final BlancoXmlUnmarshaller unmarshaller = new BlancoXmlUnmarshaller();
			final BlancoXmlDocument xmlDocument = unmarshaller.unmarshal(new ByteArrayInputStream(adjustUnicode(file)));
			for (BlancoXmlNode node : xmlDocument.getChildNodes()) {
				if (node instanceof BlancoXmlElement) {
					parseElement((BlancoXmlElement) node, true);
				}
			}

			System.out.println("Output target [" + output.getAbsolutePath() + "]"); //$NON-NLS-1$ //$NON-NLS-2$

			if (output.getParentFile().exists() == false) {
				output.getParentFile().mkdirs();
			}
			new BlancoXmlMarshaller().marshal(xmlDocument, output);
		}

		{
			progress("phase2: convert XML format."); //$NON-NLS-1$
			// 読んで書く。これにより blanco 形式の tmx となる。
			final BlancoTmx tmx = new BlancoTmxParser().parse(output);
			if (tmx.getVersion() != null && tmx.getVersion().startsWith("1.4") == false) { //$NON-NLS-1$
				System.out.println("File [" + file.getName() + "], TMX version (" + tmx.getVersion() //$NON-NLS-1$ //$NON-NLS-2$
						+ ") is NOT supported. Only (1.4) can be processed."); //$NON-NLS-1$
			}
			for (BlancoTmxTu tu : tmx.getBody().getTuList()) {
				for (BlancoTmxTuv tuv : tu.getTuvList()) {
					// Benten は文字コードとしての nbsp を処理しません。
					if (fInput.getConvertcharnbsp() == false) {
						// 半角空白に置き換えます。
						tuv.setSeg(BlancoStringUtil.replaceAll(tuv.getSeg(), (char) 0xa0, ' '));
					} else {
						// HTML として扱い &nbsp; に置き換えます。
						tuv.setSeg(BlancoStringUtil.replaceAll(tuv.getSeg(), String.valueOf((char) 0xa0), "&nbsp;")); //$NON-NLS-1$
					}

					if (fInput.getIgnorewhitespacetmxconvert()) {
						// ホワイトスペースを無視した内容の TMX へと変換する場合。
						tuv.setSeg(Strings.removeRedundantWhitespace(tuv.getSeg()));
					}
				}
			}
			new BlancoTmxSerializer().serialize(tmx, output);
		}
	}

	int fPt = 0;
	boolean fIsInPtTypeFont = false;

	void parseElement(final BlancoXmlElement eleParent, boolean isEscape) {
		// パス 1 : タグに応じて参照文字へのエンコードを実施します。
		for (BlancoXmlNode child : eleParent.getChildNodes()) {
			if (child instanceof BlancoXmlElement) {
				boolean fBpt = false;
				boolean fEpt = false;
				boolean fIt = false;
				boolean fPh = false;

				BlancoXmlElement ele = (BlancoXmlElement) child;

				if (ele.getQName().equals("bpt")) { //$NON-NLS-1$
					fPt++;
					fBpt = true;
				}
				if (ele.getQName().equals("ept")) { //$NON-NLS-1$
					fPt--;
					fEpt = true;
				}
				if (ele.getQName().equals("it")) { //$NON-NLS-1$
					fIt = true;
				}
				if (ele.getQName().equals("ph")) { //$NON-NLS-1$
					fPh = true;
				}

				parseElement(ele, (fBpt || fEpt || fIt || fPh) == false);

				if (ele.getQName().equals("bpt")) { //$NON-NLS-1$
					fBpt = false;
				}
				if (ele.getQName().equals("ept")) { //$NON-NLS-1$
					fEpt = false;
				}
				if (eleParent.getQName().equals("it")) { //$NON-NLS-1$
					fIt = false;
				}
				if (eleParent.getQName().equals("ph")) { //$NON-NLS-1$
					fPh = false;
				}

			} else if (child instanceof BlancoXmlCharacters) {
				final BlancoXmlCharacters characters = (BlancoXmlCharacters) child;

				if (isEscape == false) {
					// エンコード無し
					// System.out.println("エンコード無: " + characters.getValue());
				} else {
					// エンコード実施
					// System.out.println("エンコード有: " + characters.getValue());
					characters.setValue(escapeStringAsJavaDoc(characters.getValue()));
				}
			}
		}

		// パス 2 : 所定のタグを除去します。
		for (int index = 0; index < eleParent.getChildNodes().size(); index++) {
			final BlancoXmlNode child = eleParent.getChildNodes().get(index);
			if (child instanceof BlancoXmlElement) {

				BlancoXmlElement ele = (BlancoXmlElement) child;
				boolean isRemove = false;

				if (ele.getQName().equals("bpt")) { //$NON-NLS-1$
					isRemove = true;
					for (BlancoXmlAttribute attr : ele.getAtts()) {
						if (attr.getQName().equals("type")) { //$NON-NLS-1$
							if (attr.getValue().equals("font")) { //$NON-NLS-1$
								fIsInPtTypeFont = true;
							}
						}
					}
				}
				boolean isInPtFont = fIsInPtTypeFont;

				if (ele.getQName().equals("ept")) { //$NON-NLS-1$
					isRemove = true;
					fIsInPtTypeFont = false;
				}
				if (ele.getQName().equals("it")) { //$NON-NLS-1$
					isRemove = true;
				}
				if (ele.getQName().equals("ph")) { //$NON-NLS-1$
					isRemove = true;
				}

				if (isRemove) {
					// System.out.println("エレメント除去: " + ele.getQName());
					eleParent.getChildNodes().remove(index);

					if ((ele.getQName().equals("bpt") || ele.getQName().equals("ept")) && isInPtFont) { //$NON-NLS-1$ //$NON-NLS-2$
						// 何もしません。
					} else {
						int indexSub = index;
						for (BlancoXmlNode node : ele.getChildNodes()) {
							// コピー
							eleParent.getChildNodes().add(indexSub++, node);
						}
					}

					// 削除して次に進むと 1 つ飛ばしてしまうので、1 つ戻します。
					index--;
				}
			}
		}
	}

	private static final String escapeStringAsJavaDoc(final String originalString) {
		if (originalString == null) {
			throw new IllegalArgumentException(
					"IO Exception occurred in BlancoJavaSourceUtil.escapeStringAsJavaDoc. Null argument has given in parameter. Please set non null value."); //$NON-NLS-1$
		}

		final StringReader reader = new StringReader(originalString);
		final StringWriter writer = new StringWriter();
		try {
			for (;;) {
				final int iRead = reader.read();
				if (iRead < 0) {
					break;
				}
				switch (iRead) {
				case '&':
					writer.write("&amp;"); //$NON-NLS-1$
					break;
				case '<':
					writer.write("&lt;"); //$NON-NLS-1$
					break;
				case '>':
					writer.write("&gt;"); //$NON-NLS-1$
					break;
				case '"':
					writer.write("&quot;"); //$NON-NLS-1$
					break;
				default:
					writer.write((char) iRead);
					break;
				}
			}
			writer.flush();
		} catch (IOException e) {
			// ここに入ってくることは、ありえません。
			e.printStackTrace();
		}
		return writer.toString();
	}

	private byte[] adjustUnicode(final File file) throws IOException {
		final ByteArrayOutputStream bufStream = new ByteArrayOutputStream();
		final BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "Unicode")); //$NON-NLS-1$
		final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(bufStream, "Unicode")); //$NON-NLS-1$
		for (int index = 0;; index++) {
			int read = reader.read();
			if (read < 0) {
				break;
			}
			if (index > 10) {
				switch (read) {
				case 0x00: // NUL: これはまずいです。
				case 0x06: // ACK: ???
				case 0x10: // DLE: ???
				case 0x13: // DC3: ???
				case 0x1c: // FS : ???
					// パスします。
					continue;
				}
			}
			writer.write((char) read);
		}
		reader.close();
		writer.close();
		return bufStream.toByteArray();
	}
}
