package benten.twa.filter.engine.properties;

import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Properties;

import benten.twa.filter.engine.properties.AbstractJavaPropertiesFileProcessor;

import junit.framework.TestCase;

public class BentenTwaFilterJavaPropertyCommonTest extends TestCase {

	public void testCommon001() throws Exception {
		final String source = "# comment\nabc=def\nghi=jkl";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// キーの値を処理しない場合。
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("値を変更をしない場合には、プロパティファイルの処理前と処理後が一致すること。", source, writer.toString());
	}

	public void testCommon002() throws Exception {
		final String source = "# comment\nabc=def\nghi=jkl";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				writer.write(getEscapedPropertiesValue("変更"));
				// キーの値を処理した場合。
				return true;
			}
		}.process(sourceBytes, writer);
		assertEquals("値を変更した場合、プロパティファイルの内容が変わっていること。", "# comment\nabc=\\u5909\\u66F4\nghi=\\u5909\\u66F4", writer
				.toString());
	}

	public void testCommon003a() throws Exception {
		final String source = "abc=def\\\n ghi";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("￥マーク＋改行の場合であっても、処理しない場合にはもとのままであること。", "abc=def\\\n ghi", writer.toString());
	}

	public void testCommon003b() throws Exception {
		final String source = "abc=def\\\n ghi";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				writer.write("defghi");
				return true;
			}
		}.process(sourceBytes, writer);
		assertEquals("￥マーク＋改行の場合、その箇所が除去されること。", "abc=defghi", writer.toString());
	}

	public void testCommon004() throws Exception {
		final String source = "abc:def";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[1];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("コロンが＝の代わりになること。", "abc:def", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "abc", result[0]);
	}

	public void testCommon005() throws Exception {
		final String source = "   abc=def";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[2];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				result[1] = getEscapedPropertiesValue(props.getProperty(key));
				writer.write(result[1]);
				return true;
			}
		}.process(sourceBytes, writer);
		assertEquals("先頭に空白があったとしてもプロパティのキーとして認識されること。", "   abc=def", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "abc", result[0]);
		assertEquals("値が期待通りであること。", "def", result[1]);
	}

	public void testCommon006() throws Exception {
		final String source = "ab\\ c=def";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[2];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				result[1] = getEscapedPropertiesValue(props.getProperty(key));
				//writer.write(result[1]);
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("キーの名称に空白が含まれても破綻しないこと。", "ab\\ c=def", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "ab c", result[0]);
		assertEquals("値が期待通りであること。", "def", result[1]);
	}

	public void testCommon007a() throws Exception {
		final String source = "abc=de\\ f";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[2];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				result[1] = getEscapedPropertiesValue(props.getProperty(key));
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("キーの値にエスケープ付きで空白が含まれても破綻しないこと。", "abc=de\\ f", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "abc", result[0]);
		assertEquals("値が期待通りであること。", "de f", result[1]);
	}

	public void testCommon007b() throws Exception {
		final String source = "abc=de\\ f";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[2];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				result[1] = getEscapedPropertiesValue(props.getProperty(key));
				writer.write("de f");
				return true;
			}
		}.process(sourceBytes, writer);
		assertEquals("キーの値にエスケープ付きで空白が含まれても破綻しないこと。", "abc=de f", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "abc", result[0]);
		assertEquals("値が期待通りであること。", "de f", result[1]);
	}

	public void testCommon008() throws Exception {
		final String source = "abc=de\\nf";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		final String[] result = new String[2];
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				// プロパティファイルの値として妥当であるように加工します。
				// キーの値を処理しない場合。
				result[0] = key;
				result[1] = getEscapedPropertiesValue(props.getProperty(key));
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("キーの値に改行が含まれても破綻しないこと。", "abc=de\\nf", writer.toString());
		assertEquals("キーの名称が期待通りであること。", "abc", result[0]);
		assertEquals("値が期待通りであること。", "de\\nf", result[1]);
	}

	public void testCommon009() throws Exception {
		final String source = "abc=def\r\nghi=jkl\r\n";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("入力の改行が維持されること。", source, writer.toString());
	}

	public void testCommon010() throws Exception {
		final String source = "abc=def\rghi=jkl\r";
		final byte[] sourceBytes = source.getBytes("UTF-8");
		final StringWriter writer = new StringWriter();
		new AbstractJavaPropertiesFileProcessor() {
			@Override
			boolean replaceValue(Properties props, String key, Writer writer) throws IOException {
				return false;
			}
		}.process(sourceBytes, writer);
		assertEquals("入力の改行が維持されること。", source, writer.toString());
	}
}
