/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.core;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import benten.core.io.Files;
import benten.twa.filter.core.valueobject.BentenImportHtmlProcessInput;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376DocxEngine;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376PptxEngine;
import benten.twa.filter.engine.ecma376.BentenTwaFilterEcma376XlsxEngine;
import benten.twa.filter.engine.ecma376.Ecma376Normalizer;
import benten.twa.filter.engine.html.BentenTwaFilterHtmlEngine;
import benten.twa.filter.messages.BentenImportHtmlMessages;
import benten.twa.io.AbstractTraverseDir;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import blanco.html.normalizer.util.BlancoHtmlNormalizerUtil;

/**
 * 翻訳対象のインポート
 *
 * <pre>
 * 翻訳対象物をプロジェクトにインポートします。
 *   1.  翻訳対象物のファイルをプロジェクトにインポートします。
 *   2.  インポートされた翻訳対象物は、デフォルトではプロジェクト内の [.benten/source] ディレクトリ以下に配置されます。
 *   3.  インポートされた翻訳対象物は、エクスポートの際に利用されます。
 *   4.  インポートされた翻訳対象物は、直接編集しないでください。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTMLインポート機能」に対応します。
 *
 * @author KASHIHARA Shinji
 * @author IGA Tosiki
 */
public class BentenImportHtmlProcessImpl extends AbstractTraverseDir implements BentenImportHtmlProcess {
	/**
	 * HTMLインポート機能のためのメッセージ。
	 */
	protected static final BentenImportHtmlMessages fMsg = new BentenImportHtmlMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenImportHtmlProcessInput fInput;

	/**
	 * 処理の入力オブジェクトを設定。
	 * @param input 処理の入力オブジェクト。
	 */
	public void setInput(final BentenImportHtmlProcessInput input) {
		fInput = input;
	}

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * 処理クラスのリスト。
	 */
	protected List<BentenTwaFilterEngine> fTwaEngineList = new ArrayList<BentenTwaFilterEngine>();

	/**
	 * コンストラクタ。
	 */
	public BentenImportHtmlProcessImpl() {
		// ファイル正常化が必要なものの一覧。
		fTwaEngineList.add(new BentenTwaFilterHtmlEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376DocxEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376XlsxEngine());
		fTwaEngineList.add(new BentenTwaFilterEcma376PptxEngine());
	}

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenImportHtmlProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException("BentenImportHtmlProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File dirSource = new File(fInput.getSourcedir());
		if (dirSource.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE004(fInput.getSourcedir()));
		}
		if (dirSource.isDirectory() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE005(fInput.getSourcedir()));
		}

		// トータル件数カウント。
		class FileCounter extends BentenImportHtmlProcessImpl {
			private int fCounter = 0;

			@Override
			public void processFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			@Override
			protected void processFilterdFile(final File file, final String baseDir) throws IOException {
				fCounter++;
			}

			/**
			 * カウンタ数の取得。
			 * @return カウンタ数。
			 */
			public int getCounter() {
				return fCounter;
			}
		}
		final FileCounter inner = new FileCounter();
		inner.setInput(input);
		inner.processDir(dirSource);
		beginTask(inner.getCounter());

		if (progress(fMsg.getCoreP002())) {
			return 6;
		}

		processDir(dirSource);

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	@Override
	protected boolean canProcess(final File file) {
		// 処理できるファイルかどうかを判定。
		for (BentenTwaFilterEngine processor : fTwaEngineList) {
			if (processor.canProcess(file)) {
				return true;
			}
		}
		return false;
	}

	@Override
	public void processFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenImportHtmlProcessImpl#processFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP011(file.getName()))) {
			return;
		}

		byte[] bytes = Files.readFileToByteArray(file);
		// TODO いつの日か、以下の if 文を canProcess に集約のうえ別クラスに分離してください。
		if (file.getName().toLowerCase().endsWith(".html") || file.getName().toLowerCase().endsWith(".htm")) { //$NON-NLS-1$ //$NON-NLS-2$
			if (fInput.getNormalizehtml()) {
				// HTML の正常化を実施。
				bytes = BlancoHtmlNormalizerUtil.normalize(bytes);
			}
		}
		// TODO いつの日か、以下の if 文を canProcess に集約のうえ別クラスに分離してください。
		if (file.getName().toLowerCase().endsWith(".docx") || file.getName().toLowerCase().endsWith(".xlsx") || file.getName().toLowerCase().endsWith(".pptx")) { //$NON-NLS-1$ //$NON-NLS-2$
			// WORD 文書ファイルおよび PowerPoint ファイルについては、有無を言わさず正常化を実施します。
			bytes = new Ecma376Normalizer().normalize(file.getName(), bytes);
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcedir()), file);
		final File outputHtmlFile = new File(fInput.getTargetdir(), relativePath);

		// 該当ディレクトリーが無い場合には、ディレクトリー作成。
		// すでに存在する場合は、それはそれで良い。
		if (outputHtmlFile.getParentFile().exists() == false) {
			if (outputHtmlFile.getParentFile().mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE006(outputHtmlFile.getParentFile().getAbsolutePath()));
			}
		} else {
			if (outputHtmlFile.getParentFile().isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE007(outputHtmlFile.getParentFile().getAbsolutePath()));
			}
		}

		Files.writeByteArrayToFile(outputHtmlFile, bytes);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}

	@Override
	protected void processFilterdFile(final File file, final String baseDir) throws IOException {
		if (fInput == null) {
			throw new IllegalArgumentException(
					"BentenImportHtmlProcessImpl#processFilterdFile: 'fInput' is null. Call execute or setInput before calling this method."); //$NON-NLS-1$
		}

		if (progress(fMsg.getCoreP012(file.getName()))) {
			return;
		}

		final String relativePath = Files.relativePath(new File(fInput.getSourcedir()), file);
		final File outputFile = new File(fInput.getTargetdir(), relativePath);
		Files.copyFile(file, outputFile);

		getResultInfo().setSuccessCount(getResultInfo().getSuccessCount() + 1);
	}
}
