/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.dialogs;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;

import benten.twa.BentenTwaConstants;
import benten.twa.filter.messages.BentenConvertToXliffMessages;
import benten.twa.filter.status.BentenConvertToXliffStatus;
import benten.twa.ui.dialogs.AbstractWorkflowDialog;
import benten.ui.UiStatus;
import benten.ui.UiStatusException;
import benten.ui.fields.ResourceFolderTextField;
import benten.ui.preference.BentenProjectProperty;
import benten.ui.preference.BentenProjectProperty.ProjectProperty;

/**
 * HTML－XLIFF 変換ダイアログです。
 *
 * 翻訳対象を XLIFF に変換
 *
 * <pre>
 * 翻訳対象物を XLIFF に変換します。
 *   1.  プロジェクト内にインポートされた翻訳対象物を、XLIFF に変換します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳対象物－翻訳中間形式変換機能: HTML－XLIFF変換機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class ConvertToXliffDialog extends AbstractWorkflowDialog {

	/**
	 * HTML－XLIFF変換機能のためのメッセージ。
	 */
	protected static final BentenConvertToXliffMessages fMsg = new BentenConvertToXliffMessages();

	/** コントロール・コンテナー */
	private final ConvertToXliffControls controls = new ConvertToXliffControls();

	/** コントロール・コンテナー・クラス */
	public static class ConvertToXliffControls {

		/** 入力ディレクトリー・テキスト */
		public ResourceFolderTextField fromDirText;

		/** 出力ディレクトリー・テキスト */
		public ResourceFolderTextField toDirText;
	}

	/**
	 * コンストラクター。
	 * @param parentShell 親シェル
	 * @param selection 選択
	 */
	public ConvertToXliffDialog(final Shell parentShell, final IStructuredSelection selection) {
		super(parentShell, selection, fMsg.getWindowTitle());
	}

	@Override
	public void createControls(final Composite composite) {
		composite.setLayout(new GridLayout(3, false));

		controls.fromDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelFromDir());
		controls.fromDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.fromDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_SOURCE_DIR);
		}
		controls.fromDirText.setInfoDecoration(fMsg
				.getDialogCtrlLabelFromDirInfoDescription(BentenTwaConstants.DEFAULT_SOURCE_DIR));

		controls.toDirText = new ResourceFolderTextField(composite, fMsg.getDialogCtrlLabelToDir());
		controls.toDirText.addModifyListener(this);
		if (getSelectionResource() != null) {
			controls.toDirText.setText(getSelectionResource().getProject().getName() + "/" //$NON-NLS-1$
					+ BentenTwaConstants.DEFAULT_XLIFF_DIR);
		}
	}

	@Override
	public boolean okPressedPrompt() {
		final IProject project = controls.toDirText.getResource().getProject();
		final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);

		// このプロジェクトで処理済みとマークされている場合は確認ダイアログを表示
		if (projectStore.getBoolean(ProjectProperty.XLIFF_CONVERTED.name())) {

			final String prompt = projectStore.getString(ProjectProperty.PROMPT_XLIFF_CONVERTED.name());
			if (!MessageDialogWithToggle.ALWAYS.equals(prompt)) {

				final MessageDialogWithToggle dialog = MessageDialogWithToggle.openOkCancelConfirm(getShell(), fMsg
						.getDialogMsg01(), fMsg.getDialogMsg02(), fMsg.getDialogMsg03(), false, projectStore,
						ProjectProperty.PROMPT_XLIFF_CONVERTED.name());
				return dialog.getReturnCode() == Window.OK;
			}
		}
		return true;
	}

	@Override
	public IStatus getResultStatus() {
		final BentenConvertToXliffStatus status = new BentenConvertToXliffStatus();
		status.setFromDir(controls.fromDirText.getFile());
		status.setToDir(controls.toDirText.getFile());
		status.setProject(controls.toDirText.getResource());

		final IPreferenceStore projectStore = BentenProjectProperty.getStore(status.getProject());
		status.setTransTargetId(projectStore.getString(ProjectProperty.TRANS_TARGET_ID.name()));

		if (status.getTransTargetId() == null || status.getTransTargetId().trim().length() == 0) {
			// TRANS_TARGET_ID が空白の場合には「noname」をセットします。
			status.setTransTargetId("noname"); //$NON-NLS-1$
		}

		return status;
	}

	@Override
	public IStatus validate() throws UiStatusException {
		validatePath(controls.fromDirText);
		validatePath(controls.toDirText);
		validateFromTo(controls.fromDirText, controls.toDirText);

		// このプロジェクトで処理済みとマークされている場合（再実行）には警告を表示
		if (controls.toDirText != null) {
			final IProject project = controls.toDirText.getResource().getProject();
			final IPreferenceStore projectStore = BentenProjectProperty.getStore(project);
			if (projectStore.getBoolean(ProjectProperty.XLIFF_CONVERTED.name())) {
				throw new UiStatusException(IStatus.WARNING, fMsg.getDialogMsg04());
			}
		}
		return UiStatus.OK_STATUS;
	}
}
