/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.odf;

import java.util.List;

import blanco.xml.bind.valueobject.BlancoXmlCdata;
import blanco.xml.bind.valueobject.BlancoXmlCharacters;
import blanco.xml.bind.valueobject.BlancoXmlDocument;
import blanco.xml.bind.valueobject.BlancoXmlElement;
import blanco.xml.bind.valueobject.BlancoXmlNode;

/**
 * OpenDocument Format 形式ファイルの中に含まれる content.xml ファイルを翻訳処理するための抽象クラス。
 * 
 * 実際の翻訳処理は、このクラスを継承して実装します。
 * 
 * <UL>
 * <LI>以前は <w:instrText> を処理対象としていたが、これは不都合があるので処理対象外としました。
 * </UL>
 *
 * @author IGA Tosiki
 * @see "http://www.oasis-open.org/committees/office/"
 */
abstract class OdfXmlParser {
	/**
	 * XML ツリーをパースします。
	 * 
	 * @param document 処理対象となる XML ドキュメント。
	 */
	public void parse(final BlancoXmlDocument document) {
		final List<BlancoXmlNode> top = document.getChildNodes();
		for (BlancoXmlNode nodeRoot : top) {
			if (nodeRoot instanceof BlancoXmlElement) {
				final List<BlancoXmlNode> rootChilds = ((BlancoXmlElement) nodeRoot).getChildNodes();
				for (BlancoXmlNode nodeBody : rootChilds) {
					if (nodeBody instanceof BlancoXmlElement) {
						processChild((BlancoXmlElement) nodeBody, false);
					}
				}
			}
		}
	}

	/**
	 * XML ツリーの要素をパースします。
	 * 
	 * @param eleChild 処理対象となる XML 要素。
	 * @param isInTextElement 現在テキスト要素の配下にあるかどうか。
	 */
	void processChild(final BlancoXmlElement eleChild, boolean isInTextElement) {
		// 連続する characters について合併させます。
		for (int index = eleChild.getChildNodes().size() - 1; index >= 1; index--) {
			final BlancoXmlNode objPrev = eleChild.getChildNodes().get(index - 1);
			final BlancoXmlNode objCurrent = eleChild.getChildNodes().get(index);

			if (objPrev instanceof BlancoXmlCharacters && objCurrent instanceof BlancoXmlCharacters) {
				final BlancoXmlCharacters prevCharacters = (BlancoXmlCharacters) objPrev;
				final BlancoXmlCharacters currentCharacters = (BlancoXmlCharacters) objCurrent;

				// まとめてしまいます。
				prevCharacters.setValue(prevCharacters.getValue() + currentCharacters.getValue());
				// まとめたあとは、不要なノードを除去します。
				eleChild.getChildNodes().remove(index);
			}
		}

		if (eleChild.getLocalName().equals("p") || eleChild.getLocalName().equals("span")) { //$NON-NLS-1$
			isInTextElement = true;
		}

		for (BlancoXmlNode objLook : eleChild.getChildNodes()) {
			if (objLook instanceof BlancoXmlCharacters) {
				final BlancoXmlCharacters xmlCharacters = (BlancoXmlCharacters) objLook;
				final String text = xmlCharacters.getValue();
				if (text == null || text.length() == 0) {
					// 何もしません。
				} else {
					// こちらもテキスト要素です。これを処理します。
					if (isInTextElement) {
						fireText(xmlCharacters);
					}
				}
			} else if (objLook instanceof BlancoXmlCdata) {
				final BlancoXmlCdata cdataLook = (BlancoXmlCdata) objLook;
				final List<BlancoXmlNode> childs = cdataLook.getChildNodes();
				for (BlancoXmlNode nodeChild : childs) {
					if (nodeChild instanceof BlancoXmlElement) {
						processChild((BlancoXmlElement) nodeChild, isInTextElement);
					}
				}
			} else if (objLook instanceof BlancoXmlElement) {
				processChild((BlancoXmlElement) objLook, isInTextElement);
			}
		}
	}

	/**
	 * テキスト・イベントが発生した場合。
	 * 
	 * @param eleCharacters 処理対象となる XML 要素。
	 */
	public abstract void fireText(final BlancoXmlCharacters eleCharacters);
}
