/*******************************************************************************
 * Copyright (c) 2010  NEC Soft, Ltd.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.odf;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import benten.core.io.Files;

/**
 * OpenDocument Format 形式ファイルを翻訳処理するための抽象クラス。
 * 
 * 実際の翻訳処理は、このクラスを継承して実装します。
 *
 * @author IGA Tosiki
 * @see "http://www.oasis-open.org/committees/office/"
 */
abstract class OdfZipProcessor {
	/**
	 * ECMA-376 形式のファイルを処理します。
	 * 
	 * @param fileName 処理対象のファイル名。
	 * @param inputDocx 入力となる ODF 形式ファイルのバイト配列形式データ。
	 * @param outputDocx 出力となる ODF 形式ファイルのバイト配列形式データ。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public void processZip(final String fileName, final byte[] inputDocx, final OutputStream outputDocx)
			throws IOException {
		final ZipInputStream zipInStream = new ZipInputStream(new BufferedInputStream(new ByteArrayInputStream(
				inputDocx)));
		final ZipOutputStream zipOutStream = new ZipOutputStream(new BufferedOutputStream(outputDocx));

		int entryCount = 0;

		try {
			entryCount = processZip(zipInStream, zipOutStream);
		} finally {
			zipInStream.close();
			try {
				zipOutStream.close();
			} catch (ZipException ex) {
				if (entryCount == 0) {
					// ZIP エントリが一件も存在しないケースはこちら。
					// 仕方が無いので無視して進めます。
					throw new IllegalArgumentException("No zip entry found. ODF file '" + fileName + "' is NOT valid."); //$NON-NLS-1$ //$NON-NLS-2$
				} else {
					throw new IllegalArgumentException("ZipException occurred processing file '" + fileName //$NON-NLS-1$
							+ "'. Zip entry number is " + entryCount + "."); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
	}

	/**
	 * OpenDocument Format 形式のファイルを処理します。
	 * 
	 * @param zipInStream 入力となる ODF 形式ファイルのバイト・ストリーム形式データ。
	 * @param zipOutStream 出力となる ODF 形式ファイルのバイト・ストリーム形式データ。
	 * @return 処理した ZIP エントリの個数。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	public int processZip(final ZipInputStream zipInStream, final ZipOutputStream zipOutStream) throws IOException {
		int entryCount = 0;

		for (;; entryCount++) {
			final ZipEntry entry = zipInStream.getNextEntry();
			if (entry == null) {
				break;
			}

			final ByteArrayOutputStream bufOut = new ByteArrayOutputStream();
			Files.copy(zipInStream, bufOut);

			final ZipEntry newEntry = new ZipEntry(entry.getName());
			// 可能な範囲で ZipEntry のフィールドを複写します。
			newEntry.setTime(entry.getTime());
			if (entry.getComment() != null) {
				newEntry.setComment(entry.getComment());
			}
			// ZIP エントリを設定します。
			zipOutStream.putNextEntry(newEntry);

			if (entry.getName().equals("content.xml")) { //$NON-NLS-1$
				processXml(new ByteArrayInputStream(bufOut.toByteArray()), zipOutStream);
			} else {
				Files.copy(new ByteArrayInputStream(bufOut.toByteArray()), zipOutStream);
			}

			zipOutStream.closeEntry();
		}

		zipOutStream.flush();

		return entryCount;
	}

	/**
	 * 「???/*.xml」ファイルを処理します。
	 * 
	 * @param inStream xml ファイルの入力ストリーム。
	 * @param outStream xml ファイルの出力ストリーム。
	 */
	public abstract void processXml(final InputStream inStream, final OutputStream outStream);
}
