/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.engine.text;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import benten.core.model.HelpTransUnitId;
import benten.twa.filter.core.BentenTwaFilterEngine;
import benten.twa.filter.model.SentencePartitionList;
import benten.twa.process.BentenProcessResultInfo;
import blanco.commons.util.BlancoFileUtil;
import blanco.xliff.valueobject.BlancoXliff;
import blanco.xliff.valueobject.BlancoXliffBody;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTarget;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * TEXT を処理します。
 * 
 * @author IGA Tosiki
 */
public class BentenTwaFilterTextEngine implements BentenTwaFilterEngine {
	/** 復帰改行 \r\n */
	public static final int NEWLINE_CRLF = 1;
	/** 復帰 \r */
	public static final int NEWLINE_CR = 2;
	/** 改行 \n */
	public static final int NEWLINE_LF = 3;

	/**
	 * 連続する改行をあらわす正規表現。
	 */
	protected static final Pattern CONTINUED_NEWLINE_PATTERN = Pattern.compile("\\s*\n\\s*\n"); //$NON-NLS-1$

	/**
	 * {@inheritDoc}
	 */
	public boolean canProcess(final File file) {
		if (file.getName().toLowerCase().endsWith(".utf8") || file.getName().toLowerCase().endsWith(".utf-8")) { //$NON-NLS-1$ //$NON-NLS-2$
			return true;
		} else {
			return false;
		}
	}

	/**
	 * {@inheritDoc}
	 */
	public void extractXliff(final File sourceFile, final BlancoXliff xliff, final HelpTransUnitId id,
			final BentenProcessResultInfo resultInfo) throws IOException {
		final BlancoXliffFile xliffFile = xliff.getFileList().get(0);
		final BlancoXliffBody body = xliffFile.getBody();

		// テキストを表現します。
		xliffFile.setDatatype("x-text"); //$NON-NLS-1$

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);
		String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		// 改行コードを LF に置き換え。
		text = normalizeNewline(text);

		// 連続する改行が 2 回以上続いたら翻訳単位を分割する。
		for (;;) {
			final Matcher matcher = CONTINUED_NEWLINE_PATTERN.matcher(text);
			if (matcher.find() == false) {
				break;
			}

			final String head = text.substring(0, matcher.start());
			// text の内容を更新します。
			text = text.substring(matcher.end());

			createTransUnits(body, id, head, resultInfo);
		}

		// 分割後の最後の行を処理します。
		if (text.length() > 0) {
			createTransUnits(body, id, text, resultInfo);
		}
	}

	/**
	 * 与えられたテキストを処理します。
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param id trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	void createTransUnits(final BlancoXliffBody body, final HelpTransUnitId id, final String source,
			BentenProcessResultInfo resultInfo) {
		final SentencePartitionList sentenceList = new SentencePartitionList(source);
		if (sentenceList.size() > 1) {
			for (final String sentence : sentenceList) {
				id.incrementSubSeq(sentenceList.size());
				createTransUnit(body, id, sentence, resultInfo);
			}
		} else {
			createTransUnit(body, id, source, resultInfo);
		}
		id.incrementSeq();
	}

	/**
	 * trans-unit の作成。
	 * 
	 * <UL>
	 * <LI>このメソッドは HTML -> XLIFF で利用されます。
	 * </UL>
	 * 
	 * @param body XLIFF の body 要素モデル
	 * @param helpId trans-unit の id 属性モデル
	 * @param source source 要素の値
	 * @param resultInfo 翻訳結果情報
	 */
	private void createTransUnit(final BlancoXliffBody body, final HelpTransUnitId helpId, final String source,
			BentenProcessResultInfo resultInfo) {
		resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

		final BlancoXliffTransUnit unit = new BlancoXliffTransUnit();
		unit.setId(helpId.toString());
		unit.setSource(source);
		body.getTransUnitList().add(unit);
	}

	/**
	 * {@inheritDoc}
	 */
	public byte[] mergeXliff(final File sourceFile, final BlancoXliff xliff, final BentenProcessResultInfo resultInfo)
			throws IOException {

		final Iterator<BlancoXliffTransUnit> units = xliff.getFileList().get(0).getBody().getTransUnitList().iterator();

		final StringBuilder target = new StringBuilder();

		final byte[] bytesFile = BlancoFileUtil.file2Bytes(sourceFile);
		String text = new String(bytesFile, "UTF-8"); //$NON-NLS-1$

		// 改行コードの判別
		final int newLineType = decideNewline(text);

		// 改行コードを LF に置き換え。
		text = normalizeNewline(text);

		// 連続する改行が 2 回以上続いたら翻訳単位を分割する。
		for (;;) {
			final Matcher matcher = CONTINUED_NEWLINE_PATTERN.matcher(text);
			if (matcher.find() == false) {
				break;
			}

			final String skip = text.substring(matcher.start(), matcher.end());
			text = text.substring(matcher.end());

			target.append(mergeTextInner(units, resultInfo));

			target.append(skip);
		}

		// 分割後の最後の行を処理します。
		if (text.length() > 0) {
			target.append(mergeTextInner(units, resultInfo));
		}

		String outputString = target.toString();
		// 改行コードの復元
		outputString = restoreNewline(outputString, newLineType);

		final ByteArrayOutputStream outStream = new ByteArrayOutputStream();
		outStream.write(outputString.getBytes("UTF-8"));//$NON-NLS-1$
		outStream.flush();
		byte[] bytes = outStream.toByteArray();

		return bytes;
	}

	/**
	 * 与えられた XLIFF 翻訳単位から文字列を抽出します。
	 * 
	 * @param units
	 *            翻訳単位リスト。
	 * @param resultInfo
	 *            翻訳結果情報。
	 * @return 文字列。
	 */
	String mergeTextInner(final Iterator<BlancoXliffTransUnit> units, final BentenProcessResultInfo resultInfo) {
		final SentencePartitionList targetList = new SentencePartitionList();
		while (units.hasNext()) {
			resultInfo.setUnitCount(resultInfo.getUnitCount() + 1);

			final BlancoXliffTransUnit unit = units.next();
			final BlancoXliffTarget t = unit.getTarget();

			if (t != null && t.getTarget() != null && !t.getTarget().equals("")) { //$NON-NLS-1$
				targetList.add(t.getTarget());
			} else {
				targetList.add(unit.getSource());
			}

			if (!HelpTransUnitId.hasContinue(unit.getId())) {
				break;
			}
		}
		return targetList.join();
	}

	/**
	 * 改行コードを判定します。
	 * 
	 * @param text
	 *            判定する文字列。
	 * @return 改行コードの種別。
	 */
	static int decideNewline(final String text) {
		// 改行コードの判別
		final int crlf = text.indexOf("\r\n");
		final int cr = text.indexOf("\r");
		final int lf = text.indexOf("\n");
		int min = -1;
		if (crlf >= 0) {
			min = crlf;
		}
		if (cr >= 0) {
			if (min >= 0) {
				min = Math.min(min, cr);
			} else {
				min = cr;
			}
		}
		if (lf >= 0) {
			if (min >= 0) {
				min = Math.min(min, lf);
			} else {
				min = lf;
			}
		}

		if (min < 0) {
			// 何もしません。なぜなら改行コードは存在しないからです。
		} else if (crlf >= 0 && min == crlf) {
			return NEWLINE_CRLF;
		} else if (lf >= 0 && min == lf) {
			return NEWLINE_LF;
		} else {
			return NEWLINE_CR;
		}

		return NEWLINE_LF;
	}

	/**
	 * 改行コードを \n にそろえます。
	 * @param text 入力テキスト。
	 * @return 改行コードがそろえられたテキスト。
	 */
	static String normalizeNewline(final String text) {
		return text.replaceAll("\r\n", "\n").replaceAll("\r", "\n");
	}

	/**
	 * 改行コードを復元します。
	 * 
	 * @param text
	 *            テキスト。
	 * @param newLineType
	 *            改行コード種別。
	 * @return 改行復元後のテキスト。
	 */
	static String restoreNewline(final String text, final int newLineType) {
		switch (newLineType) {
		case NEWLINE_CRLF:
			return text.replaceAll("\n", "\r\n");
		case NEWLINE_CR:
			return text.replaceAll("\n", "\r");
		case NEWLINE_LF:
		default:
			// 何もしません。必要が無いからです。
			return text;
		}
	}
}
