/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.filter.model;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import benten.core.text.Strings;

/**
 * HTML ブロックです。
 *
 * @author KASHIHARA Shinji
 */
public class HtmlBlock {

	/**
	 * デバッグするかどうか。
	 *
	 * <UL>
	 * <LI>Benten 開発者向けのデバッグ・フラグです。
	 * <LI>Java 起動オプションで -Dbenten.debug=true を指定することにより有効となります。
	 * </UL>
	 */
	private static final boolean IS_DEBUG = Boolean.valueOf(System.getProperty("benten.debug")); //$NON-NLS-1$

	/**
	 * 先頭末尾がタグの正規表現パターン。
	 * <UL>
	 * <LI>グループ 1: 先頭にある HTML タグです。
	 * <LI>グループ 2: グループ 1 と 3 の間部分。タグが含まれる場合があります。
	 * <LI>グループ 3: 末尾にある HTML タグです。
	 * </UL>
	 */
	private static final Pattern enclosedTagPattern = Pattern.compile("(?i)(?s)^(<[^>]+>\\s*)(.+)(\\s*<[^>]+>)$"); //$NON-NLS-1$

	/** 正規表現マッチャー */
	private final Matcher matcher;

	/** 全体がブロック・タグで囲まれている場合は true */
	private final boolean isBlockTag;

	/** HTML コメント開始タグが含まれている場合は true */
	private boolean containesHtmlCommentStartTag;

	/** HTML コメント終了タグが含まれている場合は true */
	private boolean containesHtmlCommentEndTag;

	/** インライン・テキスト部分 */
	private final String inlineText;

	/**
	 * コンストラクター。
	 * @param input 入力文字列
	 */
	public HtmlBlock(final String input) {
		matcher = enclosedTagPattern.matcher(input);
		isBlockTag = matcher.find();
		if (!isBlockTag) {
			inlineText = ""; //$NON-NLS-1$
			return;
		}

		if (IS_DEBUG) {
			System.out.println(""); //$NON-NLS-1$
			System.out.println("[BLOCK ALL  ] " + Strings.removeRedundantWhitespace(toString())); //$NON-NLS-1$
			System.out.println("[BLOCK START] " + Strings.removeRedundantWhitespace(startTag())); //$NON-NLS-1$
			System.out.println("[BLOCK BODY ] " + Strings.removeRedundantWhitespace(body())); //$NON-NLS-1$
			System.out.println("[BLOCK END  ] " + Strings.removeRedundantWhitespace(endTag())); //$NON-NLS-1$
		}

		inlineText = trim(body());

		if (IS_DEBUG) {
			System.out.println("---------------------------------"); //$NON-NLS-1$
			System.out.println("[After  ] " + Strings.removeRedundantWhitespace(inlineText)); //$NON-NLS-1$
		}
	}

	/**
	 * ブロック・タグを除くテキストを取得します。
	 * @return テキスト
	 */
	public String getInlineText() {
		return inlineText;
	}

	/**
	 * ブロック・タグか判定します。
	 * @return ブロック・タグの場合は true
	 */
	public boolean isBlockTag() {
		return isBlockTag;
	}

	/**
	 * 先頭開始タグと末尾終了タグの要素名が同じか判定します。
	 * @return 同じ場合は true
	 */
	public boolean equalsStartEndTag() {
		if (startTagName().equals(endTagName())) {

			// 先頭と末尾のタグが同じ場合でも、途中に同じタグがある場合、false を返すために正規表現判定。
			// このメソッドは先頭末尾のタグが同じ場合に、除去するときの判定に使用されるため、
			// 下記のようなケースに一致する場合は false としている。
			//
			//   例) <a ...>abc</a><a ...>xyz</a>
			//       この前後タグを除去すると、abc</a><a ...>xyz となり、翻訳対象の単位としては不正。
			//
			final String pattern = "(?i)(?s).*<" + startTagName() + "(\\s|>).*"; //$NON-NLS-1$ //$NON-NLS-2$
			return !body().matches(pattern);
		}
		return false;
	}

	/**
	 * 先頭の開始タグを取得します。
	 * @return 先頭の開始タグ
	 */
	public String startTag() {
		return matcher.group(1);
	}

	/**
	 * 先頭の開始タグ名を取得します。
	 * @return 先頭の開始タグ名
	 */
	public String startTagName() {
		return getTagName(startTag());
	}

	/**
	 * 先頭末尾のブロック・タグを除いたテキストを取得します。
	 * ブロック・タグが含まれている場合があります。
	 * @return テキスト
	 */
	public String body() {
		return matcher.group(2);
	}

	/**
	 * 末尾の終了タグを取得します。
	 * @return 末尾の終了タグ
	 */
	public String endTag() {
		return matcher.group(3);
	}

	/**
	 * 末尾の終了タグ名を取得します。
	 * @return 末尾の終了タグ名
	 */
	public String endTagName() {
		return getTagName(endTag());
	}

	/**
	 * 末尾の終了タグ位置を取得します。
	 * @return 末尾の終了位置
	 */
	public int endTagIndex() {
		return matcher.start(3);
	}

	private String getTagName(final String tag) {
		return tag.trim().replaceAll("([/<>]|\\s\\w.+)", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * HTML コメント開始タグ (<!--) が含まれるか判定します。
	 * @return 含まれる場合は true
	 */
	public boolean containesHtmlCommentStartTag() {
		return containesHtmlCommentStartTag;
	}

	/**
	 * HTML コメント終了タグ (-->) が含まれるか判定します。
	 * @return 含まれる場合は true
	 */
	public boolean containesHtmlCommentEndTag() {
		return containesHtmlCommentEndTag;
	}

	@Override
	public String toString() {
		return startTag() + body() + endTag();
	}

	/**
	 * テキストをトリム。
	 * @param text トリムしたいテキスト。
	 * @return トリム後のテキスト。
	 */
	private String trim(final String text) {
		// 前後のブロック・タグ除去
		String s = HtmlBlockMatcher.trimBlockTag(text);

		// HTML コメント処理
		s = removeHtmlComment(s);

		// 読み飛ばし判定
		if (isIgnore(s)) {
			return ""; //$NON-NLS-1$
		}
		// 全体が a タグでない場合 (a タグが片側だけになるのを防止)
		if (!s.matches("(?i)(?s)<a\\s.+?>.*</a>")) { //$NON-NLS-1$

			// 前後のインライン・タグのうち、タグ内のテキストが空の場合は除去
			s = HtmlBlockMatcher.trimTag(s, "<[^</]+>\\s*</[^<]+>"); //$NON-NLS-1$
		}

		// この箇所が必要かどうか検証したが、Benten では機械的に分割作業を実施する以上、この処理は存在するのが妥当と判断した。

		// 前後の対応するインライン・タグを除去 (属性に title、alt、href がある場合は除去しない)
		final HtmlBlock block = new HtmlBlock(s);
		if (block.isBlockTag() && block.equalsStartEndTag()) {
			if (!needsTranslationAttribute(block.startTag())) {
				s = block.body();
			}
		}

		if (s.equals(text)) {
			return s;
		}
		return trim(s); // 再帰
	}

	/**
	 * HTML コメントを除去。
	 * @param s 除去前の文字列。
	 * @return コメント除去後の文字列。
	 */
	private String removeHtmlComment(String s) {
		final String HTML_COMMENT_START = "<!--"; //$NON-NLS-1$
		final String HTML_COMMENT_END = "-->"; //$NON-NLS-1$

		if (s.contains(HTML_COMMENT_START) && s.contains(HTML_COMMENT_END)) {
			// インライン・タグ内で完結する HTML コメントは削除しません。
			// インライン・タグ内は翻訳対象であるため、除去してしまうと、エクスポート時に戻しようがないためです。
			return s;
		}
		if (s.contains(HTML_COMMENT_START)) {
			final String pattern = "(?i)(?s)" + HTML_COMMENT_START + ".*"; //$NON-NLS-1$ //$NON-NLS-2$
			s = s.replaceFirst(pattern, ""); //$NON-NLS-1$
			containesHtmlCommentStartTag = true;
		} else if (s.contains(HTML_COMMENT_END)) {
			final String pattern = "(?i)(?s)" + ".*?" + HTML_COMMENT_END; //$NON-NLS-1$ //$NON-NLS-2$
			s = s.replaceFirst(pattern, ""); //$NON-NLS-1$
			containesHtmlCommentEndTag = true;
		}
		return s;
	}

	/**
	 * 無視する文字列かどうかを判定。
	 * @param s 判定文字列。
	 * @return 無視する文字列なら true。
	 */
	private boolean isIgnore(final String s) {
		// 1 文字の場合
		if (s.length() == 1) {
			return true;
		}
		// 全体が 1 つのタグ (テキスト部が無い) 、かつ翻訳が必要な属性が無い場合
		if (s.matches("(?i)(?s)<[^<]+>") && !needsTranslationAttribute(s)) { //$NON-NLS-1$
			return true;
		}
		// 全体が属性の無い 1 つのタグ (つまりテキスト部が無い) の場合
		if (s.matches("(?i)(?s)^<[^<\\s]+>$")) { //$NON-NLS-1$
			return true;
		}
		return false;
	}

	/**
	 * 翻訳が必要か判定。
	 * @param text テキスト
	 * @return 翻訳が必要な場合は true
	 */
	private boolean needsTranslationAttribute(final String text) {
		// title、alt、href 属性を含む文字列に一致する場合は true
		return text.matches("(?i)(?s).+?\\s(title|alt|href)=\".+"); //$NON-NLS-1$
	}
}
