/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.xliff.dialogs;

import java.io.ByteArrayInputStream;
import java.lang.reflect.Field;

import junit.framework.TestCase;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;

import benten.twa.ui.wizards.NewProjectCreationWizard;
import benten.twa.xliff.dialogs.ExtractXliffFragmentDialog.ExtractXliffFragmentControls;
import benten.ui.UiStatusException;
import benten.ui.fields.AbstractBrowseTextField;
import benten.ui.fields.TextField;

public class ExtractXliffFragmentDialogTest extends TestCase {

	private final ExtractXliffFragmentDialog target = new ExtractXliffFragmentDialog(null, new StructuredSelection());

	private String getValidateMessage() {
		try {
			target.validate();
			return null;
		} catch (final UiStatusException e) {
			return e.getMessage();
		}
	}

	private boolean validate(final TextField field, final String value) {
		AbstractBrowseTextField.clearPreviousInput();
		field.setText(value);
		final String message = getValidateMessage();
		if (message == null) {
			return true;
		}
		return !message.contains(field.getLabelText());
	}

	private boolean validateFromTo(final TextField field1, final TextField field2, final String value1,
			final String value2) {
		AbstractBrowseTextField.clearPreviousInput();
		field1.setText(value1);
		field2.setText(value2);
		final String message = getValidateMessage();
		if (message == null) {
			return true;
		}
		return !(message.contains(field1.getLabelText()) && message.contains(field2.getLabelText()));
	}

	public void testValidate() throws Exception {

		// テスト対象のダイアログからコントロール・コンテナーの取得
		target.createControls(new Composite(new Shell(), SWT.NONE));
		final Field field = target.getClass().getDeclaredField("controls");
		field.setAccessible(true);
		final ExtractXliffFragmentControls controls = (ExtractXliffFragmentControls) field.get(target);

		// ワークスペースにテスト用のプロジェクトを作成
		final IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		final IProject project = root.getProject("test.project");
		new NewProjectCreationWizard().createProject(project);
		final IFile file = project.getFile(new Path("test.file"));
		if (!file.exists()) {
			file.create(new ByteArrayInputStream("".getBytes()), true, null);
		}
		final IFile inXliff = project.getFolder("xliff").getFile(new Path("in.xlf"));
		if (!inXliff.exists()) {
			inXliff.create(new ByteArrayInputStream("".getBytes()), true, null);
		}

		assertFalse(validate(controls.fromXliffFileText, null));
		assertFalse(validate(controls.fromXliffFileText, ""));
		assertFalse(validate(controls.fromXliffFileText, ".先頭ピリオド"));
		assertFalse(validate(controls.fromXliffFileText, "末尾ピリオド."));
		assertFalse(validate(controls.fromXliffFileText, "存在しないリソース"));
		assertFalse(validate(controls.fromXliffFileText, "test.project"));
		assertFalse(validate(controls.fromXliffFileText, "test.project/test.file"));
		assertTrue(validate(controls.fromXliffFileText, "test.project/xliff/in.xlf"));

		assertFalse(validate(controls.toDirText, null));
		assertFalse(validate(controls.toDirText, ""));
		assertFalse(validate(controls.toDirText, ".先頭ピリオド"));
		assertFalse(validate(controls.toDirText, "末尾ピリオド."));
		assertFalse(validate(controls.toDirText, "存在しないリソース"));
		assertFalse(validate(controls.toDirText, "test.project"));
		assertFalse(validate(controls.toDirText, "test.project/test.file"));
		assertTrue(validate(controls.toDirText, "test.project/xliff.fragment"));

		assertNull(getValidateMessage());

		assertFalse(validateFromTo(controls.fromXliffFileText, controls.toDirText, "test.project/xliff/in.xlf",
				"test.project/xliff"));
		assertTrue(validateFromTo(controls.fromXliffFileText, controls.toDirText, "test.project/xliff/in.xlf",
				"test.project/xliff.fragment"));

		controls.countCheck.setSelection(true);
		controls.countText.setText("0");
		assertNotNull(getValidateMessage());

		controls.countText.setText("x");
		assertNotNull(getValidateMessage());

		controls.countText.setText(String.valueOf(Integer.MIN_VALUE));
		assertNotNull(getValidateMessage());

		controls.countText.setText(String.valueOf(Integer.MAX_VALUE));
		assertNull(getValidateMessage());
	}
}
