/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.io;

import java.io.File;
import java.io.IOException;

/**
 * ディレクトリを再帰的に走査します。
 *
 * <UL>
 * <LI>Benten 翻訳ワークフロー支援でディレクトリーを再帰的に処理したい場合に利用します。
 * </UL>
 *
 * @author IGA Tosiki
 * @author KASHIHARA Shinji
 */
public abstract class AbstractTraverseDir {
	/**
	 * 指定されたディレクトリーを走査開始。
	 * @param dir 走査するディレクトリー。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	protected void processDir(final File dir) throws IOException {
		if (dir == null) {
			throw new IllegalArgumentException("AbstractTraverseDir#processDir: argument 'dir' is null."); //$NON-NLS-1$
		}

		processDir(dir, ""); //$NON-NLS-1$
	}

	/**
	 * 内部的に利用されるディレクトリー走査。
	 * @param dir 処理対象ディレクトリー。
	 * @param baseDir 基準ディレクトリー。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	private final void processDir(final File dir, final String baseDir) throws IOException {
		if (dir.getName().equals(".svn")) { //$NON-NLS-1$
			// Subversion 関連ファイルは処理対象外とします。
			return;
		}

		final File[] files = dir.listFiles();
		if (files == null) {
			return;
		}
		for (final File file : files) {
			if (file.isDirectory()) {
				processDir(file, (baseDir.length() == 0 ? "" : baseDir + file.getName() + "/")); //$NON-NLS-1$ //$NON-NLS-2$
				continue;
			}
			if (file.isFile() == false) {
				// ファイルではないので処理しません。
				continue;
			}
			if (file.canRead() == false) {
				// 読み込みできないので処理しません。
				continue;
			}
			if (canProcess(file)) {
				// 通常に処理できるファイルです。
				processFile(file, baseDir);
			} else {
				// 通常は処理できないファイルです。
				// ただし、翻訳成果インポートではバイナリー・ファイルの読み込みの際にこちらが利用されます。
				processFilterdFile(file, baseDir);
			}
		}
	}

	/**
	 * タスクの開始時にトータル作業数を報告するためのメソッド。
	 *
	 * @param totalWork トータルの作業数。
	 */
	protected void beginTask(final int totalWork) {
	}

	/**
	 * このファイルが処理できるものであるかどうかをチェックします。
	 *
	 * 主にファイルの拡張子チェックを実施します。
	 *
	 * @param file 処理対象ファイル。必ず null 以外を与えることとします。
	 * @return 処理できるなら true。処理できないなら false。
	 */
	protected abstract boolean canProcess(final File file);

	/**
	 * 走査された先にある処理対象ファイルの処理。
	 *
	 * @param file 処理対象ファイル。必ず null 以外を与えることとします。
	 * @param baseDir 基準ディレクトリ－。ルートの場合には "" を与えます。
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected abstract void processFile(final File file, final String baseDir) throws IOException;

	/**
	 * 走査された先にあるフィルターされた（canProcess() が false）ファイルの処理。
	 *
	 * @param file フィルターされたファイル。必ず null 以外を与えることとします。
	 * @param baseDir 基準ディレクトリ－。ルートの場合には "" を与えます。
	 * @throws IOException 入出力例外が発生した場合
	 */
	protected void processFilterdFile(final File file, final String baseDir) throws IOException {
	}
}
