/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.ui;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.osgi.framework.BundleContext;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.glossary.engine.AbstractBentenGlossaryEnginePlugin;
import benten.cat.glossary.ui.messages.CatGlossaryUiPluginMessages;
import benten.ui.AbstractBentenUiPlugin;
import benten.ui.UiPlugin;
import benten.ui.UiStatus;
import benten.ui.preference.BentenPreference;
import benten.ui.preference.BentenPreference.Preference;

/**
 * CAT 用語集 UI プラグイン。
 *
 * <UL>
 * <LI>このプラグインを Eclipse 上で有効化するためのプラグイン・クラスです。
 * </UL>
 *
 * @author YAMAMOTO Koji
 */
public class CatGlossaryUiPlugin extends AbstractBentenUiPlugin {
	/**
	 * 用語集 UI プラグインのためのメッセージ。
	 */
	protected static final CatGlossaryUiPluginMessages fMsg = new CatGlossaryUiPluginMessages();

	/** 共用インスタンス */
	private static CatGlossaryUiPlugin plugin;

	/**
	 * 用語集エンジン・プラグインのマップ。
	 */
	final Map<String, AbstractBentenGlossaryEnginePlugin> fGlossaryMap = new HashMap<String, AbstractBentenGlossaryEnginePlugin>();

	/**
	 * このプラグインが保持している用語集エンジン・プラグインのインスタンス。
	 */
	private AbstractBentenGlossaryEnginePlugin fGlossaryEnginePlugin = null;

	@Override
	public void start(final BundleContext context) throws Exception {
		super.start(context);
		plugin = this;
		loadExtensions();
	}

	@Override
	public void stop(final BundleContext context) throws Exception {
		plugin = null;
		super.stop(context);
	}

	/**
	 * このプラグインの共用インスタンスを取得。
	 * @return このプラグインの共用インスタンス
	 */
	public static CatGlossaryUiPlugin getDefault() {
		return plugin;
	}

	/**
	 * このプラグインが利用する拡張をロード。
	 *
	 * <UL>
	 * <LI>デフォルトの用語集エンジン・プラグインのインスタンスを取得します。
	 * <LI>用語集をロードします。
	 * </UL>
	 */
	public void loadExtensions() {
		fGlossaryEnginePlugin = getGlossaryEnginePlugin();
		try {
			fGlossaryEnginePlugin.load();
		} catch (Exception e) {
			final UiStatus s = new UiStatus(IStatus.ERROR, fMsg.getMsg001(e.getMessage()));
			UiPlugin.openDialog(fMsg.getMsg002(), s);
		}
	}

	/**
	 * このプラグインが保持している グロッサリー・エンジンを取得。
	 *
	 * @return グロッサリー・エンジンのインスタンス。
	 */
	public BentenGlossaryEngine getGlossaryEngine() {
		return fGlossaryEnginePlugin.getEngine();
	}

	/**
	 * グロッサリー・エンジンのインスタンスを新規に作成。
	 *
	 * 所定のルールに従って拡張ポイントを検索して グロッサリー・エンジンのインスタンスを作成します。
	 *
	 * @return グロッサリー・エンジンのインスタンス。必ず null 以外が戻ります。
	 */
	public AbstractBentenGlossaryEnginePlugin getGlossaryEnginePlugin() {
		final IExtensionRegistry registry = Platform.getExtensionRegistry();
		final IExtensionPoint point = registry.getExtensionPoint("benten.cat.glossary.engine"); //$NON-NLS-1$
		if (point == null) {
			throw new IllegalArgumentException(fMsg.getGetGlossaryEnginePluginE001());
		}

		for (IExtension extension : point.getExtensions()) {
			for (IConfigurationElement cfgElem : extension.getConfigurationElements()) {
				try {
					final String name = cfgElem.getAttribute("name"); //$NON-NLS-1$
					final Object obj = cfgElem.createExecutableExtension("class"); //$NON-NLS-1$
					if (obj instanceof AbstractBentenGlossaryEnginePlugin) {
						fGlossaryMap.put(name, (AbstractBentenGlossaryEnginePlugin) obj);
					}
				} catch (final CoreException e) {
					log(e);
				}
			}
		}

		final IPreferenceStore store = BentenPreference.getStore();
		final String tmName = store.getString(Preference.GLOSSARY_NAME.name());
		final AbstractBentenGlossaryEnginePlugin enginePlugin = fGlossaryMap.get(tmName);
		if (enginePlugin == null) {
			throw new IllegalArgumentException(fMsg.getGetGlossaryEnginePluginE002());
		}

		return enginePlugin;
	}

	/**
	 * 用語集エンジン名のセットを取得。
	 * @return 用語集エンジン名のセット
	 */
	public Set<String> getTmNameSet() {
		return fGlossaryMap.keySet();
	}
}
