﻿/*
 * VsqMetaText/VsqMetaText.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Lib.Vsq.
 *
 * Boare.Lib.Vsq is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Lib.Vsq is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Drawing;
using System.Windows.Forms;

namespace Boare.Lib.Vsq {

    /// <summary>
    /// vsqのメタテキストの中身を処理するためのクラス
    /// </summary>
    [Serializable]
    public class VsqMetaText : ICloneable {
        public VsqCommon common;
        internal VsqMaster master;
        internal VsqMixer mixer;
        private VsqEventList m_events;
        /// <summary>
        /// PIT。ピッチベンド。default=0
        /// </summary>
        private VsqBPList pitchBendBPList;
        /// <summary>
        /// PBS。ピッチベンドセンシティビティ。dfault=2
        /// </summary>
        private VsqBPList pitchBendSensBPList;
        /// <summary>
        /// DYN。ダイナミクス。default=64
        /// </summary>
        private VsqBPList dynamicsBPList;
        /// <summary>
        /// BRE。ブレシネス。default=0
        /// </summary>
        private VsqBPList epRResidualBPList;
        /// <summary>
        /// BRI。ブライトネス。default=64
        /// </summary>
        private VsqBPList epRESlopeBPList;
        /// <summary>
        /// CLE。クリアネス。default=0
        /// </summary>
        private VsqBPList epRESlopeDepthBPList;
        private VsqBPList reso1FreqBPList;
        private VsqBPList reso2FreqBPList;
        private VsqBPList reso3FreqBPList;
        private VsqBPList reso4FreqBPList;
        private VsqBPList reso1BWBPList;
        private VsqBPList reso2BWBPList;
        private VsqBPList reso3BWBPList;
        private VsqBPList reso4BWBPList;
        private VsqBPList reso1AmpBPList;
        private VsqBPList reso2AmpBPList;
        private VsqBPList reso3AmpBPList;
        private VsqBPList reso4AmpBPList;
        /// <summary>
        /// GEN。ジェンダーファクター。default=64
        /// </summary>
        private VsqBPList genderFactorBPList;
        /// <summary>
        /// POR。ポルタメントタイミング。default=64
        /// </summary>
        private VsqBPList portamentoTimingBPList;
        /// <summary>
        /// OPE。オープニング。default=127
        /// </summary>
        private VsqBPList openingBPList;

        public object Clone() {
            VsqMetaText res = new VsqMetaText();
            if ( common != null ) {
                res.common = (VsqCommon)common.Clone();
            }
            if ( master != null ) {
                res.master = (VsqMaster)master.Clone();
            }
            if ( mixer != null ) {
                res.mixer = (VsqMixer)mixer.Clone();
            }
            if ( m_events != null ) {
                res.m_events = new VsqEventList();
                foreach ( VsqEvent item in m_events ) {
                    res.m_events.Add( (VsqEvent)item.Clone() );
                }
            }
            if ( pitchBendBPList != null ) {
                res.pitchBendBPList = (VsqBPList)pitchBendBPList.Clone();
            }
            if ( pitchBendSensBPList != null ) {
                res.pitchBendSensBPList = (VsqBPList)pitchBendSensBPList.Clone();
            }
            if ( dynamicsBPList != null ) {
                res.dynamicsBPList = (VsqBPList)dynamicsBPList.Clone();
            }
            if ( epRResidualBPList != null ) {
                res.epRResidualBPList = (VsqBPList)epRResidualBPList.Clone();
            }
            if ( epRESlopeBPList != null ) {
                res.epRESlopeBPList = (VsqBPList)epRESlopeBPList.Clone();
            }
            if ( epRESlopeDepthBPList != null ) {
                res.epRESlopeDepthBPList = (VsqBPList)epRESlopeDepthBPList.Clone();
            }
            if ( reso1FreqBPList != null ) {
                res.reso1FreqBPList = (VsqBPList)reso1FreqBPList.Clone();
            }
            if ( reso2FreqBPList != null ) {
                res.reso2FreqBPList = (VsqBPList)reso2FreqBPList.Clone();
            }
            if ( reso3FreqBPList != null ) {
                res.reso3FreqBPList = (VsqBPList)reso3FreqBPList.Clone();
            }
            if ( reso4FreqBPList != null ) {
                res.reso4FreqBPList = (VsqBPList)reso4FreqBPList.Clone();
            }
            if ( reso1BWBPList != null ) {
                res.reso1BWBPList = (VsqBPList)reso1BWBPList.Clone();
            }
            if ( reso2BWBPList != null ) {
                res.reso2BWBPList = (VsqBPList)reso2BWBPList.Clone();
            }
            if ( reso3BWBPList != null ) {
                res.reso3BWBPList = (VsqBPList)reso3BWBPList.Clone();
            }
            if ( reso4BWBPList != null ) {
                res.reso4BWBPList = (VsqBPList)reso4BWBPList.Clone();
            }
            if ( reso1AmpBPList != null ) {
                res.reso1AmpBPList = (VsqBPList)reso1AmpBPList.Clone();
            }
            if ( reso2AmpBPList != null ) {
                res.reso2AmpBPList = (VsqBPList)reso2AmpBPList.Clone();
            }
            if ( reso3AmpBPList != null ) {
                res.reso3AmpBPList = (VsqBPList)reso3AmpBPList.Clone();
            }
            if ( reso4AmpBPList != null ) {
                res.reso4AmpBPList = (VsqBPList)reso4AmpBPList.Clone();
            }
            if ( genderFactorBPList != null ) {
                res.genderFactorBPList = (VsqBPList)genderFactorBPList.Clone();
            }
            if ( portamentoTimingBPList != null ) {
                res.portamentoTimingBPList = (VsqBPList)portamentoTimingBPList.Clone();
            }
            if ( openingBPList != null ) {
                res.openingBPList = (VsqBPList)openingBPList.Clone();
            }
            return res;
        }

        public VsqEventList Events{
            get {
                return m_events;
            }
        }

        internal VsqBPList this[string curve] {
            get {
                switch ( curve.Trim().ToLower() ) {
                    case "bre":
                        return this.BRE;
                    case "bri":
                        return this.BRI;
                    case "cle":
                        return this.CLE;
                    case "dyn":
                        return this.DYN;
                    case "gen":
                        return this.GEN;
                    case "ope":
                        return this.OPE;
                    case "pbs":
                        return this.PBS;
                    case "pit":
                        return this.PIT;
                    case "por":
                        return this.POR;
                    default:
                        return null;
                }
            }
            set {
                switch ( curve.Trim().ToLower() ) {
                    case "bre":
                        this.BRE = value;
                        break;
                    case "bri":
                        this.BRI = value;
                        break;
                    case "cle":
                        this.CLE = value;
                        break;
                    case "dyn":
                        this.DYN = value;
                        break;
                    case "gen":
                        this.GEN = value;
                        break;
                    case "ope":
                        this.OPE = value;
                        break;
                    case "pbs":
                        this.PBS = value;
                        break;
                    case "pit":
                        this.PIT = value;
                        break;
                    case "por":
                        this.POR = value;
                        break;
                }
            }
        }

        /// <summary>
        /// Editor画面上で上からindex番目のカーブを表すBPListを求めます
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public VsqBPList GetCurve( int index ) {
            switch ( index ) {
                case 1:
                    return DYN;
                case 2:
                    return BRE;
                case 3:
                    return BRI;
                case 4:
                    return CLE;
                case 5:
                    return OPE;
                case 6:
                    return GEN;
                case 7:
                    return POR;
                case 8:
                    return PIT;
                case 9:
                    return PBS;
                default:
                    return null;
            }
        }


        /// <summary>
        /// Editor画面上で上からindex番目のカーブの名前を調べます
        /// </summary>
        /// <param name="index"></param>
        /// <returns></returns>
        public static string GetCurveName( int index ) {
            switch ( index ) {
                case 0:
                    return "VEL";
                case 1:
                    return "DYN";
                case 2:
                    return "BRE";
                case 3:
                    return "BRI";
                case 4:
                    return "CLE";
                case 5:
                    return "OPE";
                case 6:
                    return "GEN";
                case 7:
                    return "POR";
                case 8:
                    return "PIT";
                case 9:
                    return "PBS";
                default:
                    return "";
            }
        }

        /// <summary>
        /// pitchBendBPListへのエイリアス
        /// </summary>
        internal VsqBPList PIT {
            get {
                return pitchBendBPList;
            }
            set {
                pitchBendBPList = value;
            }
        }

        /// <summary>
        /// pitchBendSensBPListへのエイリアス
        /// </summary>
        public VsqBPList PBS {
            get {
                return pitchBendSensBPList;
            }
            set {
                pitchBendSensBPList = value;
            }
        }

        /// <summary>
        /// dynamicsBPListへのエイリアス
        /// </summary>
        public VsqBPList DYN {
            get {
                return dynamicsBPList;
            }
            set {
                dynamicsBPList = value;
            }
        }

        /// <summary>
        /// epRResidualBPListへのエイリアス
        /// </summary>
        public VsqBPList BRE {
            get {
                return epRResidualBPList;
            }
            set {
                epRResidualBPList = value;
            }
        }

        /// <summary>
        /// epRESlopeBPListへのエイリアス
        /// </summary>
        public VsqBPList BRI {
            get {
                return epRESlopeBPList;
            }
            set {
                epRESlopeBPList = value;
            }
        }

        /// <summary>
        /// epRSlopeDepthBPListへのエイリアス
        /// </summary>
        public VsqBPList CLE {
            get {
                return epRESlopeDepthBPList;
            }
            set {
                epRESlopeDepthBPList = value;
            }
        }

        /// <summary>
        /// genderFactorBPListへのエイリアス
        /// </summary>
        public VsqBPList GEN {
            get {
                return genderFactorBPList;
            }
            set {
                genderFactorBPList = value;
            }
        }

        /// <summary>
        /// portamentTimingBPListへのエイリアス
        /// </summary>
        public VsqBPList POR {
            get {
                return portamentoTimingBPList;
            }
            set {
                portamentoTimingBPList = value;
            }
        }

        /// <summary>
        /// openingBPListへのエイリアス
        /// </summary>
        public VsqBPList OPE {
            get {
                return openingBPList;
            }
            set {
                openingBPList = value;
            }
        }

        /// <summary>
        /// Singerプロパティに指定されている
        /// </summary>
        public string Singer {
            get {
                foreach ( VsqEvent item in m_events ) {
                    if ( item.ID.type == VsqIDType.Singer ) {
                        return item.ID.IconHandle.IDS;
                    }
                }
                return "";
            }
            set {
                foreach ( VsqEvent item in m_events ) {
                    if ( item.ID.type == VsqIDType.Singer ) {
                        item.ID.IconHandle.IDS = value;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// EOSイベントが記録されているクロックを取得します。
        /// </summary>
        /// <returns></returns>
        public int GetIndexOfEOS() {
            int result;
            if ( m_events.Count > 0 ) {
                int ilast = m_events.Count - 1;
                result = m_events[ilast].Clock;
            } else {
                result = -1;
            }
            return result;
        }

        /// <summary>
        /// このインスタンスから、IDとHandleのリストを構築します
        /// </summary>
        /// <param name="id"></param>
        /// <param name="handle"></param>
        void BuildIDHandleList( out List<VsqID> id, out List<VsqHandle> handle ) {
            id = new List<VsqID>();
            handle = new List<VsqHandle>();
            int current_id = -1;
            int current_handle = -1;
            List<VsqEvent> events = new List<VsqEvent>();
            foreach ( VsqEvent item in m_events ) {
                events.Add( item );
            }
            events.Sort();
            foreach( VsqEvent item in events ){
                VsqID id_item = (VsqID)item.ID.Clone();
                current_id++;
                item.ID.value = current_id;
                id_item.value = current_id;
                // IconHandle
                if ( item.ID.IconHandle != null ) {
                    current_handle++;
                    VsqHandle handle_item = (VsqHandle)item.ID.IconHandle.Clone();
                    handle_item.Index = current_handle;
                    handle.Add( handle_item );
                    id_item.IconHandle_index = current_handle;
                }
                // LyricHandle
                if ( item.ID.LyricHandle != null ) {
                    current_handle++;
                    VsqHandle handle_item = (VsqHandle)item.ID.LyricHandle.Clone();
                    handle_item.Index = current_handle;
                    handle.Add( handle_item );
                    id_item.LyricHandle_index = current_handle;
                }
                // VibratoHandle
                if ( item.ID.VibratoHandle != null ) {
                    current_handle++;
                    VsqHandle handle_item = (VsqHandle)item.ID.VibratoHandle.Clone();
                    handle_item.Index = current_handle;
                    handle.Add( handle_item );
                    id_item.VibratoHandle_index = current_handle;
                }
                id.Add( id_item );
            }
        }

        /// <summary>
        /// このインスタンスの内容を指定されたファイルに出力します。
        /// </summary>
        /// <param name="sw"></param>
        /// <param name="encode"></param>
        public void Print( TextMemoryStream sw, bool encode, int eos, int start ) {
            if ( common != null ) {
                common.write( sw );
            }
            if ( master != null ) {
                master.write( sw );
            }
            if ( mixer != null ) {
                mixer.write( sw );
            }
            List<VsqID> id;
            List<VsqHandle> handle;
            BuildIDHandleList( out id, out handle );
            writeEventList( sw, eos );
            int i;
            for ( i = 0; i < id.Count; i++ ) {
                id[i].write( sw );
            }
            for ( i = 0; i < handle.Count; i++ ) {
                handle[i].write( sw, encode );
            }
            if ( pitchBendBPList.Count > 0 ) {
                pitchBendBPList.Print( sw, start, "[PitchBendBPList]" );
            }
            if ( pitchBendSensBPList.Count > 0 ) {
                pitchBendSensBPList.Print( sw, start, "[PitchBendSensBPList]" );
            }
            if ( dynamicsBPList.Count > 0 ) {
                dynamicsBPList.Print( sw, start, "[DynamicsBPList]" );
            }
            if ( epRResidualBPList.Count > 0 ) {
                epRResidualBPList.Print( sw, start, "[EpRResidualBPList]" );
            }
            if ( epRESlopeBPList.Count > 0 ) {
                epRESlopeBPList.Print( sw, start, "[EpRESlopeBPList]" );
            }
            if ( epRESlopeDepthBPList.Count > 0 ) {
                epRESlopeDepthBPList.Print( sw, start, "[EpRESlopeDepthBPList]" );
            }

            if ( reso1FreqBPList.Count > 0 ) {
                reso1FreqBPList.Print( sw, start, "[Reso1FreqBPList]" );
            }
            if ( reso2FreqBPList.Count > 0 ) {
                reso2FreqBPList.Print( sw, start, "[Reso2FreqBPList]" );
            }
            if ( reso3FreqBPList.Count > 0 ) {
                reso3FreqBPList.Print( sw, start, "[Reso3FreqBPList]" );
            }
            if ( reso4FreqBPList.Count > 0 ) {
                reso4FreqBPList.Print( sw, start, "[Reso4FreqBPList]" );
            }

            if ( reso1BWBPList.Count > 0 ) {
                reso1BWBPList.Print( sw, start, "[Reso1BWBPList]" );
            }
            if ( reso2BWBPList.Count > 0 ) {
                reso2BWBPList.Print( sw, start, "[Reso2BWBPList]" );
            }
            if ( reso3BWBPList.Count > 0 ) {
                reso3BWBPList.Print( sw, start, "[Reso3BWBPList]" );
            }
            if ( reso4BWBPList.Count > 0 ) {
                reso4BWBPList.Print( sw, start, "[Reso4BWBPList]" );
            }

            if ( reso1AmpBPList.Count > 0 ) {
                reso1AmpBPList.Print( sw, start, "[Reso1AmpBPList]" );
            }
            if ( reso2AmpBPList.Count > 0 ) {
                reso2AmpBPList.Print( sw, start, "[Reso2AmpBPList]" );
            }
            if ( reso3AmpBPList.Count > 0 ) {
                reso3AmpBPList.Print( sw, start, "[Reso3AmpBPList]" );
            }
            if ( reso4AmpBPList.Count > 0 ) {
                reso4AmpBPList.Print( sw, start, "[Reso4AmpBPList]" );
            }

            if ( genderFactorBPList.Count > 0 ) {
                genderFactorBPList.Print( sw, start, "[GenderFactorBPList]" );
            }
            if ( portamentoTimingBPList.Count > 0 ) {
                portamentoTimingBPList.Print( sw, start, "[PortamentoTimingBPList]" );
            }
            if ( openingBPList.Count > 0 ) {
                openingBPList.Print( sw, start, "[OpeningBPList]" );
            }
        }

        private void writeEventList( TextMemoryStream sw, int eos ) {
            sw.WriteLine( "[EventList]" );
            List<VsqEvent> temp = new List<VsqEvent>();
            foreach ( VsqEvent item in m_events ) {
                temp.Add( item );
            }
            temp.Sort();
            int i = 0;
            while ( i < temp.Count ) {
                VsqEvent item = temp[i];
                if ( !item.ID.Equals( VsqID.EOS ) ) {
                    string ids = "ID#" + i.ToString( "0000" );
                    int clock = temp[i].Clock;
                    while ( i + 1 < temp.Count && clock == temp[i + 1].Clock ) {
                        i++;
                        ids += ",ID#" + i.ToString( "0000" );
                    }
                    sw.WriteLine( clock + "=" + ids );
                }
                i++;
            }
            sw.WriteLine( eos + "=EOS" );
        }

        /// <summary>
        /// 何も無いVsqMetaTextを構築する。これは、Master Track用のMetaTextとしてのみ使用されるべき
        /// </summary>
        public VsqMetaText() {
        }

        /// <summary>
        /// 最初のトラック以外の一般のメタテキストを構築。(Masterが作られない)
        /// </summary>
        public VsqMetaText( string name, string singer )
            : this( name, 0, singer, false ) {
        }

        /// <summary>
        /// 最初のトラックのメタテキストを構築。(Masterが作られる)
        /// </summary>
        /// <param name="pre_measure"></param>
        public VsqMetaText( string name, string singer, int pre_measure )
            : this( name, pre_measure, singer, true ) {
        }

        private VsqMetaText( string name, int pre_measure, string singer, bool is_first_track ) {
            common = new VsqCommon( name, Color.FromArgb( 179, 181, 123 ), 1, 1 );
            pitchBendBPList = new VsqBPList( 0, -8192, 8192 );
            pitchBendBPList.Add( 0, pitchBendBPList.Default );

            pitchBendSensBPList = new VsqBPList( 2, 0, 24 );
            pitchBendSensBPList.Add( 0, pitchBendSensBPList.Default );

            dynamicsBPList = new VsqBPList( 64, 0, 127 );
            dynamicsBPList.Add( 0, dynamicsBPList.Default );

            epRResidualBPList = new VsqBPList( 0, 0, 127 );
            epRResidualBPList.Add( 0, epRResidualBPList.Default );

            epRESlopeBPList = new VsqBPList( 64, 0, 127 );
            epRESlopeBPList.Add( 0, epRESlopeBPList.Default );

            epRESlopeDepthBPList = new VsqBPList( 0, 0, 127 );
            epRESlopeDepthBPList.Add( 0, epRESlopeDepthBPList.Default );

            reso1FreqBPList = new VsqBPList( 255, 0, 255 );
            reso1FreqBPList.Add( 0, reso1FreqBPList.Default );

            reso2FreqBPList = new VsqBPList( 255, 0, 255 );
            reso2FreqBPList.Add( 0, reso2FreqBPList.Default );

            reso3FreqBPList = new VsqBPList( 255, 0, 255 );
            reso3FreqBPList.Add( 0, reso3FreqBPList.Default );

            reso4FreqBPList = new VsqBPList( 255, 0, 255 );
            reso4FreqBPList.Add( 0, reso4FreqBPList.Default );

            reso1BWBPList = new VsqBPList( 255, 0, 255 );
            reso1BWBPList.Add( 0, reso1BWBPList.Default );

            reso2BWBPList = new VsqBPList( 255, 0, 255 );
            reso2BWBPList.Add( 0, reso2BWBPList.Default );

            reso3BWBPList = new VsqBPList( 255, 0, 255 );
            reso3BWBPList.Add( 0, reso3BWBPList.Default );

            reso4BWBPList = new VsqBPList( 255, 0, 255 );
            reso4BWBPList.Add( 0, reso4BWBPList.Default );

            reso1AmpBPList = new VsqBPList( 255, 0, 255 );
            reso1AmpBPList.Add( 0, reso1AmpBPList.Default );

            reso2AmpBPList = new VsqBPList( 255, 0, 255 );
            reso2AmpBPList.Add( 0, reso2AmpBPList.Default );

            reso3AmpBPList = new VsqBPList( 255, 0, 255 );
            reso3AmpBPList.Add( 0, reso3AmpBPList.Default );

            reso4AmpBPList = new VsqBPList( 255, 0, 255 );
            reso4AmpBPList.Add( 0, reso4AmpBPList.Default );

            genderFactorBPList = new VsqBPList( 64, 0, 127 );
            genderFactorBPList.Add( 0, genderFactorBPList.Default );

            portamentoTimingBPList = new VsqBPList( 64, 0, 127 );
            portamentoTimingBPList.Add( 0, portamentoTimingBPList.Default );

            openingBPList = new VsqBPList( 127, 0, 127 );
            openingBPList.Add( 0, openingBPList.Default );

            if ( is_first_track ) {
                master = new VsqMaster( pre_measure );
            } else {
                master = null;
            }
            m_events = new VsqEventList();
            VsqID id = new VsqID( 0 );
            id.type = VsqIDType.Singer;
            id.IconHandle = new IconHandle();
            id.IconHandle.Type = VsqHandleType.Singer;
            id.IconHandle.IconID = "$07010000";
            id.IconHandle.IDS = singer;
            id.IconHandle.Original = 0;
            id.IconHandle.Caption = "";
            id.IconHandle.Length = 1;
            id.IconHandle.Language = 0;
            id.IconHandle.Program = 0;
            m_events.Add( new VsqEvent( 0, id ) );
        }
        
        public VsqMetaText( TextMemoryStream sr ) {
            List<KeyValuePair<int, int>> t_event_list = new List<KeyValuePair<int, int>>();
            Dictionary<int, VsqID> __id = new Dictionary<int, VsqID>();
            Dictionary<int, VsqHandle> __handle = new Dictionary<int, VsqHandle>();
            pitchBendBPList = new VsqBPList( 0, -8192, 8192 );
            pitchBendSensBPList = new VsqBPList( 2, 0, 24 );
            dynamicsBPList = new VsqBPList( 64, 0, 127 );
            epRResidualBPList = new VsqBPList( 0 , 0, 127);
            epRESlopeBPList = new VsqBPList( 64, 0, 127 );
            epRESlopeDepthBPList = new VsqBPList( 0, 0, 127 );
            reso1FreqBPList = new VsqBPList( 255, 0, 255 );
            reso2FreqBPList = new VsqBPList( 255, 0, 255 );
            reso3FreqBPList = new VsqBPList( 255, 0, 255 );
            reso4FreqBPList = new VsqBPList( 255, 0, 255 );
            reso1BWBPList = new VsqBPList( 255, 0, 255 );
            reso2BWBPList = new VsqBPList( 255, 0, 255 );
            reso3BWBPList = new VsqBPList( 255, 0, 255 );
            reso4BWBPList = new VsqBPList( 255, 0, 255 );
            reso1AmpBPList = new VsqBPList( 255, 0, 255 );
            reso2AmpBPList = new VsqBPList( 255, 0, 255 );
            reso3AmpBPList = new VsqBPList( 255, 0, 255 );
            reso4AmpBPList = new VsqBPList( 255, 0, 255 );
            genderFactorBPList = new VsqBPList( 64, 0, 127 );
            portamentoTimingBPList = new VsqBPList( 64, 0, 127 );
            openingBPList = new VsqBPList( 127, 0, 127 );

            string last_line = sr.ReadLine();
            while ( true ) {
                #region "TextMemoryStreamから順次読込み"
                if ( last_line.Length == 0 ) {
                    break;
                }
                switch ( last_line ) {
                    case "[Common]":
                        common = new VsqCommon( sr, ref last_line );
                        break;
                    case "[Master]":
                        master = new VsqMaster( sr, ref last_line );
                        break;
                    case "[Mixer]":
                        mixer = new VsqMixer( sr, ref last_line );
                        break;
                    case "[EventList]":
                        last_line = sr.ReadLine();
                        while ( !last_line.StartsWith( "[" ) ) {
                            string[] spl2 = last_line.Split( new char[] { '=' } );
                            int clock = int.Parse( spl2[0] );
                            int id_number = -1;
                            if ( spl2[1] != "EOS" ) {
                                string[] ids = spl2[1].Split( ",".ToCharArray() );
                                for ( int i = 0; i < ids.Length; i++ ) {
                                    string[] spl3 = ids[i].Split( new char[] { '#' } );
                                    id_number = int.Parse( spl3[1] );
                                    t_event_list.Add( new KeyValuePair<int,int>( clock, id_number ) );
                                }
                            } else {
                                t_event_list.Add( new KeyValuePair<int,int>( clock, -1) );
                            }
                            if ( sr.Peek() < 0 ) {
                                break;
                            } else {
                                last_line = sr.ReadLine();
                            }
                        }
                        break;
                    case "[PitchBendBPList]":
                        last_line = pitchBendBPList.Read( sr );
                        break;
                    case "[PitchBendSensBPList]":
                        last_line = pitchBendSensBPList.Read( sr );
                        break;
                    case "[DynamicsBPList]":
                        last_line = dynamicsBPList.Read( sr );
                        break;
                    case "[EpRResidualBPList]":
                        last_line = epRResidualBPList.Read( sr );
                        break;
                    case "[EpRESlopeBPList]":
                        last_line = epRESlopeBPList.Read( sr );
                        break;
                    case "[EpRESlopeDepthBPList]":
                        last_line = epRESlopeDepthBPList.Read( sr );
                        break;
                    case "[Reso1FreqBPList]":
                        last_line = reso1FreqBPList.Read( sr );
                        break;
                    case "[Reso2FreqBPList]":
                        last_line = reso2FreqBPList.Read( sr );
                        break;
                    case "[Reso3FreqBPList]":
                        last_line = reso3FreqBPList.Read( sr );
                        break;
                    case "[Reso4FreqBPList]":
                        last_line = reso4FreqBPList.Read( sr );
                        break;
                    case "[Reso1BWBPList]":
                        last_line = reso1BWBPList.Read( sr );
                        break;
                    case "[Reso2BWBPList]":
                        last_line = reso2BWBPList.Read( sr );
                        break;
                    case "[Reso3BWBPList]":
                        last_line = reso3BWBPList.Read( sr );
                        break;
                    case "[Reso4BWBPList]":
                        last_line = reso4BWBPList.Read( sr );
                        break;
                    case "[Reso1AmpBPList]":
                        last_line = reso1AmpBPList.Read( sr );
                        break;
                    case "[Reso2AmpBPList]":
                        last_line = reso2AmpBPList.Read( sr );
                        break;
                    case "[Reso3AmpBPList]":
                        last_line = reso3AmpBPList.Read( sr );
                        break;
                    case "[Reso4AmpBPList]":
                        last_line = reso4AmpBPList.Read( sr );
                        break;
                    case "[GenderFactorBPList]":
                        last_line = genderFactorBPList.Read( sr );
                        break;
                    case "[PortamentoTimingBPList]":
                        last_line = portamentoTimingBPList.Read( sr );
                        break;
                    case "[OpeningBPList]":
                        last_line = openingBPList.Read( sr );
                        break;
                    default:
                        string buffer = last_line;
                        buffer = buffer.Replace( "[", "" );
                        buffer = buffer.Replace( "]", "" );
                        string[] spl = buffer.Split( new char[] { '#' } );
                        int index = int.Parse( spl[1] );
                        if ( last_line.StartsWith( "[ID#" ) ) {
                            __id.Add( index, new VsqID( sr, index, ref last_line ) );
                        } else if ( last_line.StartsWith( "[h#" ) ) {
                            __handle.Add( index, new VsqHandle( sr, index, ref last_line ) );
                        }
                        break;
                #endregion
                }

                if ( sr.Peek() < 0 ) {
                    break;
                }
            }

            // まずhandleをidに埋め込み
            for ( int i = 0; i < __id.Count; i++ ) {
                if ( __handle.ContainsKey( __id[i].IconHandle_index ) ) {
                    __id[i].IconHandle = __handle[__id[i].IconHandle_index].ConvertToIconHandle();
                }
                if ( __handle.ContainsKey( __id[i].LyricHandle_index ) ) {
                    __id[i].LyricHandle = __handle[__id[i].LyricHandle_index].ConvertToLyricHandle();
                }
                if ( __handle.ContainsKey( __id[i].VibratoHandle_index ) ) {
                    __id[i].VibratoHandle = __handle[__id[i].VibratoHandle_index].ConvertToVibratoHandle();
                }
            }

            // idをeventListに埋め込み
            m_events = new VsqEventList();
            for ( int i = 0; i < t_event_list.Count; i++ ) {
                int clock = t_event_list[i].Key;
                int id_number = t_event_list[i].Value;
                if ( __id.ContainsKey( id_number ) ) {
                    m_events.Add( new VsqEvent( clock, (VsqID)__id[id_number].Clone() ) );
                }
            }
        }


        public static bool test( string fpath ) {
            VsqMetaText metaText;
            using ( TextMemoryStream sr = new TextMemoryStream( fpath, Encoding.Unicode ) ) {
                metaText = new VsqMetaText( sr );
            }
            string result = "test.txt";


            StreamReader honmono = new StreamReader( fpath );
            TextMemoryStream copy = new TextMemoryStream( FileAccess.ReadWrite );
            metaText.Print( copy, true, 1000, 100 );
            copy.Rewind();
            while ( honmono.Peek() >= 0 && copy.Peek() >= 0 ) {
                string hon = honmono.ReadLine();
                string cop = copy.ReadLine();
                if ( hon != cop ) {
                    Console.WriteLine( "honmono,copy=" + hon + "," + cop );
                    honmono.Close();
                    copy.Close();
                    return false;
                }
            }
            honmono.Close();
            copy.Close();

            return true;
        }
    }

    public enum VsqIDType {
        Singer,
        Anote,
        Unknown
    }

    public enum VsqHandleType {
        Lyric,
        Vibrato,
        Singer
    }

}
