﻿/*
 * VSTiProxy.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
//#define TEST
using System;
using System.IO;
using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.Threading;
using System.ComponentModel;
using System.Security;
using System.Windows.Forms;

using Boare.Lib.Vsq;
using Boare.Lib.Media;

namespace Boare.Cadencii {

    public delegate void WaveIncomingEventHandler( double[] L, double[] R );
    public delegate void RenderingFinishedEventHandler();

    public static class VSTiProxyTrackType {
        public const int TempoTrack = 0;
        public const int MainTrack = 1;
    }

    public static class VSTiProxy {
        public const string RENDERER_DSB2 = "DSB2";
        public const string RENDERER_DSB3 = "DSB3";
        public const string RENDERER_UTU0 = "UTU0";
        private const int _SAMPLE_RATE = 44100;
        private const int _BLOCK_SIZE = 44100;

        public static string CurrentUser = "";
        private static WaveWriter s_wave;
        private static bool s_rendering = false;
        private static int s_trim_remain = 0;
        private static object s_locker;
        private static double s_amplify_left = 1.0;
        private static double s_amplify_right = 1.0;
        private static string s_working_renderer = "";
        private static List<VstiRenderer> m_vstidrv = new List<VstiRenderer>();
        private static bool s_direct_play;
        private static List<WaveReader> s_reader = new List<WaveReader>();
        private static long s_total_append = 0;
        private static double s_wave_read_offset_seconds = 0.0;

        //public static event RenderingFinishedEventHandler RenderingFinished;

        private class StartRenderArg {
            public string renderer;
            public NrpnData[] nrpn;
            public TempoTableEntry[] tempo;
            public double amplify_left;
            public double amplify_right;
            public int trim_msec;
            public long total_samples;
            public string[] files;
            public double wave_read_offset_seconds;
            public bool mode_infinite;
        }

        private class StartUtauRenderArg {
            public VsqFileEx vsq;
            public int track;
            public List<SingerConfig> utau_singers;
            public string path_resampler;
            public string path_wavtool;
            public string path_temp;
            public bool invoke_with_wine;
            public int sample_rate;
            public int trim_msec;
            public bool mode_infinite;
        }

        private class VstiRenderer {
#if UNMANAGED_VSTIDRV
            public vstildr DllInstance = null;
#else
            public vstidrv DllInstance = null;
#endif
            public bool Loaded = false;
            public string Path = "";
            public string Name = "";
            //public volatile Boare.Lib.Media.FirstBufferWrittenCallback FirstBufferWrittenEventHandler = null;
        }

        static VSTiProxy() {
#if DEBUG
            Common.DebugWriteLine( "VSTiProxy..cctor" );
#endif
            AppManager.LoadConfig();
            s_locker = new object();
            PlaySound.Init( _BLOCK_SIZE, _SAMPLE_RATE );
#if !DEBUG
            try {
#endif
#if UNMANAGED_VSTIDRV
                string driver = Path.Combine( Application.StartupPath, "vstidrv.dll" );
#if DEBUG
                Common.DebugWriteLine( "driver=" + driver );
#endif
                System.Reflection.Assembly asm = System.Reflection.Assembly.LoadFile( driver );
#if DEBUG
                Common.DebugWriteLine( "asm.FullName=" + asm.FullName );
#endif
                Type[] types = null;
                try {
                    types = asm.GetTypes();
                } catch ( System.Reflection.ReflectionTypeLoadException rte ) {
#if DEBUG
                    foreach ( Exception ex in rte.LoaderExceptions ) {
                        bocoree.debug.push_log( "ex=" + ex );
                    }
#endif
                    return;
                }
                string driver_fullname = "";
                foreach ( Type t in types ) {
                    if ( t.IsClass && t.IsPublic && !t.IsAbstract ) {
                        try {
                            Type intfc = t.GetInterface( typeof( vstildr ).FullName );
                            if ( intfc != null ) {
                                driver_fullname = t.FullName;
                                break;
                            }
                        } catch {
                        }
                    }
                }
                if ( driver_fullname == "" ) {
                    return;
                }

                string vocalo2_dll_path = VocaloSysUtil.getDllPathVsti2();
                if ( vocalo2_dll_path.ToLower().EndsWith( "_demo.dll" ) ) {
                    vocalo2_dll_path = "";
                }
                string vocalo1_dll_path = VocaloSysUtil.getDllPathVsti1();
                if ( vocalo2_dll_path != "" && File.Exists( vocalo2_dll_path ) ) {
                    VstiRenderer vr = new VstiRenderer();
                    vr.Path = vocalo2_dll_path;
                    vr.Loaded = false;
                    vr.DllInstance = (vstildr)asm.CreateInstance( driver_fullname );
                    vr.Name = RENDERER_DSB3;
                    m_vstidrv.Add( vr );
                }
                if ( vocalo1_dll_path != "" && File.Exists( vocalo1_dll_path ) ) {
                    VstiRenderer vr = new VstiRenderer();
                    vr.Path = vocalo1_dll_path;
                    vr.Loaded = false;
                    vr.DllInstance = (vstildr)asm.CreateInstance( driver_fullname );
                    vr.Name = RENDERER_DSB2;
                    m_vstidrv.Add( vr );
                }
#else
                string vocalo2_dll_path = VocaloSysUtil.getDllPathVsti2();
                string vocalo1_dll_path = VocaloSysUtil.getDllPathVsti1();
                if ( vocalo2_dll_path != "" && File.Exists( vocalo2_dll_path ) ) {
                    VstiRenderer vr = new VstiRenderer();
                    vr.Path = vocalo2_dll_path;
                    vr.Loaded = false;
                    vr.DllInstance = new vstidrv();
                    vr.Name = RENDERER_DSB3;
                    m_vstidrv.Add( vr );
                }
                if ( vocalo1_dll_path != "" && File.Exists( vocalo1_dll_path ) ) {
                    VstiRenderer vr = new VstiRenderer();
                    vr.Path = vocalo1_dll_path;
                    vr.Loaded = false;
                    vr.DllInstance = new vstidrv();
                    vr.Name = RENDERER_DSB2;
                    m_vstidrv.Add( vr );
                }
#endif
#if !DEBUG
            } catch ( Exception ex ){
                Common.DebugWriteLine( "    ex=" + ex );
                bocoree.debug.push_log( "    ex=" + ex );
            }
#endif

#if TEST
                bocoree.debug.push_log( "vstidrv.Count=" + m_vstidrv.Count );
#endif
            for ( int i = 0; i < m_vstidrv.Count; i++ ) {
#if TEST
                bocoree.debug.push_log( "Name=" + m_vstidrv[i].Name + "; Path=" + m_vstidrv[i].Path );
#endif
                string dll_path = m_vstidrv[i].Path;
                bool loaded = false;
                try {
                    char[] str = dll_path.ToCharArray();
                    if ( dll_path != "" ) {
                        loaded = m_vstidrv[i].DllInstance.Init( str, _BLOCK_SIZE, _SAMPLE_RATE );
                        //m_vstidrv[i].FirstBufferWrittenEventHandler = new FirstBufferWrittenCallback( HandleFirstBufferWrittenEvent );
                        //m_vstidrv[i].DllInstance.SetFirstBufferWrittenCallback( m_vstidrv[i].FirstBufferWrittenEventHandler );
                    } else {
                        loaded = false;
                    }
                    m_vstidrv[i].Loaded = loaded;
#if DEBUG
#if TEST
                    bocoree.debug.push_log( "VSTiProxy..cctor()" );
                    bocoree.debug.push_log( "    dll_path=" + dll_path );
                    bocoree.debug.push_log( "    loaded=" + loaded );

#endif
#endif
                } catch ( Exception ex ) {
#if TEST
                    bocoree.debug.push_log( "    ex=" + ex );
#endif
                }
            }
        }

        public static bool IsRendererAvailable( string renderer ) {
            for ( int i = 0; i < m_vstidrv.Count; i++ ) {
                if ( renderer.StartsWith( m_vstidrv[i].Name ) && m_vstidrv[i].Loaded ) {
                    return true;
                }
            }
            if ( renderer.StartsWith( RENDERER_UTU0 ) ) {
                if ( AppManager.EditorConfig.PathResampler != "" && File.Exists( AppManager.EditorConfig.PathResampler ) &&
                     AppManager.EditorConfig.PathWavtool != "" && File.Exists( AppManager.EditorConfig.PathWavtool ) ) {
                    return true;
                }
            }
            return false;
        }

        public static int SampleRate {
            get {
                return _SAMPLE_RATE;
            }
        }

        public static void Terminate() {
            for ( int i = 0; i < m_vstidrv.Count; i++ ) {
                if ( m_vstidrv[i].DllInstance != null ) {
                    m_vstidrv[i].DllInstance.Terminate();
                }
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="vsq">レンダリング元</param>
        /// <param name="track">レンダリング対象のトラック番号</param>
        /// <param name="file">レンダリング結果を出力するファイル名。空文字ならファイルには出力されない</param>
        /// <param name="start_sec">レンダリングの開始位置</param>
        /// <param name="end_sec">レンダリングの終了位置</param>
        /// <param name="amplify_left">左チャンネルの増幅率</param>
        /// <param name="amplify_right">右チャンネルの増幅率</param>
        /// <param name="ms_presend">プリセンドするミリセカンド</param>
        /// <param name="direct_play">レンダリングと同時に再生するかどうか</param>
        /// <param name="files">レンダリング結果と同時に再生するWAVEファイルのリスト</param>
        /// <param name="wave_read_offset_seconds">filesに指定したファイルの先頭から読み飛ばす秒数</param>
        public static void Render(
            VsqFileEx vsq,
            int track,
            WaveWriter wave_writer,
            double start_sec,
            double end_sec,
            double amplify_left,
            double amplify_right,
            int ms_presend,
            bool direct_play,
            string[] files,
            double wave_read_offset_seconds,
            bool mode_infinite,
            string temp_dir
        ) {
#if TEST
            Common.DebugWriteLine( "VSTiProxy+Render" );
            Common.DebugWriteLine( "    start_sec,end_sec=" + start_sec + "," + end_sec );
#endif
            s_working_renderer = VSTiProxy.RENDERER_DSB3;
            s_direct_play = direct_play;
            if ( direct_play ) {
                PlaySound.Reset();
            }
            s_wave = wave_writer;
            s_amplify_left = amplify_left;
            s_amplify_right = amplify_right;
            for ( int i = 0; i < s_reader.Count; i++ ) {
                s_reader[i].Close();
            }
            s_reader.Clear();
            for ( int i = 0; i < files.Length; i++ ) {
                s_reader.Add( new WaveReader( files[i] ) );
            }
            s_total_append = 0;
            s_wave_read_offset_seconds = wave_read_offset_seconds;

            string version = vsq.Track[track].getCommon().Version;
            if ( version.StartsWith( VSTiProxy.RENDERER_DSB2 ) ) {
                s_working_renderer = VSTiProxy.RENDERER_DSB2;
            } else if ( version.StartsWith( VSTiProxy.RENDERER_UTU0 ) ) {
                s_working_renderer = VSTiProxy.RENDERER_UTU0;
            }
            VsqFileEx split = (VsqFileEx)vsq.Clone();
            split.updateTotalClocks();
            int clock_start = (int)vsq.getClockFromSec( start_sec );
            int clock_end = (int)vsq.getClockFromSec( end_sec );

            if ( clock_end < vsq.TotalClocks ) {
                split.removePart( clock_end, split.TotalClocks + 480 );
            }

            int extra_note_clock = (int)vsq.getClockFromSec( (float)end_sec + 10.0f );
            int extra_note_clock_end = (int)vsq.getClockFromSec( (float)end_sec + 10.0f + 3.1f ); //ブロックサイズが1秒分で、バッファの個数が3だから +3.1f。0.1fは安全のため。
            VsqEvent extra_note = new VsqEvent( extra_note_clock, new VsqID( 0 ) );
            extra_note.ID.type = VsqIDType.Anote;
            extra_note.ID.Note = 60;
            extra_note.ID.Length = extra_note_clock_end - extra_note_clock;
            extra_note.ID.VibratoHandle = null;
            extra_note.ID.LyricHandle = new LyricHandle( "a", "a" );
            split.Track[track].addEvent( extra_note );

            double trim_sec = 0.0; // レンダリング結果から省かなければならない秒数。
            if ( clock_start < split.getPreMeasureClocks() ) {
                trim_sec = split.getSecFromClock( clock_start );
            } else {
                split.removePart( vsq.getPreMeasureClocks(), clock_start );
                trim_sec = split.getSecFromClock( split.getPreMeasureClocks() );
            }
            split.updateTotalClocks();
            split.reflectPitch();
#if TEST
            split.write( "split.vsq" );
#endif
            long total_samples = (long)((end_sec - start_sec) * _SAMPLE_RATE);
            int trim_msec = (int)(trim_sec * 1000.0);
            if ( direct_play ) {
                if ( s_working_renderer == VSTiProxy.RENDERER_UTU0 ) {
                    StartUtauRenderArg sura = new StartUtauRenderArg();
                    sura.vsq = split;
                    sura.track = track;
                    sura.utau_singers = AppManager.EditorConfig.UtauSingers;
                    sura.path_resampler = AppManager.EditorConfig.PathResampler;
                    sura.path_wavtool = AppManager.EditorConfig.PathWavtool;
                    sura.path_temp = temp_dir;
                    sura.invoke_with_wine = AppManager.EditorConfig.InvokeUtauCoreWithWine;
                    sura.sample_rate = _SAMPLE_RATE;
                    sura.trim_msec = trim_msec;
                    sura.mode_infinite = mode_infinite;

                    Thread thread = new Thread( new ParameterizedThreadStart( RenderUtauWithDirectPlay ) );
                    thread.Priority = ThreadPriority.BelowNormal;
                    thread.Start( sura );
                } else {
                    VsqNrpn[] nrpn = VsqFile.generateNRPN( split, track, ms_presend );
                    NrpnData[] nrpn_data = VsqNrpn.convert( nrpn );
                    StartRenderArg sra = new StartRenderArg();
                    sra.renderer = s_working_renderer;
                    sra.nrpn = nrpn_data;
                    sra.tempo = split.getTempoList().ToArray();
                    sra.amplify_left = amplify_left;
                    sra.amplify_right = amplify_right;
                    sra.trim_msec = trim_msec;
                    sra.total_samples = total_samples;
                    sra.files = files;
                    sra.wave_read_offset_seconds = wave_read_offset_seconds;
                    sra.mode_infinite = mode_infinite;

                    Thread thread = new Thread( new ParameterizedThreadStart( RenderWithDirectPlay ) );
                    thread.Priority = ThreadPriority.BelowNormal;
                    thread.Start( sra );
                }
            } else {
                if ( s_working_renderer == VSTiProxy.RENDERER_UTU0 ) {
                    RenderUtau.WaveIncoming += vstidrv_WaveIncoming;
                    RenderUtau.RenderingFinished += vstidrv_RenderingFinished;
                    s_trim_remain = 0;
                    RenderUtau.StartRendering( split,
                                               track,
                                               AppManager.EditorConfig.UtauSingers,
                                               AppManager.EditorConfig.PathResampler,
                                               AppManager.EditorConfig.PathWavtool,
                                               temp_dir,
                                               AppManager.EditorConfig.InvokeUtauCoreWithWine,
                                               _SAMPLE_RATE,
                                               trim_msec,
                                               mode_infinite );
                    RenderUtau.WaveIncoming -= vstidrv_WaveIncoming;
                    RenderUtau.RenderingFinished -= vstidrv_RenderingFinished;
                } else {
                    VsqNrpn[] nrpn = VsqFile.generateNRPN( split, track, ms_presend );
                    NrpnData[] nrpn_data = VsqNrpn.convert( nrpn );
                    RenderCor( s_working_renderer,
                               nrpn_data,
                               split.getTempoList().ToArray(),
                               amplify_left,
                               amplify_right,
                               direct_play,
                               trim_msec,
                               total_samples,
                               files,
                               wave_read_offset_seconds,
                               mode_infinite );
                }
            }
        }

        private static void RenderWithDirectPlay( object argument ) {
            StartRenderArg sra = (StartRenderArg)argument;
            RenderCor( sra.renderer,
                       sra.nrpn,
                       sra.tempo,
                       sra.amplify_left,
                       sra.amplify_right,
                       true,
                       sra.trim_msec,
                       sra.total_samples,
                       sra.files,
                       sra.wave_read_offset_seconds,
                       sra.mode_infinite );
        }

        private static void RenderUtauWithDirectPlay( object argument ) {
            StartUtauRenderArg sura = (StartUtauRenderArg)argument;
            RenderUtau.WaveIncoming += vstidrv_WaveIncoming;
            RenderUtau.RenderingFinished += vstidrv_RenderingFinished;
            RenderUtau.StartRendering( sura.vsq,
                                       sura.track,
                                       sura.utau_singers,
                                       sura.path_resampler,
                                       sura.path_wavtool,
                                       sura.path_temp,
                                       sura.invoke_with_wine,
                                       sura.sample_rate,
                                       sura.trim_msec,
                                       sura.mode_infinite );
            RenderUtau.WaveIncoming -= vstidrv_WaveIncoming;
            RenderUtau.RenderingFinished -= vstidrv_RenderingFinished;
        }

        private static unsafe void RenderCor(
            string renderer,
            NrpnData[] nrpn,
            TempoTableEntry[] tempo,
            double amplify_left,
            double amplify_right,
            bool direct_play,
            int presend_msec,
            long total_samples,
            string[] files,
            double wave_read_offset_seconds,
            bool mode_infinite
        ) {
#if TEST
            Console.WriteLine( "VSTiProxy.RenderToWave( NRPN[], TempoTableEntry[], int, string )" );
            bocoree.debug.push_log( "VSTiPRoxy+RenderToWave(NrpnData[], TempoTableEntry[], int, string)" );
#endif
            s_direct_play = direct_play;
            int index_driver = -1;
            for ( int i = 0; i < m_vstidrv.Count; i++ ) {
                if ( m_vstidrv[i].Name == renderer ) {
                    index_driver = i;
                    break;
                }
            }
            if ( index_driver < 0 ) {
                return;
            }
            if ( !m_vstidrv[index_driver].Loaded ) {
                return;
            }
#if !UNMANAGED_VSTIDRV
            if ( m_vstidrv == null ) {
                return;
            }
#endif
            int tempo_count = tempo.Length;
            float first_tempo = 125.0f;
            if ( tempo.Length > 0 ) {
                first_tempo = (float)(60e6 / (double)tempo[0].Tempo);
            }
            byte[] masterEventsSrc = new byte[tempo_count * 3];
            int[] masterClocksSrc = new int[tempo_count];
            int count = -3;
            for ( int i = 0; i < tempo.Length; i++ ) {
                count += 3;
                masterClocksSrc[i] = tempo[i].Clock;
                byte b0 = (byte)(tempo[i].Tempo >> 16);
                uint u0 = (uint)(tempo[i].Tempo - (b0 << 16));
                byte b1 = (byte)(u0 >> 8);
                byte b2 = (byte)(u0 - (u0 << 8));
#if DEBUG
                //Console.WriteLine( "    b0-b2=0x" + Convert.ToString( b0, 16 ) + ", 0x" + Convert.ToString( b1, 16 ) + ", 0x" + Convert.ToString( b2, 16 ) );
#endif
                masterEventsSrc[count] = b0;
                masterEventsSrc[count + 1] = b1;
                masterEventsSrc[count + 2] = b2;
            }
            m_vstidrv[index_driver].DllInstance.SendEvent( masterEventsSrc, masterClocksSrc, VSTiProxyTrackType.TempoTrack );

            int numEvents = nrpn.Length;
            byte[] bodyEventsSrc = new byte[numEvents * 3];
            int[] bodyClocksSrc = new int[numEvents];
            count = -3;
            int last_clock = 0;
            for ( int i = 0; i < numEvents; i++ ) {
                count += 3;
                bodyEventsSrc[count] = 0xb0;
                bodyEventsSrc[count + 1] = nrpn[i].getParameter();
                bodyEventsSrc[count + 2] = nrpn[i].Value;
                bodyClocksSrc[i] = nrpn[i].getClock();
                last_clock = nrpn[i].getClock();
            }

            int index = tempo_count - 1;
            for ( int i = tempo_count - 1; i >= 0; i-- ) {
                if ( tempo[i].Clock < last_clock ) {
                    index = i;
                    break;
                }
            }
            int last_tempo = tempo[index].Tempo;
            int trim_remain = GetErrorSamples( first_tempo ) + (int)(presend_msec / 1000.0 * _SAMPLE_RATE);
            s_trim_remain = trim_remain;

#if DEBUG
            Common.DebugWriteLine( "    s_trim_remain=" + s_trim_remain );
#endif
            m_vstidrv[index_driver].DllInstance.SendEvent( bodyEventsSrc, bodyClocksSrc, VSTiProxyTrackType.MainTrack );

            s_rendering = true;
            if ( s_wave != null ) {
                if ( s_trim_remain < 0 ) {
                    double[] d = new double[-s_trim_remain];
                    for ( int i = 0; i < -s_trim_remain; i++ ) {
                        d[i] = 0.0;
                    }
                    s_wave.Append( d, d );
                    s_trim_remain = 0;
                }
            }
#if DEBUG
            bocoree.debug.push_log( "    first_tempo=" + first_tempo );
            bocoree.debug.push_log( "    s_trim_remain=" + s_trim_remain );
#endif

            s_amplify_left = amplify_left;
            s_amplify_right = amplify_right;
            //if ( file != null ) {
                m_vstidrv[index_driver].DllInstance.WaveIncoming += vstidrv_WaveIncoming;
            //}
            m_vstidrv[index_driver].DllInstance.RenderingFinished += vstidrv_RenderingFinished;
            m_vstidrv[index_driver].DllInstance.StartRendering( total_samples, mode_infinite );
            while ( s_rendering ) {
                Application.DoEvents();
            }
            s_rendering = false;
            //if ( file != null ) {
                m_vstidrv[index_driver].DllInstance.WaveIncoming -= vstidrv_WaveIncoming;
            //}
            m_vstidrv[index_driver].DllInstance.RenderingFinished -= vstidrv_RenderingFinished;
            if ( s_direct_play ) {
#if DEBUG
                Common.DebugWriteLine( "VSTiProxy.RenderCor; calling PlaySound.WaitForExit..." );
#endif
                PlaySound.WaitForExit();
#if DEBUG
                Common.DebugWriteLine( "...done" );
#endif
            }
            s_wave = null;
            /*if ( RenderingFinished != null ) {
                RenderingFinished();
            }*/
        }

        static void vstidrv_RenderingFinished() {
            s_rendering = false;
            /*if ( RenderingFinished != null ) {
                RenderingFinished();
            }*/
        }

        static unsafe void vstidrv_WaveIncoming( double[] L, double[] R ) {
#if TEST
            bocoree.debug.push_log( "VSTiProxy.vstidrv_WaveIncoming" );
            bocoree.debug.push_log( "    requiring lock of s_locker..." );
#endif
            lock ( s_locker ) {
                if ( s_trim_remain > 0 ) {
                    if ( s_trim_remain >= L.Length ) {
                        s_trim_remain -= L.Length;
                        return;
                    }
                    int actual_append = L.Length - s_trim_remain;
                    double[] dL = new double[actual_append];
                    double[] dR = new double[actual_append];
                    for ( int i = 0; i < actual_append; i++ ) {
                        dL[i] = L[i + s_trim_remain] * s_amplify_left;
                        dR[i] = R[i + s_trim_remain] * s_amplify_right;
                    }
                    if ( s_wave != null ) {
                        s_wave.Append( dL, dR );
                    }
                    long start = s_total_append + (long)(s_wave_read_offset_seconds * _SAMPLE_RATE);
                    for ( int i = 0; i < s_reader.Count; i++ ) {
                        double[] reader_r;
                        double[] reader_l;
                        s_reader[i].Read( start, actual_append, out reader_l, out reader_r );
                        for ( int j = 0; j < actual_append; j++ ) {
                            dL[j] += reader_l[j];
                            dR[j] += reader_r[j];
                        }
                        reader_l = null;
                        reader_r = null;
                    }
                    if ( s_direct_play ) {
                        PlaySound.Append( dL, dR, actual_append );
                    }
                    dL = null;
                    dR = null;
                    s_trim_remain = 0;
                    s_total_append += actual_append;
                } else {
                    int length = L.Length;
                    for ( int i = 0; i < length; i++ ) {
                        L[i] = L[i] * s_amplify_left;
                        R[i] = R[i] * s_amplify_right;
                    }
                    if ( s_wave != null ) {
                        s_wave.Append( L, R );
                    }
                    long start = s_total_append + (long)(s_wave_read_offset_seconds * _SAMPLE_RATE);
                    for ( int i = 0; i < s_reader.Count; i++ ) {
                        double[] reader_r;
                        double[] reader_l;
                        s_reader[i].Read( start, length, out reader_l, out reader_r );
                        for ( int j = 0; j < length; j++ ) {
                            L[j] += reader_l[j];
                            R[j] += reader_r[j];
                        }
                        reader_l = null;
                        reader_r = null;
                    }
                    if ( s_direct_play ) {
                        PlaySound.Append( L, R, L.Length );
                    }
                    s_total_append += length;
                }
            }
#if DEBUG
            bocoree.debug.push_log( "...done(vstidrv_WaveIncoming)" );
#endif
        }

        public static double GetProgress() {
            if ( s_working_renderer == VSTiProxy.RENDERER_UTU0 ) {
                return RenderUtau.GetProgress();
            } else {
                for ( int i = 0; i < m_vstidrv.Count; i++ ) {
                    if ( m_vstidrv[i].Name == s_working_renderer ) {
                        return m_vstidrv[i].DllInstance.GetProgress();
                    }
                }
                return 0.0;
            }
        }

        public static void AbortRendering() {
            if ( s_rendering ) {
                s_rendering = false;
            }
            if ( s_working_renderer == VSTiProxy.RENDERER_UTU0 ) {
                RenderUtau.AbortRendering();
            } else {
                for ( int i = 0; i < m_vstidrv.Count; i++ ) {
                    if ( m_vstidrv[i].Loaded ) {
                        m_vstidrv[i].DllInstance.AbortRendering();
                    }
                }
            }
            for ( int i = 0; i < s_reader.Count; i++ ) {
                s_reader[i].Close();
                s_reader[i] = null;
            }
            s_reader.Clear();
        }

        public static int GetErrorSamples( float tempo ) {
            const float a0 = -17317.563f;
            const float a1 = 86.7312112f;
            const float a2 = -0.237323499f;
            if ( tempo <= 240 ) {
                return 4666;
            } else {
                float x = tempo - 240;
                return (int)((a2 * x + a1) * x + a0);
            }
        }

        public static float GetPlayTime() {
            double pos = PlaySound.GetPosition();
#if DEBUG
            //Common.DebugWriteLine( "VSTiProxy.GetPlayTime; pos=" + pos );
#endif
            return (float)pos;
        }

#if OBSOLUTE
        public delegate void WaveIncomingEventHandler( double[] L, double[] R );
        public delegate void FirstBufferWrittenEventHandler();
        public delegate void RenderingFinishedEventHandler();

        public static unsafe class vstidrv {
            private delegate void __WaveIncomingCallback( double* L, double* R, int length );
            private delegate void __FirstBufferWrittenCallback();
            private delegate void __RenderingFinishedCallback();

            private volatile static __WaveIncomingCallback s_wave_incoming_callback;
            private volatile static __FirstBufferWrittenCallback s_first_buffer_written_callback;
            private volatile static __RenderingFinishedCallback s_rendering_finished_callback;

            private volatile static FirstBufferWrittenEventHandler s_first_buffer_written_callback_body;
            private volatile static WaveIncomingEventHandler s_wave_incoming_callback_body;

            //public static event WaveIncomingEventHandler WaveIncoming;
            public static event RenderingFinishedEventHandler RenderingFinished;

            static vstidrv() {
                s_wave_incoming_callback = new __WaveIncomingCallback( HandleWaveIncomingCallback );
                s_first_buffer_written_callback = new __FirstBufferWrittenCallback( HandleFirstBufferWrittenCallback );
                s_rendering_finished_callback = new __RenderingFinishedCallback( HandleRenderingFinishedCallback );
                try {
                    vstidrv_setFirstBufferWrittenCallback( Marshal.GetFunctionPointerForDelegate( s_first_buffer_written_callback ) );
                    vstidrv_setWaveIncomingCallback( Marshal.GetFunctionPointerForDelegate( s_wave_incoming_callback ) );
                    vstidrv_setRenderingFinishedCallback( Marshal.GetFunctionPointerForDelegate( s_rendering_finished_callback ) );
                } catch ( Exception ex ) {
#if DEBUG
                    Common.DebugWriteLine( "vstidrv.cctor()" );
                    Common.DebugWriteLine( "    ex=" + ex );
#endif
                }
            }

            public static void SetWaveIncomingCallback( WaveIncomingEventHandler handler ) {
                s_wave_incoming_callback_body = handler;
            }

            public static void SetFirstBufferWrittenCallback( FirstBufferWrittenEventHandler handler ) {
                s_first_buffer_written_callback_body = handler;
            }

            public static void Terminate() {
                try {
                    vstidrv_Terminate();
                } catch {
                }
            }

            private static void HandleWaveIncomingCallback( double* L, double* R, int length ) {
#if TEST
                bocoree.debug.push_log( "vstidrv.HandleWaveIncomingCallback" );
                bocoree.debug.push_log( "    length=" + length );
#endif
                if ( s_wave_incoming_callback_body != null ) {
                    double[] ret_l = new double[length];
                    double[] ret_r = new double[length];
                    for ( int i = 0; i < length; i++ ) {
                        ret_l[i] = L[i];
                        ret_r[i] = R[i];
                    }
                    s_wave_incoming_callback_body( ret_l, ret_r );
                }
            }

            private static void HandleFirstBufferWrittenCallback() {
#if DEBUG
                Common.DebugWriteLine( "vstidrv+HandleFirstBufferWrittenCallback" );
#endif
                if ( s_first_buffer_written_callback_body != null ) {
                    s_first_buffer_written_callback_body();
                }
            }

            private static void HandleRenderingFinishedCallback() {
                if ( RenderingFinished != null ) {
                    RenderingFinished();
                }
            }

            public static bool Init( char[] dll_path, int block_size, int sample_rate ) {
#if DEBUG
                Common.DebugWriteLine( "VSTIProxy.Init" );
#endif
                bool ret = false;
                try {
                    byte[] b_dll_path = new byte[dll_path.Length + 1];
                    for ( int i = 0; i < dll_path.Length; i++ ) {
                        b_dll_path[i] = (byte)dll_path[i];
                    }
                    b_dll_path[dll_path.Length] = 0x0;
                    fixed ( byte* ptr = &b_dll_path[0] ) {
                        ret = vstidrv_Init( ptr, block_size, sample_rate );
                    }
                } catch( Exception ex) {
#if DEBUG
                    Common.DebugWriteLine( "    ex=" + ex );
#endif
                    ret = false;
                }
                return ret;
            }

            public static int SendEvent( byte* src, int* deltaFrames, int numEvents, int targetTrack ) {
                int ret = 0;
                try {
                    ret = vstidrv_SendEvent( src, deltaFrames, numEvents, targetTrack );
                } catch {
                    ret = -1;
                }
                return ret;
            }

            public static int StartRendering(
                long total_samples,
                double amplify_left,
                double amplify_right,
                int error_samples,
                bool event_enabled,
                bool direct_play_enabled,
                string[] files,
                double wave_read_offset_seconds,
                bool mode_infinite
            ) {
                int ret = 0;
                try {
                    IntPtr intptr_files = Marshal.AllocHGlobal( sizeof( char* ) * files.Length );
                    char** ptr_files = (char**)intptr_files.ToPointer();
                    IntPtr[] cont_intptr_files = new IntPtr[files.Length];
                    for ( int i = 0; i < files.Length; i++ ) {
                        cont_intptr_files[i] = Marshal.AllocHGlobal( sizeof( char ) * (files.Length + 1) );
                        ptr_files[i] = (char*)cont_intptr_files[i].ToPointer();
                        for ( int j = 0; j < files[i].Length; j++ ) {
                            ptr_files[i][j] = files[i][j];
                        }
                        ptr_files[i][files[i].Length] = '\0';
                    }
                    ret = vstidrv_StartRendering( total_samples,
                                                  amplify_left,
                                                  amplify_right,
                                                  error_samples,
                                                  event_enabled,
                                                  direct_play_enabled,
                                                  ptr_files,
                                                  files.Length,
                                                  wave_read_offset_seconds,
                                                  mode_infinite );
                    for ( int i = 0; i < files.Length; i++ ) {
                        Marshal.FreeHGlobal( cont_intptr_files[i] );
                    }
                    Marshal.FreeHGlobal( intptr_files );
                } catch {
                    ret = -1;
                }
                return ret;
            }

            public static void AbortRendering() {
                try {
                    vstidrv_AbortRendering();
                } catch {
                }
            }

            public static double GetProgress() {
                double ret = 0.0;
                try {
                    ret = vstidrv_GetProgress();
                } catch {
                }
                return ret;
            }

            public static float GetPlayTime() {
                float ret = -1.0f;
#if DEBUG
                //Common.DebugWriteLine( "vstidrv+GetPlayTime" );
#endif
                try {
                    ret = vstidrv_GetPlayTime();
                } catch ( Exception ex ) {
#if DEBUG
                    Common.DebugWriteLine( "    ex=" + ex );
#endif
                    ret = -1.0f;
                }
#if DEBUG
                Common.DebugWriteLine( "    ret=" + ret );
#endif
                return ret;
            }

            public static void WaveOutReset() {
#if DEBUG
                Common.DebugWriteLine( "WaveOutReset" );
#endif
                try {
                    vstidrv_WaveOutReset();
                } catch( Exception ex ) {
#if DEBUG
                    Common.DebugWriteLine( "    ex=" + ex );
#endif
                }
            }

            public static int JoyInit() {
#if DEBUG
                Common.DebugWriteLine( "JoyInit" );
#endif
                int ret = 0;
                try {
                    ret = vstidrv_JoyInit();
                } catch ( Exception ex ) {
#if DEBUG
                    Common.DebugWriteLine( "    ex=" + ex );
#endif
                    ret = 0;
                }
                return ret;
            }

            public static bool JoyGetStatus( int index, out byte[] buttons, out int pov ) {
                bool ret = false;
                pov = -1;
                try {
                    int len = vstidrv_JoyGetNumButtons( index );
                    buttons = new byte[len];
                    IntPtr iptr = Marshal.AllocHGlobal( sizeof( byte ) * len );
                    byte* ptr_buttons = (byte*)iptr.ToPointer();
                    for ( int i = 0; i < len; i++ ) {
                        ptr_buttons[i] = 0x0;
                    }
                    int src_pov;
                    ret = vstidrv_JoyGetStatus( index, ptr_buttons, len, &src_pov );
                    for ( int i = 0; i < buttons.Length; i++ ) {
                        buttons[i] = ptr_buttons[i];
                    }
                    Marshal.FreeHGlobal( iptr );
                    pov = src_pov;
                } catch ( Exception ex ){
                    buttons = new byte[0];
                    pov = -1;
                    ret = false;
                }
                return ret;
            }

            public static int JoyGetNumJoyDev() {
                int ret = 0;
                try {
                    ret = vstidrv_JoyGetNumJoyDev();
                } catch ( Exception ex ){
                    ret = 0;
                }
                return ret;
            }

            //void vstidrv_setFirstBufferWrittenCallback( FirstBufferWrittenCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setFirstBufferWrittenCallback( IntPtr proc );

            //void vstidrv_setWaveIncomingCallback( WaveIncomingCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setWaveIncomingCallback( IntPtr proc );

            //void vstidrv_setRenderingFinishedCallback( RenderingFinishedCallback proc );
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_setRenderingFinishedCallback( IntPtr proc );

            //bool vstidrv_Init( string dll_path, int block_size, int sample_rate );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_Init( byte* dll_path, int block_size, int sample_rate );

            //int  vstidrv_SendEvent( unsigned char *src, int *deltaFrames, int numEvents, int targetTrack );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_SendEvent( byte* src, int* deltaFrames, int numEvents, int targetTrack );

            //int vstidrv_StartRendering( __int64 total_samples, double amplify_left, double amplify_right, int error_samples, bool event_enabled, bool direct_play_enabled, wchar_t** files, int num_files, double wave_read_offset_seconds );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_StartRendering( long total_samples,
                double amplify_left,
                double amplify_right,
                int error_samples,
                bool event_enabled,
                bool direct_play_enabled,
                char** files,
                int num_files,
                double wave_read_offset_seconds,
                bool mode_infinite );


            //void vstidrv_AbortRendering();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_AbortRendering();

            //double vstidrv_GetProgress();        
            [DllImport( "vstidrv.dll" )]
            private static extern double vstidrv_GetProgress();

            //float vstidrv_GetPlayTime();
            [DllImport( "vstidrv.dll" )]
            private static extern float vstidrv_GetPlayTime();

            //void vstidrv_WaveOutReset();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_WaveOutReset();

            //void vstidrv_Terminate();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_Terminate();

            //int vstidrv_JoyInit();
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyInit();

            //bool vstidrv_JoyIsJoyAttatched( int index );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_JoyIsJoyAttatched( int index );

            //bool vstidrv_JoyGetStatus( int index, unsigned char *buttons, int *pov );
            [DllImport( "vstidrv.dll" )]
            private static extern bool vstidrv_JoyGetStatus( int index, byte* buttons, int len, int* pov );

            //int vstidrv_JoyGetNumButtons( int index );
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyGetNumButtons( int index );

            //void vstidrv_JoyReset();
            [DllImport( "vstidrv.dll" )]
            private static extern void vstidrv_JoyReset();

            //int vstidrv_JoyGetNumJoyDev();
            [DllImport( "vstidrv.dll" )]
            private static extern int vstidrv_JoyGetNumJoyDev();

        }
#endif
    }

}
