﻿/*
 * RenderUtau.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
//#define USE_THREAD
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Drawing;

using Boare.Lib.Vsq;
using bocoree;

namespace Boare.Cadencii {

    public static class RenderUtau {
        public const string FILEBASE = "temp.wav";
        private static List<RenderQueue> resampler_queue = new List<RenderQueue>();
        private static bool s_abort_required = false;
        private static double[] left;
        private static double[] right;
        private static double s_progress = 0.0;
        private static Dictionary<string, ValuePair<string, DateTime>> s_cache = new Dictionary<string, ValuePair<string, DateTime>>();
        private const int _MAX_CACHE = 512;

        public static event WaveIncomingEventHandler WaveIncoming;
        public static event RenderingFinishedEventHandler RenderingFinished;

        /// <summary>
        /// 原音設定の引数．
        /// </summary>
        struct OtoArgs {
            public string Alias;
            public int msOffset;
            public int msConsonant;
            public int msBlank;
            public int msPreUtterance;
            public int msOverlap;
        }

        class RenderQueue {
            public string ResamplerArg;
            public string WavtoolArgPrefix;
            public string WavtoolArgSuffix;
            public OtoArgs Oto;
            public double secEnd;
            public string FileName;
            public bool ResamplerFinished;
        }

        public static void ClearCache() {
            foreach ( KeyValuePair<string, ValuePair<string, DateTime>> item in s_cache ) {
                string file = item.Value.Key;
                try {
                    File.Delete( file );
                } catch {
                }
            }
            s_cache.Clear();
        }

        public static double GetProgress() {
            return s_progress;
        }

        public static void AbortRendering() {
            s_abort_required = true;
        }

        public static void StartRendering(
            VsqFileEx vsq,
            int track,
            List<SingerConfig> singer_config_sys,
            string resampler,
            string wavtool,
            string temp_dir,
            bool invoke_with_wine,
            int sample_rate,
            int trim_msec,
            bool mode_infinite
        ) {
            try {
                double sample_length = vsq.getSecFromClock( vsq.TotalClocks ) * sample_rate;
                s_abort_required = false;
                s_progress = 0.0;
                if ( !Directory.Exists( temp_dir ) ) {
                    Directory.CreateDirectory( temp_dir );
                }
#if DEBUG
                //StreamWriter sw = new StreamWriter( Path.Combine( temp_dir, "log.txt" ) );
#endif
                // 原音設定を読み込み
                Dictionary<int, Dictionary<string, OtoArgs>> config = new Dictionary<int, Dictionary<string, OtoArgs>>();
                List<SingerConfig> singers = singer_config_sys;
                for( int pc = 0; pc < singers.Count; pc++ ){
                    string singer_name = singers[pc].VOICENAME;
                    string singer_path = singers[pc].VOICEIDSTR;
#if DEBUG
                    //TODO: mono on linuxにて、singer_pathが取得できていない？
#endif
                    string config_file = Path.Combine( singer_path, "oto.ini" );
                    Dictionary<string, OtoArgs> list = new Dictionary<string, OtoArgs>();
                    if ( File.Exists( config_file ) ) {
                        using ( cp932reader sr = new cp932reader( config_file ) ) {
                            string line;
                            while ( sr.Peek() >= 0 ) {
                                line = sr.ReadLine();
                                string[] spl = line.Split( '=' );
                                string a1 = spl[0];
                                string a2 = spl[1];
                                int index = a1.IndexOf( ".wav" );
                                a1 = a1.Substring( 0, index );
                                spl = a2.Split( ',' );
                                OtoArgs oa = new OtoArgs();
                                oa.Alias = spl[0];
                                oa.msOffset = int.Parse( spl[1] );
                                oa.msConsonant = int.Parse( spl[2] );
                                oa.msBlank = int.Parse( spl[3] );
                                oa.msPreUtterance = int.Parse( spl[4] );
                                oa.msOverlap = int.Parse( spl[5] );
                                list.Add( a1, oa );
                            }
                        }
                    }
                    config.Add( pc, list );
                }

                string file = Path.Combine( temp_dir, FILEBASE );
                if ( File.Exists( file ) ) {
                    File.Delete( file );
                }
                string file_whd = Path.Combine( temp_dir, FILEBASE + ".whd" );
                if ( File.Exists( file_whd ) ) {
                    File.Delete( file_whd );
                }
                string file_dat = Path.Combine( temp_dir, FILEBASE + ".dat" );
                if ( File.Exists( file_dat ) ) {
                    File.Delete( file_dat );
                }
#if DEBUG
                AppManager.DebugWriteLine( "temp_dir=" + temp_dir );
                AppManager.DebugWriteLine( "file_whd=" + file_whd );
                AppManager.DebugWriteLine( "file_dat=" + file_dat );
#endif

                int count = -1;
                double sec_end = 0;
                double sec_end_old = 0;
                int program_change = 0;
                resampler_queue.Clear();
                VsqBPList pitch_curve = vsq.getPitchCurve( track );
                for ( Iterator itr = vsq.Track[track].getEventIterator(); itr.hasNext(); ) {
                    VsqEvent item = (VsqEvent)itr.next();
                    if ( item.ID.type == VsqIDType.Singer ) {
                        program_change = item.ID.IconHandle.Program;
                        continue;
                    } else if ( item.ID.type == VsqIDType.Unknown ) {
                        continue;
                    }
                    if ( s_abort_required ) {
                        return;
                    }
                    count++;
                    double sec_start = vsq.getSecFromClock( item.Clock );
                    sec_end_old = sec_end;
                    sec_end = vsq.getSecFromClock( item.Clock + item.ID.Length );
                    float t_temp = (float)(item.ID.Length / (sec_end - sec_start) / 8.0);
                    if ( (count == 0 && sec_start > 0.0) || (sec_start > sec_end_old) ) {
                        double sec_start2 = sec_end_old;
                        double sec_end2 = sec_start;
                        float t_temp2 = (float)(item.Clock / (sec_end2 - sec_start2) / 8.0);
                        string singer = "";
                        if ( 0 <= program_change && program_change < singers.Count ) {
                            singer = singers[program_change].VOICEIDSTR;
                        }
                        RenderQueue rq = new RenderQueue();
                        rq.ResamplerArg = "";
                        rq.WavtoolArgPrefix = "\"" + file + "\" \"" + Path.Combine( singer, "R.wav" ) + "\" 0 " + item.Clock + "@" + string.Format( "{0:f2}", t_temp2 );
                        rq.WavtoolArgSuffix = " 0 0";
                        rq.Oto = new OtoArgs();
                        rq.FileName = "";
                        rq.secEnd = sec_end2;
                        rq.ResamplerFinished = true;
                        resampler_queue.Add( rq );
                        count++;
                    }
                    string lyric = item.ID.LyricHandle.L0.Phrase;
                    string note = NoteStringFromNoteNumber( item.ID.Note );
                    int millisec = (int)((sec_end - sec_start) * 1000) + 50;

                    OtoArgs oa = new OtoArgs();
                    if ( config.ContainsKey( program_change ) && config[program_change].ContainsKey( lyric ) ) {
                        oa = config[program_change][lyric];
                    }
                    string singer2 = "";
                    if ( 0 <= program_change && program_change < singers.Count ) {
                        singer2 = singers[program_change].VOICEIDSTR;
                    }
                    oa.msPreUtterance = item.UstEvent.PreUtterance;
                    oa.msOverlap = item.UstEvent.VoiceOverlap;
                    RenderQueue rq2 = new RenderQueue();
                    string arg_prefix = "\"" + Path.Combine( singer2, lyric + ".wav" ) + "\"";
                    string arg_suffix = "\"" + note + "\" 100 " + item.UstEvent.Flags + "L" + " " + oa.msOffset + " " + millisec + " " + oa.msConsonant + " " + oa.msBlank + " 100 " + item.UstEvent.Moduration;

                    // ピッチを取得
                    string pitch = "";
                    bool allzero = true;
                    const int delta_clock = 5;  //ピッチを取得するクロック間隔
                    int tempo = 120;
                    double delta_sec = delta_clock / (8.0 * tempo); //ピッチを取得する時間間隔
                    if ( item.ID.VibratoHandle == null ) {
                        int pit_count = (int)((sec_end - sec_start) / delta_sec) + 1;
                        int pitindex = 0;
                        for ( int i = 0; i < pit_count; i++ ) {
                            double gtime = sec_start + delta_sec * i;
                            int clock = (int)vsq.getClockFromSec( gtime );
                            float pvalue = pitch_curve.getValue( clock, ref pitindex ) / 100.0f;
                            if ( pvalue != 0 ) {
                                allzero = false;
                            }
                            pitch += " " + pvalue.ToString( "0.00" );
                            if ( i == 0 ) {
                                pitch += "Q" + tempo;
                            }
                        }
                    } else {
                        // ビブラートが始まるまでのピッチを取得
                        double sec_vibstart = vsq.getSecFromClock( item.Clock + item.ID.VibratoDelay );
                        int pit_count = (int)((sec_vibstart - sec_start) / delta_sec);
                        int pitindex =0;
                        int totalcount = 0;
                        for ( int i = 0; i < pit_count; i++ ) {
                            double gtime = sec_start + delta_sec * i;
                            int clock = (int)vsq.getClockFromSec( gtime );
                            float pvalue = pitch_curve.getValue( clock, ref pitindex ) / 100.0f;
                            pitch += " " + pvalue.ToString( "0.00" );
                            if ( totalcount == 0 ) {
                                pitch += "Q" + tempo;
                            }
                            totalcount++;
                        }
                        List<PointF> ret = FormMain.GetVibratoPoints( vsq,
                                                                      item.ID.VibratoHandle.RateBP,
                                                                      item.ID.VibratoHandle.StartRate,
                                                                      item.ID.VibratoHandle.DepthBP,
                                                                      item.ID.VibratoHandle.StartDepth,
                                                                      item.Clock + item.ID.VibratoDelay,
                                                                      item.ID.Length - item.ID.VibratoDelay,
                                                                      (float)delta_sec );
                        for ( int i = 0; i < ret.Count; i++ ) {
                            float gtime = ret[i].X;
                            int clock = (int)vsq.getClockFromSec( gtime );
                            float pvalue = pitch_curve.getValue( clock, ref pitindex ) / 100.0f;
                            pitch += " " + (pvalue + ret[i].Y * 100.0f).ToString( "0.00" );
                            if ( totalcount == 0 ) {
                                pitch += "Q" + tempo;
                            }
                            totalcount++;
                        }
                        allzero = totalcount == 0;
                    }
                    
                    //4_あ_C#4_550.wav
                    string filename = Path.Combine( temp_dir, misc.getmd5( s_cache.Count + arg_prefix + arg_suffix + pitch ) + ".wav" );

                    rq2.ResamplerArg = arg_prefix + " \"" + filename + "\" " + arg_suffix;
                    if ( !allzero ) {
                        rq2.ResamplerArg += pitch;
                    }

                    string search_key = arg_prefix + arg_suffix + pitch;
                    bool exist_in_cache = s_cache.ContainsKey( search_key );
                    if ( !exist_in_cache ) {
                        if ( s_cache.Count + 1 >= _MAX_CACHE ) {
                            DateTime old = DateTime.Now;
                            string delfile = "";
                            string delkey = "";
                            foreach ( KeyValuePair<string, ValuePair<string, DateTime>> targ in s_cache ) {
                                if ( old.CompareTo( targ.Value.Value ) < 0 ) {
                                    old = targ.Value.Value;
                                    delfile = targ.Value.Key;
                                    delkey = targ.Key;
                                }
                            }
                            try {
#if DEBUG
                                bocoree.debug.push_log( "deleting... \"" + delfile + "\"" );
#endif
                                File.Delete( delfile );
                            } catch {
                            }
                            s_cache.Remove( delkey );
                        }
                        s_cache.Add( search_key, new ValuePair<string, DateTime>( filename, DateTime.Now ) );
                    } else {
                        filename = s_cache[search_key].Key;
                    }

                    rq2.WavtoolArgPrefix = "\"" + file + "\" \"" + filename + "\" 0 " + item.ID.Length + "@" + string.Format( "{0:f2}", t_temp );
                    UstEnvelope env = item.UstEvent.Envelope;
                    if ( env == null ) {
                        env = new UstEnvelope();
                    }
                    rq2.WavtoolArgSuffix = " " + env.p1 + " " + env.p2 + " " + env.p3 + " " + env.v1 + " " + env.v2 + " " + env.v3 + " " + env.v4;
                    rq2.WavtoolArgSuffix += " " + oa.msOverlap + " " + env.p4 + " " + env.p5 + " " + env.v5;
                    rq2.Oto = oa;
                    rq2.FileName = filename;
                    rq2.secEnd = sec_end;
                    rq2.ResamplerFinished = exist_in_cache;
                    resampler_queue.Add( rq2 );
                }

#if USE_THREAD
                Thread t = new Thread( new ParameterizedThreadStart( ResampleProc ) );
                ResampleProcArg rpa = new ResampleProcArg();
                rpa.InvokeWithWine = invoke_with_wine;
                rpa.PathResampler = resampler;
                rpa.PathTemp = temp_dir;
                t.Priority = ThreadPriority.Lowest;
                t.Start( rpa );
#endif
#if DEBUG
                bocoree.debug.push_log( "s_cache:" );
                foreach ( KeyValuePair<string, ValuePair<string, DateTime>> item in s_cache ) {
                    bocoree.debug.push_log( "    arg=" + item.Key );
                    bocoree.debug.push_log( "    file=" + item.Value.Key );
                }
#endif

                int num_queues = resampler_queue.Count;
                int processed_sample = 0; //WaveIncomingで受け渡した波形の合計サンプル数
                int channel = 0; // .whdに記録されたチャンネル数
                int byte_per_sample = 0;
                // 引き続き、wavtoolを呼ぶ作業に移行
                bool first = true;
                int trim_remain = (int)(trim_msec / 1000.0 * sample_rate); //先頭から省かなければならないサンプル数の残り
#if DEBUG
                bocoree.debug.push_log( "trim_remain=" + trim_remain );
#endif
                VsqBPList dyn_curve = vsq.Track[track].getCurve( "dyn" );
                for ( int i = 0; i < num_queues; i++ ) {
#if USE_THREAD
                    while ( !s_abort_required && !resampler_queue[i].ResampleFinished ) {
                        Application.DoEvents();
                    }
#else
                    RenderQueue rq = resampler_queue[i];
                    if ( !rq.ResamplerFinished ) {
                        string arg = rq.ResamplerArg;
#if DEBUG
                        bocoree.debug.push_log( "resampler arg=" + arg );
#endif
                        using ( Process process = new Process() ) {
                            process.StartInfo.FileName = (invoke_with_wine ? "wine \"" : "\"") + resampler + "\"";
                            process.StartInfo.Arguments = arg;
                            process.StartInfo.WorkingDirectory = temp_dir;
                            process.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Hidden;
                            process.Start();
                            process.WaitForExit();
                        }
                    }
#endif
                    if ( s_abort_required ) {
                        break;
                    }

                    // wavtoolを起動
                    double sec_fin; // 今回のwavtool起動によってレンダリングが完了したサンプル長さ
                    RenderQueue p = resampler_queue[i]; //Phon p = s_resampler_queue[i];
                    OtoArgs oa_next;
                    if ( i + 1 < num_queues ) {
                        oa_next = resampler_queue[i + 1].Oto;
                    } else {
                        oa_next = new OtoArgs();
                    }
                    sec_fin = p.secEnd - oa_next.msOverlap / 1000.0;
#if DEBUG
                    AppManager.DebugWriteLine( "sec_fin=" + sec_fin );
#endif
                    int mten = p.Oto.msPreUtterance + oa_next.msOverlap - oa_next.msPreUtterance;
                    string arg_wavtool = p.WavtoolArgPrefix + (mten >= 0 ? ("+" + mten) : ("-" + (-mten))) + p.WavtoolArgSuffix;
                    ProcessWavtool( arg_wavtool, file, temp_dir, wavtool, invoke_with_wine );

                    // できたwavを読み取ってWaveIncomingイベントを発生させる
                    int sample_end = (int)(sec_fin * sample_rate);
#if DEBUG
                    AppManager.DebugWriteLine( "RenderUtau.StartRendering; sample_end=" + sample_end );
#endif
                    // whdを読みに行く
                    if ( first ) {
                        using ( FileStream whd = new FileStream( file_whd, FileMode.Open, FileAccess.Read ) ) {
                            #region whdを読みに行く
                            whd.Seek( 0, SeekOrigin.Begin );
                            // RIFF
                            byte[] buf = new byte[4];
                            int gcount = whd.Read( buf, 0, 4 );
                            if ( buf[0] != 'R' || buf[1] != 'I' || buf[2] != 'F' || buf[3] != 'F' ) {
#if DEBUG
                                AppManager.DebugWriteLine( "RenderUtau.startRendering; whd header error" );
                                AppManager.DebugWriteLine( ((char)buf[0]).ToString() + "" + ((char)buf[1]).ToString() + "" + ((char)buf[2]).ToString() + "" + ((char)buf[3]).ToString() + " must be RIFF" );
#endif
                                continue;
                            }
                            // ファイルサイズ
                            whd.Read( buf, 0, 4 );
                            // WAVE
                            whd.Read( buf, 0, 4 );
                            if ( buf[0] != 'W' || buf[1] != 'A' || buf[2] != 'V' || buf[3] != 'E' ) {
#if DEBUG
                                AppManager.DebugWriteLine( "RenderUtau.startRendering; whd header error" );
                                AppManager.DebugWriteLine( ((char)buf[0]).ToString() + "" + ((char)buf[1]).ToString() + "" + ((char)buf[2]).ToString() + "" + ((char)buf[3]).ToString() + " must be WAVE" );
#endif
                                continue;
                            }
                            // fmt 
                            whd.Read( buf, 0, 4 );
                            if ( buf[0] != 'f' || buf[1] != 'm' || buf[2] != 't' || buf[3] != ' ' ) {
#if DEBUG
                                AppManager.DebugWriteLine( "RenderUtau.startRendering; whd header error" );
                                AppManager.DebugWriteLine( ((char)buf[0]).ToString() + "" + ((char)buf[1]).ToString() + "" + ((char)buf[2]).ToString() + "" + ((char)buf[3]).ToString() + " must be fmt " );
#endif
                                continue;
                            }
                            // fmt チャンクのサイズ
                            whd.Read( buf, 0, 4 );
                            long loc_end_of_fmt = whd.Position; //fmtチャンクの終了位置．ここは一定値でない可能性があるので読込み
                            loc_end_of_fmt += buf[0] | buf[1] << 8 | buf[2] << 16 | buf[3] << 24;
                            // format ID
                            whd.Read( buf, 0, 2 );
                            int id = buf[0] | buf[1] << 8;
                            if ( id != 0x0001 ) { //0x0001はリニアPCM
                                continue;
                            }
                            // チャンネル数
                            whd.Read( buf, 0, 2 );
                            channel = buf[1] << 8 | buf[0];
                            // サンプリングレート
                            whd.Read( buf, 0, 4 );
                            int this_sample_rate = buf[0] | buf[1] << 8 | buf[2] << 16 | buf[3] << 24;
                            // データ速度
                            whd.Read( buf, 0, 4 );
                            // ブロックサイズ
                            whd.Read( buf, 0, 2 );
                            // 1チャンネル、1サンプルあたりのビット数
                            whd.Read( buf, 0, 2 );
                            int bit_per_sample = buf[1] << 8 | buf[0];
                            byte_per_sample = bit_per_sample / 8;
                            whd.Seek( loc_end_of_fmt, SeekOrigin.Begin );
                            // data
                            whd.Read( buf, 0, 4 );
                            if ( buf[0] != 'd' || buf[1] != 'a' || buf[2] != 't' || buf[3] != 'a' ) {
#if DEBUG
                                AppManager.DebugWriteLine( "RenderUtau.startRendering; whd header error" );
                                AppManager.DebugWriteLine( ((char)buf[0]).ToString() + "" + ((char)buf[1]).ToString() + "" + ((char)buf[2]).ToString() + "" + ((char)buf[3]).ToString() + " must be data" );
#endif
                                continue;
                            }
                            // size of data chunk
                            whd.Read( buf, 0, 4 );
                            int size = buf[3] << 24 | buf[2] << 16 | buf[1] << 8 | buf[0];
                            int total_samples = size / (channel * byte_per_sample);
                            #endregion
                        }
                        first = false;
                    }

                    // datを読みに行く
                    int sampleFrames = sample_end - processed_sample;
#if DEBUG
                    AppManager.DebugWriteLine( "RenderUtau.StartRendering; sampleFrames=" + sampleFrames + "; channel=" + channel + "; byte_per_sample=" + byte_per_sample );
#endif
                    if ( channel > 0 && byte_per_sample > 0 && sampleFrames > 0 && WaveIncoming != null ) {
                        int length = (sampleFrames > sample_rate ? sample_rate : sampleFrames);
                        int remain = sampleFrames;
                        left = new double[length];
                        right = new double[length];
                        const float k_inv64 = 1.0f / 64.0f;
                        const float k_inv32768 = 1.0f / 32768.0f;
                        const int buflen = 1024;
                        byte[] wavbuf = new byte[buflen];
                        int pos = 0;
                        FileStream dat = null;
                        try {
                            dat = new FileStream( file_dat, FileMode.Open, FileAccess.Read );
                            dat.Seek( processed_sample * channel * byte_per_sample, SeekOrigin.Begin );
                            double sec_start = processed_sample / (double)sample_rate;
                            double sec_per_sa = 1.0 / (double)sample_rate;
                            int index = 0;
                            #region チャンネル数／ビット深度ごとの読み取り操作
                            if ( byte_per_sample == 1 ) {
                                if ( channel == 1 ) {
                                    while ( remain > 0 ) {
                                        int len = dat.Read( wavbuf, 0, buflen );
                                        if ( len <= 0 ) {
                                            break;
                                        }
                                        int c = 0;
                                        while ( len > 0 && remain > 0 ) {
                                            len -= 1;
                                            remain--;
                                            if ( trim_remain > 0 ) {
                                                c++;
                                                trim_remain--;
                                                continue;
                                            }
                                            double gtime_dyn = sec_start + pos * sec_per_sa;
                                            int clock = (int)vsq.getClockFromSec( gtime_dyn );
                                            int dyn = dyn_curve.getValue( clock, ref index );
                                            float amp = (float)dyn * k_inv64;
                                            float v = (wavbuf[c] - 64.0f) * k_inv64 * amp;
                                            c++;
                                            left[pos] = v;
                                            right[pos] = v;
                                            pos++;
                                            if ( pos >= length ) {
                                                WaveIncoming( left, right );
                                                pos = 0;
                                            }
                                        }
                                    }
                                } else {
                                    while ( remain > 0 ) {
                                        int len = dat.Read( wavbuf, 0, buflen );
                                        if ( len <= 0 ) {
                                            break;
                                        }
                                        int c = 0;
                                        while ( len > 0 && remain > 0 ) {
                                            len -= 2;
                                            remain--;
                                            if ( trim_remain > 0 ) {
                                                c += 2;
                                                trim_remain--;
                                                continue;
                                            }
                                            double gtime_dyn = sec_start + pos * sec_per_sa;
                                            int clock = (int)vsq.getClockFromSec( gtime_dyn );
                                            int dyn = dyn_curve.getValue( clock, ref index );
                                            float amp = (float)dyn * k_inv64;
                                            float vl = (wavbuf[c] - 64.0f) * k_inv64 * amp;
                                            float vr = (wavbuf[c + 1] - 64.0f) * k_inv64 * amp;
                                            left[pos] = vl;
                                            right[pos] = vr;
                                            c += 2;
                                            pos++;
                                            if ( pos >= length ) {
                                                WaveIncoming( left, right );
                                                pos = 0;
                                            }
                                        }
                                    }
                                }
                            } else if ( byte_per_sample == 2 ) {
                                if ( channel == 1 ) {
                                    while ( remain > 0 ) {
                                        int len = dat.Read( wavbuf, 0, buflen );
                                        if ( len <= 0 ) {
                                            break;
                                        }
                                        int c = 0;
                                        while ( len > 0 && remain > 0 ) {
                                            len -= 2;
                                            remain--;
                                            if ( trim_remain > 0 ) {
                                                c += 2;
                                                trim_remain--;
                                                continue;
                                            }
                                            double gtime_dyn = sec_start + pos * sec_per_sa;
                                            int clock = (int)vsq.getClockFromSec( gtime_dyn );
                                            int dyn = dyn_curve.getValue( clock, ref index );
                                            float amp = (float)dyn * k_inv64;
                                            float v = ((short)(wavbuf[c] | wavbuf[c + 1] << 8)) * k_inv32768 * amp;
                                            left[pos] = v;
                                            right[pos] = v;
                                            c += 2;
                                            pos++;
                                            if ( pos >= length ) {
                                                WaveIncoming( left, right );
                                                pos = 0;
                                            }
                                        }
                                    }
                                } else {
                                    while ( remain > 0 ) {
                                        int len = dat.Read( wavbuf, 0, buflen );
                                        if ( len <= 0 ) {
                                            break;
                                        }
                                        int c = 0;
                                        while ( len > 0 && remain > 0 ) {
                                            len -= 4;
                                            remain--;
                                            if ( trim_remain > 0 ) {
                                                c += 4;
                                                trim_remain--;
                                                continue;
                                            }
                                            double gtime_dyn = sec_start + pos * sec_per_sa;
                                            int clock = (int)vsq.getClockFromSec( gtime_dyn );
                                            int dyn = dyn_curve.getValue( clock, ref index );
                                            float amp = (float)dyn * k_inv64;
                                            float vl = ((short)(wavbuf[c] | wavbuf[c + 1] << 8)) * k_inv32768 * amp;
                                            float vr = ((short)(wavbuf[c + 2] | wavbuf[c + 3] << 8)) * k_inv32768 * amp;
                                            left[pos] = vl;
                                            right[pos] = vr;
                                            c += 4;
                                            pos++;
                                            if ( pos >= length ) {
                                                WaveIncoming( left, right );
                                                pos = 0;
                                            }
                                        }
                                    }
                                }
                            }
                            #endregion
                        } catch ( Exception ex ) {
                            if ( dat != null ) {
                                dat.Close();
                                dat = null;
                            }
#if DEBUG
                            AppManager.DebugWriteLine( "RenderUtau.StartRendering; ex=" + ex );
#endif
                        } finally {
                            if ( dat != null ) {
                                dat.Close();
                                dat = null;
                            }
                        }
#if DEBUG
                        AppManager.DebugWriteLine( "calling WaveIncoming..." );
#endif
                        if ( pos > 0 ) {
                            double[] bufl = new double[pos];
                            double[] bufr = new double[pos];
                            for ( int j = 0; j < pos; j++ ) {
                                bufl[j] = left[j];
                                bufr[j] = right[j];
                            }
                            WaveIncoming( bufl, bufr );
                            bufl = null;
                            bufr = null;
                        }
                        left = null;
                        right = null;
                        GC.Collect();
#if DEBUG
                        AppManager.DebugWriteLine( "...done(calling WaveIncoming)" );
#endif
                        processed_sample += (sampleFrames - remain);
                        s_progress = processed_sample / sample_length * 100.0;
                    }
                }

                if ( mode_infinite ) {
                    double[] silence_l = new double[44100];
                    double[] silence_r = new double[44100];
                    while ( !s_abort_required ) {
                        WaveIncoming( silence_l, silence_r );
                    }
                    silence_l = null;
                    silence_r = null;
                }

            } catch ( Exception ex ) {
#if DEBUG
                AppManager.DebugWriteLine( "RenderUtau.StartRendering; ex=" + ex );
#endif
            } finally {
                if ( RenderingFinished != null ) {
                    RenderingFinished();
                }
            }
        }

        class ResampleProcArg {
            public bool InvokeWithWine;
            public string PathResampler;
            public string PathTemp;
        }

#if USE_THREAD
        public static void ResampleProc(object argument) {
            ResampleProcArg rpa = (ResampleProcArg)argument;
            for ( int i = 0; i < resampler_queue.Count && !s_abort_required; i++ ) {
                RenderQueue rq = resampler_queue[i];
                if ( rq.ResamplerArg != "" ) {
                    string arg = rq.ResamplerArg;
#if DEBUG
                    bocoree.debug.push_log( "resampler arg=" + arg );
#endif
                    using ( Process process = new Process() ) {
                        process.StartInfo.FileName = (rpa.InvokeWithWine ? "wine \"" : "\"") + rpa.PathResampler + "\"";
                        process.StartInfo.Arguments = arg;
                        process.StartInfo.WorkingDirectory = rpa.PathTemp;
                        process.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Hidden;
                        process.Start();
                        process.WaitForExit();
                    }
                }
                resampler_queue[i].ResampleFinished = true;
            }
        }
#endif

        private static void ProcessWavtool( string arg, string filebase, string temp_dir, string wavtool, bool invoke_with_wine ) {
#if DEBUG
            bocoree.debug.push_log( "wavtool arg=" + arg );
#endif

            using ( Process process = new Process() ) {
                process.StartInfo.FileName = (invoke_with_wine ? "wine \"" : "\"") + wavtool + "\"";
                process.StartInfo.Arguments = arg;
                process.StartInfo.WorkingDirectory = temp_dir;
                process.StartInfo.WindowStyle = System.Diagnostics.ProcessWindowStyle.Hidden;
                process.Start();
                process.WaitForExit();
            }
        }

        private static string NoteStringFromNoteNumber( int note_number ) {
            int odd = note_number % 12;
            string head = (new string[] { "C", "C#", "D", "D#", "E", "F", "F#", "G", "G#", "A", "A#", "B" })[odd];
            return head + (note_number / 12 - 1);
        }
    }

}
