﻿/*
 * AppManager.cs
 * Copyright (c) 2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Reflection;
using System.Runtime.Serialization.Formatters.Binary;
using System.Threading;
using System.Windows.Forms;
using System.Xml.Serialization;

using Boare.Lib.Vsq;
/*
-コントロールカーブのテンプレートの登録＆貼付け機能
-ショートカットキーのカスタマイズ
-選択したノートのプロパティを一括変更する機能
-選択したノートを指定したクロック数分一括シフトする機能
-MIDIキーボード（？用語があってるか不明）を使えるようにする
-前回レンダリングしたWAVEファイルをリサイクルする（起動ごとにレンダリングするのは面倒？）
-横軸を変更するショートカットキー
-音符の時間位置を秒指定で入力できるモード
-コントロールカーブのリアルタイムかつアナログ的な入力（ジョイスティックorゲームコントローラ）
-コントロールカーブエディタを幾つか積めるような機能
 */
/*DONE:
 */
namespace Boare.Cadencii {

    public delegate void BSimpleDelegate<T>( T arg );

    public static class AppManager {
        /// <summary>
        /// 鍵盤の表示幅(pixel)
        /// </summary>
        public const int _KEY_LENGTH = 68;
        private const string _CONFIG_PATH = "config.xml";

        /// <summary>
        /// エディタの設定
        /// </summary>
        public static EditorConfig EditorConfig = new EditorConfig();
        /// <summary>
        /// List&lt;BezierCurves&gt;用のシリアライザ
        /// </summary>
        public static XmlSerializer XmlSerializerListBezierCurves = new XmlSerializer( typeof( List<BezierCurves> ) );
        /// <summary>
        /// コピーorカットなどで複製されたエントリ
        /// </summary>
        public static List<VsqEvent> CopiedEvent = null;
        /// <summary>
        /// コピーorカットで複製されたテンポ
        /// </summary>
        public static List<TempoTableEntry> CopiedTempo = null;
        /// <summary>
        /// コピーorカットで複製された拍子
        /// </summary>
        public static List<TimeSigTableEntry> CopiedTimesig = null;
        /// <summary>
        /// コピーorカットで複製されたカーブ
        /// </summary>
        public static Dictionary<CurveType, List<BPPair>> CopiedCurve = null;
        /// <summary>
        /// コピーorカットで複製されたベジエ曲線
        /// </summary>
        public static Dictionary<CurveType, List<BezierChain>> CopiedBezier = null;
        /// <summary>
        /// コピーの開始位置。貼付け時に、この値をもとにクロックのシフトを行う
        /// </summary>
        public static int CopyStartedClock;

        #region Static Readonly Fields
        public static readonly Color[] s_HILIGHT = new Color[] { 
            Color.FromArgb( 181, 220, 16 ),
            Color.FromArgb( 231, 244, 49 ),
            Color.FromArgb( 252, 230, 29 ),
            Color.FromArgb( 247, 171, 20 ),
            Color.FromArgb( 249, 94, 17 ),
            Color.FromArgb( 234, 27, 47 ),
            Color.FromArgb( 175, 20, 80 ),
            Color.FromArgb( 183, 24, 149 ),
            Color.FromArgb( 105, 22, 158 ),
            Color.FromArgb( 22, 36, 163 ),
            Color.FromArgb( 37, 121, 204 ),
            Color.FromArgb( 29, 179, 219 ),
            Color.FromArgb( 24, 239, 239 ),
            Color.FromArgb( 25, 206, 175 ),
            Color.FromArgb( 23, 160, 134 ),
            Color.FromArgb( 79, 181, 21 ) };
        public static readonly Color[] s_RENDER = new Color[]{
            Color.FromArgb( 19, 143, 52 ),
            Color.FromArgb( 158, 154, 18 ),
            Color.FromArgb( 160, 143, 23 ),
            Color.FromArgb( 145, 98, 15 ),
            Color.FromArgb( 142, 52, 12 ),
            Color.FromArgb( 142, 19, 37 ),
            Color.FromArgb( 96, 13, 47 ),
            Color.FromArgb( 117, 17, 98 ),
            Color.FromArgb( 62, 15, 99 ),
            Color.FromArgb( 13, 23, 84 ),
            Color.FromArgb( 25, 84, 132 ),
            Color.FromArgb( 20, 119, 142 ),
            Color.FromArgb( 19, 142, 139 ),
            Color.FromArgb( 17, 122, 102 ),
            Color.FromArgb( 13, 86, 72 ),
            Color.FromArgb( 43, 91, 12 ) };
        #endregion

        #region Private Static Fields
        private static int s_base_tempo = 480000;
        private static SolidBrush s_hilight_brush = new SolidBrush( Color.CornflowerBlue );
        private static object s_locker;
        private static XmlSerializer s_serizlizer = null;
        #endregion

        public delegate void MainFormClosedEventHandler( EditorManager manager );
        public delegate void EditorConfigChangedEventHandler( EditorManager manager );


        static AppManager() {
            s_locker = new object();
            SymbolTable.LoadDictionary();
            LoadConfig();
            VSTiProxy.CurrentUser = "";

            #region Apply User Dictionary Configuration
            List<BKeyValuePair<string, bool>> current = new List<BKeyValuePair<string, bool>>();
            for ( int i = 0; i < SymbolTable.Count; i++ ) {
                current.Add( new BKeyValuePair<string, bool>( SymbolTable.GetSymbolTable( i ).Name, false ) );
            }
            List<BKeyValuePair<string, bool>> config_data = new List<BKeyValuePair<string, bool>>();
            for ( int i = 0; i < EditorConfig.UserDictionaries.Count; i++ ) {
                string[] spl = EditorConfig.UserDictionaries[i].Split( "\t".ToCharArray(), 2 );
                config_data.Add( new BKeyValuePair<string, bool>( spl[0], (spl[1] == "T" ? true : false) ) );
#if DEBUG
                    Common.DebugWriteLine( "    " + spl[0] + "," + spl[1] );
#endif
            }
            List<KeyValuePair<string, bool>> common = new List<KeyValuePair<string, bool>>();
            for ( int i = 0; i < config_data.Count; i++ ) {
                for ( int j = 0; j < current.Count; j++ ) {
                    if ( config_data[i].Key == current[j].Key ) {
                        current[j].Value = true; //こっちのboolは、AppManager.EditorConfigのUserDictionariesにもKeyが含まれているかどうかを表すので注意
                        common.Add( new KeyValuePair<string, bool>( config_data[i].Key, config_data[i].Value ) );
                        break;
                    }
                }
            }
            for ( int i = 0; i < current.Count; i++ ) {
                if ( !current[i].Value ) {
                    common.Add( new KeyValuePair<string, bool>( current[i].Key, false ) );
                }
            }
            SymbolTable.ChangeOrder( common.ToArray() );
            #endregion

            Boare.Lib.AppUtil.Messaging.LoadMessages();
            Boare.Lib.AppUtil.Messaging.Language = EditorConfig.Language;

            KeySoundPlayer.Init();
        }

        /// <summary>
        /// アプリケーションデータの保存位置を取得します
        /// Gets the path for application data
        /// </summary>
        public static string ApplicationDataPath {
            get {
                return Application.StartupPath;
            }
        }

        /// <summary>
        /// 位置クオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetPositionQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.PositionQuantize, EditorConfig.PositionQuantizeTriplet );
        }

        /// <summary>
        /// 音符長さクオンタイズ時の音符の最小単位を、クロック数に換算したものを取得します
        /// </summary>
        /// <returns></returns>
        public static int GetLengthQuantizeClock() {
            return QuantizeModeUtil.GetQuantizeClock( EditorConfig.LengthQuantize, EditorConfig.LengthQuantizeTriplet );
        }

        public static void SaveConfig() {
            // ユーザー辞書の情報を取り込む
            EditorConfig.UserDictionaries.Clear();
            for ( int i = 0; i < SymbolTable.Count; i++ ) {
                EditorConfig.UserDictionaries.Add( SymbolTable.GetSymbolTable( i ).Name + "\t" + (SymbolTable.GetSymbolTable( i ).Enabled ? "T" : "F") );
            }

            if ( s_serizlizer == null ) {
                s_serizlizer = new XmlSerializer( typeof( EditorConfig ) );
            }
            string file = Path.Combine( ApplicationDataPath, _CONFIG_PATH );
            using ( FileStream fs = new FileStream( file, FileMode.Create ) ) {
                s_serizlizer.Serialize( fs, EditorConfig );
            }
        }

        public static void LoadConfig() {
            string config_file = Path.Combine( ApplicationDataPath, _CONFIG_PATH );
            EditorConfig ret = null;
            if ( File.Exists( config_file ) ) {
                FileStream fs = null;
                try {
                    if ( s_serizlizer == null ) {
                        s_serizlizer = new XmlSerializer( typeof( EditorConfig ) );
                    }
                    fs = new FileStream( config_file, FileMode.Open );
                    ret = (EditorConfig)s_serizlizer.Deserialize( fs );
                } catch {
                } finally {
                    if ( fs != null ) {
                        fs.Close();
                    }
                }
            } else {
                config_file = Path.Combine( Application.StartupPath, _CONFIG_PATH );
                if ( File.Exists( config_file ) ) {
                    FileStream fs = null;
                    try {
                        if ( s_serizlizer == null ) {
                            s_serizlizer = new XmlSerializer( typeof( EditorConfig ) );
                        }
                        fs = new FileStream( config_file, FileMode.Open );
                        ret = (EditorConfig)s_serizlizer.Deserialize( fs );
                    } catch {
                    } finally {
                        if ( fs != null ) {
                            fs.Close();
                        }
                    }
                }
            }
            if ( ret == null ) {
                ret = new EditorConfig();
            }
            EditorConfig = ret;
            for ( int i = 0; i < SymbolTable.Count; i++ ) {
                SymbolTable st = SymbolTable.GetSymbolTable( i );
                bool found = false;
                foreach ( string s in EditorConfig.UserDictionaries ) {
                    string[] spl = s.Split( "\t".ToCharArray(), 2 );
                    if ( st.Name == spl[0] ) {
                        found = true;
                        break;
                    }
                }
                if ( !found ) {
                    EditorConfig.UserDictionaries.Add( st.Name + "\tT" );
                }
            }
        }

        public static string _VERSION {
            get {
                string prefix = "";
#if DEBUG
                prefix = "(build: debug)";
#else
                prefix = "(build: release)";
#endif
                return GetAssemblyFileVersion( typeof( AppManager ) ) + " " + prefix;
            }
        }

        public static string GetAssemblyFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return afva.Version;
        }

        public static string GetAssemblyNameAndFileVersion( Type t ) {
            Assembly a = Assembly.GetAssembly( t );
            AssemblyFileVersionAttribute afva = (AssemblyFileVersionAttribute)Attribute.GetCustomAttribute( a, typeof( AssemblyFileVersionAttribute ) );
            return a.GetName().Name + " v" + afva.Version;
        }

        public static SolidBrush HilightBrush {
            get {
                return s_hilight_brush;
            }
        }

        public static Color HilightColor {
            get {
                return s_hilight_brush.Color;
            }
            set {
                s_hilight_brush = new SolidBrush( value );
            }
        }

        /// <summary>
        /// ベースとなるテンポ。
        /// </summary>
        public static int BaseTempo {
            get {
                return s_base_tempo;
            }
            set {
                s_base_tempo = value;
            }
        }
    }

}
