﻿/*
 * FormVibratoConfig.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the BSD License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.Windows.Forms;

using Boare.Lib.AppUtil;
using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    partial class FormVibratoConfig : Form {
        private VibratoHandle m_vibrato;
        private readonly string _NON_VIBRATO = "Non Vibrato";
        private int m_note_length;

        /// <summary>
        /// コンストラクタ．引数vibrato_handleには，Cloneしたものを渡さなくてよい．
        /// </summary>
        /// <param name="vibrato_handle"></param>
        public FormVibratoConfig( VibratoHandle vibrato_handle, int note_length, DefaultVibratoLength default_vibrato_length ) {
#if DEBUG
            Common.DebugWriteLine( "FormVibratoConfig.ctor(Vsqhandle,int,DefaultVibratoLength)" );
            Common.DebugWriteLine( "    (vibrato_handle==null)=" + (vibrato_handle == null) );
#endif
            if ( vibrato_handle != null ) {
                m_vibrato = (VibratoHandle)vibrato_handle.Clone();
            }

            InitializeComponent();
            ApplyLanguage();

            comboVibratoType.Items.Clear();
            comboVibratoType.Items.Add( _NON_VIBRATO );
            comboVibratoType.SelectedIndex = 0;
            int count = 0;
            foreach ( VibratoType type in Enum.GetValues( typeof( VibratoType ) ) ) {
                comboVibratoType.Items.Add( VibratoTypeUtil.ToString( type ) );
                count++;
                if ( vibrato_handle != null ) {
                    if ( vibrato_handle.IconID == VibratoTypeUtil.getIconIDFromVibratoType( type ) ) {
                        comboVibratoType.SelectedIndex = count;
                    }
                }
            }

            txtVibratoLength.Enabled = vibrato_handle != null;
            if ( vibrato_handle != null ) {
                txtVibratoLength.Text = (int)((float)vibrato_handle.Length / (float)note_length * 100.0f) + "";
            } else {
                switch ( default_vibrato_length ) {
                    case DefaultVibratoLength.L100:
                        txtVibratoLength.Text = "100";
                        break;
                    case DefaultVibratoLength.L50:
                        txtVibratoLength.Text = "50";
                        break;
                    case DefaultVibratoLength.L66:
                        txtVibratoLength.Text = "66";
                        break;
                    case DefaultVibratoLength.L75:
                        txtVibratoLength.Text = "75";
                        break;
                }
            }

            this.comboVibratoType.SelectedIndexChanged += new System.EventHandler( this.comboVibratoType_SelectedIndexChanged );
            this.txtVibratoLength.TextChanged += new System.EventHandler( txtVibratoLength_TextChanged );

            m_note_length = note_length;
            Misc.ApplyFontRecurse( this, AppManager.EditorConfig.BaseFont );
        }

        public void ApplyLanguage() {
            Text = _( "Vibrato property" );
            lblVibratoLength.Text = _( "Vibrato length" ) + "(&L)";
            lblVibratoType.Text = _( "Vibrato type" ) + "(&T)";
            btnOK.Text = _( "OK" );
            btnCancel.Text = _( "Cancel" );
        }

        public static string _( string id ) {
            return Messaging.GetMessage( id );
        }

        /// <summary>
        /// 編集済みのビブラート設定．既にCloneされているので，改めてCloneしなくて良い
        /// </summary>
        public VibratoHandle VibratoHandle {
            get {
                return m_vibrato;
            }
        }

        private void btnOK_Click( object sender, EventArgs e ) {
            this.DialogResult = DialogResult.OK;
        }

        private void comboVibratoType_SelectedIndexChanged( object sender, EventArgs e ) {
            int index = comboVibratoType.SelectedIndex;
            if ( index >= 0 ) {
                string s = (string)comboVibratoType.Items[index];
                if ( s == _NON_VIBRATO ) {
                    m_vibrato = null;
                    txtVibratoLength.Enabled = false;
                    return;
                } else {
                    txtVibratoLength.Enabled = true;
                    foreach ( VibratoType vt in Enum.GetValues( typeof( VibratoType ) ) ) {
                        string vt_s = VibratoTypeUtil.ToString( vt );
                        if ( s == vt_s ) {
                            int percent;
                            try{
                                percent = int.Parse( txtVibratoLength.Text );
                            }catch{
                                return;
                            }
                            m_vibrato = VibratoTypeUtil.getDefaultVibratoHandle( vt, (int)(m_note_length * percent / 100.0f) );
                            return;
                        }
                    }
                }
            }
        }

        private void txtVibratoLength_TextChanged( object sender, System.EventArgs e ) {
#if DEBUG
            Common.DebugWriteLine( "txtVibratoLength_TextChanged" );
            Common.DebugWriteLine( "    (m_vibrato==null)=" + (m_vibrato == null) );
#endif
            int percent = 0;
            try {
                percent = int.Parse( txtVibratoLength.Text );
                if ( percent < 0 ) {
                    percent = 0;
                } else if ( 100 < percent ) {
                    percent = 100;
                }
            } catch {
                return;
            }
            if ( percent == 0 ) {
                m_vibrato = null;
                txtVibratoLength.Enabled = false;
            } else {
                if ( m_vibrato != null ) {
                    int new_length = (int)(m_note_length * percent / 100.0f);
                    m_vibrato.Length = new_length;
                }
            }
        }
    }

}
