﻿/*
 * VsqFileEx.cs
 * Copyright (c) 2008-2009 kbinani
 *
 * This file is part of Boare.Cadencii.
 *
 * Boare.Cadencii is free software; you can redistribute it and/or
 * modify it under the terms of the GPLv3 License.
 *
 * Boare.Cadencii is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */
using System;
using System.Collections.Generic;
using System.IO;
using System.Xml.Serialization;

using Boare.Lib.Vsq;

namespace Boare.Cadencii {

    [Serializable]
    public class VsqFileEx : VsqFile, ICloneable, ICommandRunnable {
        static XmlSerializer s_vsq_serializer = new XmlSerializer( typeof( VsqFileEx ) );

        public AttachedCurve AttachedCurves;

        /* /// <summary>
        /// VSQファイルの指定されたクロック範囲のイベント等を削除します
        /// </summary>
        /// <param name="vsq">編集対象のVsqFileインスタンス</param>
        /// <param name="clock_start">削除を行う範囲の開始クロック</param>
        /// <param name="clock_end">削除を行う範囲の終了クロック</param>
        public override void removePart( int clock_start, int clock_end ) {
            base.removePart( clock_start, clock_end );
            int dclock = clock_end - clock_start;
            for ( int track = 1; track < Track.Count; track++ ) {
                VsqBPList buf_bplist = (VsqBPList)m_pitch[track - 1].Clone();
                m_pitch[track - 1].clear();
                int value_at_clock_end = buf_bplist.getValue( clock_end );
                bool clock_end_added = false;
                for ( Iterator itr = buf_bplist.keyClockIterator(); itr.hasNext(); ) {
                    int key = (int)itr.next();
                    if ( key < clock_start ) {
                        m_pitch[track - 1].add( key, buf_bplist.getValue( key ) );
                    } else if ( clock_end <= key ) {
                        if ( clock_end == key ) {
                            clock_end_added = true;
                        }
                        m_pitch[track - 1].add( key - dclock, buf_bplist.getValue( key ) );
                    }
                }
                if ( !clock_end_added ) {
                    m_pitch[track - 1].add( clock_end - dclock, value_at_clock_end );
                }
            }
        }*/

        /*public VsqBPList getPitchCurve( int track ) {
            return m_pitch[track - 1];
        }*/

        public override object Clone() {
            VsqFileEx ret = new VsqFileEx( "Miku", 1, 4, 4, 500000 );
            ret.Track = new List<VsqTrack>();
            for ( int i = 0; i < Track.Count; i++ ) {
                ret.Track.Add( (VsqTrack)Track[i].Clone() );
            }
#if USE_TEMPO_LIST

#else
            ret.TempoTable = new List<TempoTableEntry>();
            for ( int i = 0; i < TempoTable.Count; i++ ) {
                ret.TempoTable.Add( (TempoTableEntry)TempoTable[i].Clone() );
            }
#endif
            ret.TimesigTable = new List<TimeSigTableEntry>();
            for ( int i = 0; i < TimesigTable.Count; i++ ) {
                ret.TimesigTable.Add( (TimeSigTableEntry)TimesigTable[i].Clone() );
            }
            ret.m_tpq = m_tpq;
            ret.TotalClocks = TotalClocks;
            ret.m_base_tempo = m_base_tempo;
            ret.Master = (VsqMaster)Master.Clone();
            ret.Mixer = (VsqMixer)Mixer.Clone();
            //ret.m_premeasure_clocks = m_premeasure_clocks;
            ret.AttachedCurves = (AttachedCurve)AttachedCurves.Clone();
            /*ret.m_pitch.Clear();
            for ( int i = 0; i < m_pitch.Count; i++ ) {
                ret.m_pitch.Add( (VsqBPList)m_pitch[i].Clone() );
            }*/
            return ret;
        }

        /// <summary>
        /// トラックを削除するコマンドを発行します。VstRendererを取り扱う関係上、VsqCommandを使ってはならない。
        /// </summary>
        /// <param name="track"></param>
        /// <returns></returns>
        public static CadenciiCommand generateCommandDeleteTrack( int track ) {
            CadenciiCommand command = new CadenciiCommand();
            command.Type = CadenciiCommandType.DeleteTrack;
            command.Args = new object[1];
            command.Args[0] = track;
            return command;
        }

        public static CadenciiCommand generateCommandTrackReplace( int track, VsqTrack item, BezierCurves attached_curve ) {
        //public static CadenciiCommand generateCommandTrackReplace( int track, VsqTrack item, BezierCurves attached_curve, VsqBPList pitch ) {
            CadenciiCommand command = new CadenciiCommand();
            command.Type = CadenciiCommandType.TrackReplace;
            command.Args = new object[3];
            command.Args[0] = track;
            command.Args[1] = item.Clone();
            command.Args[2] = attached_curve.Clone();
            //command.Args[3] = pitch.Clone();
            return command;
        }

        /// <summary>
        /// トラックを追加するコマンドを発行します．VstRendererを取り扱う関係上、VsqCommandを使ってはならない。
        /// </summary>
        /// <param name="track"></param>
        /// <returns></returns>
        public static CadenciiCommand generateCommandAddTrack( VsqTrack track, VsqMixerEntry mixer, int position, BezierCurves attached_curve ) {
            CadenciiCommand command = new CadenciiCommand();
            command.Type = CadenciiCommandType.AddTrack;
            command.Args = new object[4];
            command.Args[0] = track.Clone();
            command.Args[1] = mixer;
            command.Args[2] = position;
            command.Args[3] = attached_curve.Clone();
            //command.Args[4] = pitch.Clone();
            return command;
        }

        public static CadenciiCommand generateCommandAddBezierChain( int track, CurveType curve_type, int chain_id, int clock_resolution, BezierChain chain ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.AddBezierChain;
            ret.Args = new object[5];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = (BezierChain)chain.Clone();
            ret.Args[3] = clock_resolution;
            ret.Args[4] = chain_id;
            return ret;
        }

        public static CadenciiCommand generateCommandDeleteBezierChain( int track, CurveType curve_type, int chain_id, int clock_resolution ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.DeleteBezierChain;
            ret.Args = new object[4];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = chain_id;
            ret.Args[3] = clock_resolution;
            return ret;
        }

        public static CadenciiCommand generateCommandReplaceBezierChain( int track, CurveType curve_type, int chain_id, BezierChain chain, int clock_resolution ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.ReplaceBezierChain;
            ret.Args = new object[5];
            ret.Args[0] = track;
            ret.Args[1] = curve_type;
            ret.Args[2] = chain_id;
            ret.Args[3] = chain;
            ret.Args[4] = clock_resolution;
            return ret;
        }

        public static CadenciiCommand generateCommandReplace( VsqFileEx vsq ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.Replace;
            ret.Args = new object[1];
            ret.Args[0] = (VsqFileEx)vsq.Clone();
            return ret;
        }

        public static CadenciiCommand generateCommandReplaceAttachedCurveRange( int track, Dictionary<CurveType, List<BezierChain>> attached_curves ) {
            CadenciiCommand ret = new CadenciiCommand();
            ret.Type = CadenciiCommandType.ReplaceAttachedCurveRange;
            ret.Args = new object[2];
            ret.Args[0] = track;
            ret.Args[1] = attached_curves;
            return ret;
        }

        public VsqCommand preprocessSpecialCommand( VsqCommand command ) {
#if DEBUG
            AppManager.DebugWriteLine( "preprocessSpecialCommand; command.Type=" + command.Type );
#endif
            if ( command.Type == VsqCommandType.TrackEditCurve ) {
                #region TrackEditCurve
                int track = (int)command.Args[0];
                string curve = (string)command.Args[1];
                List<BPPair> com = (List<BPPair>)command.Args[2];
                VsqBPList target = target = Track[track].getCurve( curve );

                VsqCommand inv = null;
                List<BPPair> edit = new List<BPPair>();
                if ( com != null ) {
                    if ( com.Count > 0 ) {
                        int start_clock = com[0].Clock;
                        int end_clock = com[0].Clock;
                        foreach ( BPPair item in com ) {
                            start_clock = Math.Min( start_clock, item.Clock );
                            end_clock = Math.Max( end_clock, item.Clock );
                        }
                        Track[track].setEditedStart( start_clock );
                        Track[track].setEditedEnd( end_clock );
                        int start_value = target.getValue( start_clock );
                        int end_value = target.getValue( end_clock );
                        for ( Iterator i = target.keyClockIterator(); i.hasNext(); ) {
                            int clock = (int)i.next();
                            if ( start_clock <= clock && clock <= end_clock ) {
                                edit.Add( new BPPair( clock, target.getValue( clock ) ) );
                            }
                        }
                        bool start_found = false;
                        bool end_found = false;
                        for ( int i = 0; i < edit.Count; i++ ) {
                            if ( edit[i].Clock == start_clock ) {
                                start_found = true;
                                edit[i].Value = start_value;
                                if ( start_found && end_found ) {
                                    break;
                                }
                            }
                            if ( edit[i].Clock == end_clock ) {
                                end_found = true;
                                edit[i].Value = end_value;
                                if ( start_found && end_found ) {
                                    break;
                                }
                            }
                        }
                        if ( !start_found ) {
                            edit.Add( new BPPair( start_clock, start_value ) );
                        }
                        if ( !end_found ) {
                            edit.Add( new BPPair( end_clock, end_value ) );
                        }

                        // 並べ替え
                        edit.Sort();
                        inv = VsqCommand.generateCommandTrackEditCurve( track, curve, edit );
                    } else if ( com.Count == 0 ) {
                        inv = VsqCommand.generateCommandTrackEditCurve( track, curve, new List<BPPair>() );
                    }
                }

                updateTotalClocks();
                if ( com.Count == 0 ) {
                    return inv;
                } else if ( com.Count == 1 ) {
                    bool found = false;
                    for ( Iterator itr = target.keyClockIterator(); itr.hasNext(); ) {
                        int clock = (int)itr.next();
                        if ( clock == com[0].Clock ) {
                            found = true;
                            target.add( clock, com[0].Value );
                            break;
                        }
                    }
                    if ( !found ) {
                        target.add( com[0].Clock, com[0].Value );
                    }
                } else {
                    int start_clock = com[0].Clock;
                    int end_clock = com[com.Count - 1].Clock;
                    bool removed = true;
                    while ( removed ) {
                        removed = false;
                        for ( Iterator itr = target.keyClockIterator(); itr.hasNext(); ) {
                            int clock = (int)itr.next();
                            if ( start_clock <= clock && clock <= end_clock ) {
                                target.remove( clock );
                                removed = true;
                                break;
                            }
                        }
                    }
                    foreach ( BPPair item in com ) {
                        target.add( item.Clock, item.Value );
                    }
                }
                return inv;
                #endregion
            } else if ( command.Type == VsqCommandType.TrackEditCurveRange ) {
                #region TrackEditCurveRange
                int track = (int)command.Args[0];
                string[] curves = (string[])command.Args[1];
                List<BPPair>[] coms = (List<BPPair>[])command.Args[2];
                List<BPPair>[] inv_coms = new List<BPPair>[curves.Length];
                VsqCommand inv = null;

                for ( int k = 0; k < curves.Length; k++ ) {
                    string curve = curves[k];
                    VsqBPList target = Track[track].getCurve( curve );
                    List<BPPair> com = coms[k];
                    List<BPPair> edit = new List<BPPair>();
                    if ( com != null ) {
                        if ( com.Count > 0 ) {
                            int start_clock = com[0].Clock;
                            int end_clock = com[0].Clock;
                            foreach ( BPPair item in com ) {
                                start_clock = Math.Min( start_clock, item.Clock );
                                end_clock = Math.Max( end_clock, item.Clock );
                            }
                            Track[track].setEditedStart( start_clock );
                            Track[track].setEditedEnd( end_clock );
                            int start_value = target.getValue( start_clock );
                            int end_value = target.getValue( end_clock );
                            for ( Iterator itr = target.keyClockIterator(); itr.hasNext(); ) {
                                int clock = (int)itr.next();
                                if ( start_clock <= clock && clock <= end_clock ) {
                                    edit.Add( new BPPair( clock, target.getValue( clock ) ) );
                                }
                            }
                            bool start_found = false;
                            bool end_found = false;
                            for ( int i = 0; i < edit.Count; i++ ) {
                                if ( edit[i].Clock == start_clock ) {
                                    start_found = true;
                                    edit[i].Value = start_value;
                                    if ( start_found && end_found ) {
                                        break;
                                    }
                                }
                                if ( edit[i].Clock == end_clock ) {
                                    end_found = true;
                                    edit[i].Value = end_value;
                                    if ( start_found && end_found ) {
                                        break;
                                    }
                                }
                            }
                            if ( !start_found ) {
                                edit.Add( new BPPair( start_clock, start_value ) );
                            }
                            if ( !end_found ) {
                                edit.Add( new BPPair( end_clock, end_value ) );
                            }

                            // 並べ替え
                            edit.Sort();
                            inv_coms[k] = edit;
                        } else if ( com.Count == 0 ) {
                            inv_coms[k] = new List<BPPair>();
                        }
                    }

                    updateTotalClocks();
                    if ( com.Count == 0 ) {
                        return inv;
                    } else if ( com.Count == 1 ) {
                        bool found = false;
                        for ( Iterator itr = target.keyClockIterator(); itr.hasNext(); ) {
                            int clock = (int)itr.next();
                            if ( clock == com[0].Clock ) {
                                found = true;
                                target.add( clock, com[0].Value );
                                break;
                            }
                        }
                        if ( !found ) {
                            target.add( com[0].Clock, com[0].Value );
                        }
                    } else {
                        int start_clock = com[0].Clock;
                        int end_clock = com[com.Count - 1].Clock;
                        bool removed = true;
                        while ( removed ) {
                            removed = false;
                            for ( Iterator itr = target.keyClockIterator(); itr.hasNext(); ) {
                                int clock = (int)itr.next();
                                if ( start_clock <= clock && clock <= end_clock ) {
                                    target.remove( clock );
                                    removed = true;
                                    break;
                                }
                            }
                        }
                        foreach ( BPPair item in com ) {
                            target.add( item.Clock, item.Value );
                        }
                    }
                }
                return VsqCommand.generateCommandTrackEditCurveRange( track, curves, inv_coms );
                #endregion
            }
            return null;
        }

        public ICommand executeCommand( ICommand com ) {
#if DEBUG
            AppManager.DebugWriteLine( "VsqFileEx.Execute" );
#endif
            CadenciiCommand command = (CadenciiCommand)com;
#if DEBUG
            AppManager.DebugWriteLine( "VsqFileEx.Execute; command.Type=" + command.Type );
#endif
            CadenciiCommand ret = null;
            if ( command.Type == CadenciiCommandType.VsqCommand ) {
                ret = new CadenciiCommand();
                ret.Type = CadenciiCommandType.VsqCommand;
                if ( command.VsqCommand.Type == VsqCommandType.TrackEditCurve || command.VsqCommand.Type == VsqCommandType.TrackEditCurveRange ) {
                    ret.VsqCommand = preprocessSpecialCommand( command.VsqCommand );
                } else {
                    ret.VsqCommand = base.executeCommand( command.VsqCommand );
                }
            } else {
                if ( command.Type == CadenciiCommandType.AddBezierChain ) {
                    #region AddBezierChain
#if DEBUG
                    AppManager.DebugWriteLine( "    AddBezierChain" );
#endif
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    BezierChain chain = (BezierChain)command.Args[2];
                    int clock_resolution = (int)command.Args[3];
                    int added_id = (int)command.Args[4];
                    AttachedCurves[track - 1].AddBezierChain( curve_type, chain, added_id );
                    ret = generateCommandDeleteBezierChain( track, curve_type, added_id, clock_resolution );
                    if ( chain.Count > 1 ) {
                        int min = (int)chain.points[0].Base.X;
                        int max = min;
                        for ( int i = 1; i < chain.points.Count; i++ ) {
                            min = Math.Min( min, (int)chain.points[i].Base.X );
                            max = Math.Max( max, (int)chain.points[i].Base.X );
                        }
                        int max_value = curve_type.Maximum;
                        int min_value = curve_type.Minimum;
                        if ( min < max ) {
                            List<BPPair> edit = new List<BPPair>();
                            int last_value = int.MaxValue;
                            for ( int clock = min; clock < max; clock += clock_resolution ) {
                                int value = (int)chain.GetValue( (float)clock );
                                if ( value < min_value ) {
                                    value = min_value;
                                } else if ( max_value < value ) {
                                    value = max_value;
                                }
                                if ( value != last_value ) {
                                    edit.Add( new BPPair( clock, value ) );
                                    last_value = value;
                                }
                            }
                            int value2;
                            value2 = Track[track].getCurve( curve_type.Name ).getValue( max );
                            edit.Add( new BPPair( max, value2 ) );
                            command.VsqCommand = VsqCommand.generateCommandTrackEditCurve( track, curve_type.Name, edit );
                        }
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.DeleteBezierChain ) {
                    #region DeleteBezierChain
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    int chain_id = (int)command.Args[2];
                    int clock_resolution = (int)command.Args[3];
                    BezierChain chain = (BezierChain)AttachedCurves[track - 1].GetBezierChain( curve_type, chain_id ).Clone();
                    AttachedCurves[track - 1].Remove( curve_type, chain_id );
                    ret = generateCommandAddBezierChain( track, curve_type, chain_id, clock_resolution, chain );
                    if ( command.VsqCommand != null && ret != null ) {
                        if ( command.VsqCommand.Type == VsqCommandType.TrackEditCurve || command.VsqCommand.Type == VsqCommandType.TrackEditCurveRange ) {
                            ret.VsqCommand = preprocessSpecialCommand( command.VsqCommand );
                        } else {
                            ret.VsqCommand = base.executeCommand( command.VsqCommand );
                        }
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.ReplaceBezierChain ) {
                    #region ReplaceBezierChain
                    int track = (int)command.Args[0];
                    CurveType curve_type = (CurveType)command.Args[1];
                    int chain_id = (int)command.Args[2];
                    BezierChain chain = (BezierChain)command.Args[3];
                    int clock_resolution = (int)command.Args[4];
                    BezierChain target = (BezierChain)AttachedCurves[track - 1].GetBezierChain( curve_type, chain_id ).Clone();
                    AttachedCurves[track - 1].SetBezierChain( curve_type, chain_id, chain );
                    ret = generateCommandReplaceBezierChain( track, curve_type, chain_id, target, clock_resolution );
                    if ( chain.Count == 1 ) {
                        int ex_min = (int)chain.points[0].Base.X;
                        int ex_max = ex_min;
                        if ( target.points.Count > 1 ) {
                            for ( int i = 1; i < target.points.Count; i++ ) {
                                ex_min = Math.Min( ex_min, (int)target.points[i].Base.X );
                                ex_max = Math.Max( ex_max, (int)target.points[i].Base.X );
                            }
                            if ( ex_min < ex_max ) {
                                int default_value = curve_type.Default;
                                List<BPPair> edit = new List<BPPair>();
                                edit.Add( new BPPair( ex_min, default_value ) );
                                edit.Add( new BPPair( ex_max, default_value ) );
                                command.VsqCommand = VsqCommand.generateCommandTrackEditCurve( track, curve_type.Name, edit );
                            }
                        }
                    } else if ( chain.Count > 1 ) {
                        int min = (int)chain.points[0].Base.X;
                        int max = min;
                        for ( int i = 1; i < chain.points.Count; i++ ) {
                            min = Math.Min( min, (int)chain.points[i].Base.X );
                            max = Math.Max( max, (int)chain.points[i].Base.X );
                        }
                        int ex_min = min;
                        int ex_max = max;
                        if ( target.points.Count > 0 ) {
                            ex_min = (int)target.points[0].Base.X;
                            ex_max = ex_min;
                            for ( int i = 1; i < target.points.Count; i++ ) {
                                ex_min = Math.Min( ex_min, (int)target.points[i].Base.X );
                                ex_max = Math.Max( ex_max, (int)target.points[i].Base.X );
                            }
                        }
                        int max_value = curve_type.Maximum;
                        int min_value = curve_type.Minimum;
                        int default_value = curve_type.Default;
                        List<BPPair> edit = new List<BPPair>();
                        if ( ex_min < min ) {
                            edit.Add( new BPPair( ex_min, default_value ) );
                        }
                        if ( min < max ) {
                            int last_value = int.MaxValue;
                            for ( int clock = min; clock < max; clock += clock_resolution ) {
                                int value = (int)chain.GetValue( (float)clock );
                                if ( value < min_value ) {
                                    value = min_value;
                                } else if ( max_value < value ) {
                                    value = max_value;
                                }
                                if ( last_value != value ) {
                                    edit.Add( new BPPair( clock, value ) );
                                    last_value = value;
                                }
                            }
                            int value2 = 0;
                            if ( max < ex_max ) {
                                value2 = (int)chain.GetValue( max );
                                if ( value2 < min_value ) {
                                    value2 = min_value;
                                } else if ( max_value < value2 ) {
                                    value2 = max_value;
                                }
                            } else {
                                value2 = Track[track].getCurve( curve_type.Name ).getValue( max );
                            }
                            edit.Add( new BPPair( max, value2 ) );
                        }
                        if ( max < ex_max ) {
                            if ( max + 1 < ex_max ) {
                                edit.Add( new BPPair( max + 1, default_value ) );
                            }
                            edit.Add( new BPPair( ex_max, default_value ) );
                        }
                        if ( edit.Count > 0 ) {
                            command.VsqCommand = VsqCommand.generateCommandTrackEditCurve( track, curve_type.Name, edit );
                        }
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.Replace ) {
                    #region Replace
                    VsqFileEx vsq = (VsqFileEx)command.Args[0];
                    VsqFileEx inv = (VsqFileEx)this.Clone();
                    Track.Clear();
                    for ( int i = 0; i < vsq.Track.Count; i++ ) {
                        Track.Add( (VsqTrack)vsq.Track[i].Clone() );
                    }
                    TempoTable.Clear();
                    for ( int i = 0; i < vsq.TempoTable.Count; i++ ) {
                        TempoTable.Add( (TempoTableEntry)vsq.TempoTable[i].Clone() );
                    }
                    TimesigTable.Clear();
                    for ( int i = 0; i < vsq.TimesigTable.Count; i++ ) {
                        TimesigTable.Add( (TimeSigTableEntry)vsq.TimesigTable[i].Clone() );
                    }
                    m_tpq = vsq.m_tpq;
                    TotalClocks = vsq.TotalClocks;
                    m_base_tempo = vsq.m_base_tempo;
                    Master = (VsqMaster)vsq.Master.Clone();
                    Mixer = (VsqMixer)vsq.Mixer.Clone();
                    AttachedCurves = (AttachedCurve)vsq.AttachedCurves.Clone();
                    updateTotalClocks();
                    ret = generateCommandReplace( inv );
                    #endregion
                } else if ( command.Type == CadenciiCommandType.ReplaceAttachedCurveRange ) {
                    #region ReplaceAttachedCurveRange
                    int track = (int)command.Args[0];
                    Dictionary<CurveType, List<BezierChain>> curves = (Dictionary<CurveType, List<BezierChain>>)command.Args[1];
                    Dictionary<CurveType, List<BezierChain>> inv = new Dictionary<CurveType, List<BezierChain>>();
                    foreach ( CurveType ct in curves.Keys ) {
                        List<BezierChain> chains = new List<BezierChain>();
                        List<BezierChain> src = this.AttachedCurves[track - 1][ct];
                        for ( int i = 0; i < src.Count; i++ ){
                            chains.Add( (BezierChain)src[i].Clone() );
                        }
                        inv.Add( ct, chains );

                        this.AttachedCurves[track - 1][ct].Clear();
                        foreach ( BezierChain bc in curves[ct] ) {
                            this.AttachedCurves[track - 1][ct].Add( bc );
                        }
                    }
                    ret = generateCommandReplaceAttachedCurveRange( track, inv );
                    #endregion
                } else if ( command.Type == CadenciiCommandType.AddTrack ) {
                    #region AddTrack
                    VsqTrack track = (VsqTrack)command.Args[0];
                    VsqMixerEntry mixer = (VsqMixerEntry)command.Args[1];
                    int position = (int)command.Args[2];
                    BezierCurves attached_curve = (BezierCurves)command.Args[3];
                    ret = VsqFileEx.generateCommandDeleteTrack( position );
                    if ( Track.Count <= 17 ) {
                        Track.Insert( position, (VsqTrack)track.Clone() );
                        AttachedCurves.Insert( position - 1, attached_curve );
                        Mixer.Slave.Insert( position - 1, (VsqMixerEntry)mixer.Clone() );
                    }
                    #endregion
                } else if ( command.Type == CadenciiCommandType.DeleteTrack ) {
                    #region DeleteTrack
                    int track = (int)command.Args[0];
                    ret = VsqFileEx.generateCommandAddTrack( Track[track], Mixer.Slave[track - 1], track, AttachedCurves[track - 1] );
                    Track.RemoveAt( track );
                    AttachedCurves.RemoveAt( track - 1 );
                    Mixer.Slave.RemoveAt( track - 1 );
                    updateTotalClocks();
                    #endregion
                } else if ( command.Type == CadenciiCommandType.TrackReplace ) {
                    #region TrackReplace
                    int track = (int)command.Args[0];
                    VsqTrack item = (VsqTrack)command.Args[1];
                    BezierCurves bezier_curves = (BezierCurves)command.Args[2];
                    ret = VsqFileEx.generateCommandTrackReplace( track, Track[track], AttachedCurves[track - 1] );
                    Track[track] = item;
                    AttachedCurves[track - 1] = bezier_curves;
                    updateTotalClocks();
                    #endregion
                }
                if ( command.VsqCommand != null && ret != null ) {
#if DEBUG
                    AppManager.DebugWriteLine( "VsqFileEx.executeCommand; command.VsqCommand.Type=" + command.VsqCommand.Type );
#endif
                    if ( command.VsqCommand.Type == VsqCommandType.TrackEditCurve || command.VsqCommand.Type == VsqCommandType.TrackEditCurveRange ) {
                        ret.VsqCommand = preprocessSpecialCommand( command.VsqCommand );
                    } else {
                        ret.VsqCommand = base.executeCommand( command.VsqCommand );
                    }
                }
            }
            return ret;
        }

        public VsqFileEx()
            : this( "Miku", 1, 4, 4, 500000 ) {
            Track.Clear();
            TempoTable.Clear();
            TimesigTable.Clear();
        }

        public VsqFileEx( string singer, int pre_measure, int numerator, int denominator, int tempo ) :
            base( singer, pre_measure, numerator, denominator, tempo ) {
            AttachedCurves = new AttachedCurve();
            for ( int i = 1; i < Track.Count; i++ ) {
                AttachedCurves.Add( new BezierCurves() );
            }
        }

        public VsqFileEx( string _fpath ) :
            base( _fpath ){
            AttachedCurves = new AttachedCurve();

            string xml = Path.Combine( Path.GetDirectoryName( _fpath ), Path.GetFileName( _fpath ) + ".xml" );
            if ( File.Exists( xml ) ) {
                AttachedCurve tmp = null;
                FileStream fs = null;
                try {
                    fs = new FileStream( xml, FileMode.Open );
                    tmp = (AttachedCurve)AppManager.XmlSerializerListBezierCurves.Deserialize( fs );
                } catch ( Exception ex ) {
                    bocoree.debug.push_log( "ex=" + ex );
                    // 1.4.xのxmlとして読み込みを試みる
                    if ( fs != null ) {
                        fs.Close();
                        fs = null;
                    }
                    Rescue14xXml rx = new Rescue14xXml();
                    tmp = rx.Rescue( xml, Track.Count - 1 );
                } finally {
                    if ( fs != null ) {
                        fs.Close();
                    }
                }
                if ( tmp != null ) {
                    foreach ( BezierCurves bc in tmp.Curves ) {
                        foreach ( CurveType ct in AppManager.CURVE_USAGE ) {
                            List<BezierChain> list = bc[ct];
                            for ( int i = 0; i < list.Count; i++ ) {
                                list[i].ID = i + 1;
                                for ( int j = 0; j < list[i].points.Count; j++ ) {
                                    list[i].points[j].ID = j + 1;
                                }
                            }
                        }
                    }
                    AttachedCurves = tmp;
                }
            } else {
                for ( int i = 1; i < Track.Count; i++ ) {
                    AttachedCurves.Add( new BezierCurves() );
                }
            }

            // UTAUでエクスポートしたIconHandleは、IDS=UTAUとなっているので探知する
            for ( int i = 1; i < Track.Count; i++ ) {
                for ( Iterator itr = Track[i].getSingerEventIterator(); itr.hasNext(); ) {
                    VsqEvent ve = (VsqEvent)itr.next();
                    if ( ve.ID.IconHandle.IDS.ToLower() == "utau" ) {
                        Track[i].getCommon().Version = "UTU000";
                        break;
                    }
                }
            }
        }

        public VsqFileEx( UstFile ust )
            : this( "Miku", 1, 4, 4, ust.getBaseTempo() ) {
            int clock_count = 480 * 4; //pre measure = 1、4分の4拍子としたので
            VsqBPList pitch = new VsqBPList( 0, -2400, 2400 );
            for ( Iterator itr = ust.getTrack( 0 ).getNoteEventIterator(); itr.hasNext(); ) {
                UstEvent ue = (UstEvent)itr.next();
                if ( ue.Lyric != "R" ) {
                    VsqID id = new VsqID( 0 );
                    id.Length = ue.Length;
                    string psymbol = "a";
                    if ( !SymbolTable.attatch( ue.Lyric, out psymbol ) ) {
                        psymbol = "a";
                    }
                    id.LyricHandle = new LyricHandle( ue.Lyric, psymbol );
                    id.Note = ue.Note;
                    id.type = VsqIDType.Anote;
                    VsqEvent ve = new VsqEvent( clock_count, id );
                    ve.UstEvent = (UstEvent)ue.Clone();
                    Track[1].addEvent( ve );

                    if ( ue.Pitches != null ) {
                        // PBTypeクロックごとにデータポイントがある
                        int clock = clock_count - ue.PBType;
                        for ( int i = 0; i < ue.Pitches.Length; i++ ) {
                            clock += ue.PBType;
                            pitch.add( clock, (int)ue.Pitches[i] );
                        }
                    }
                }
                if ( ue.Tempo > 0.0f ) {
                    TempoTable.Add( new TempoTableEntry( clock_count, (int)(60e6 / ue.Tempo), 0.0 ) );
                }
                clock_count += ue.Length;
            }
            updateTempoInfo();
            updateTotalClocks();
            updateTimesigInfo();
            reflectPitch( this, 1, pitch );
        }

        /// <summary>
        /// master==MasterPitchControl.Pitchの場合、m_pitchからPITとPBSを再構成。
        /// master==MasterPitchControl.PITandPBSの場合、PITとPBSからm_pitchを再構成
        /// </summary>
        public static void reflectPitch( VsqFile vsq, int track, VsqBPList pitch ) {
            //double offset = AttachedCurves[track - 1].MasterTuningInCent * 100;
            List<int> keyclocks = new List<int>( pitch.getKeys() );
            VsqBPList pit = new VsqBPList( 0, -8192, 8191 );
            VsqBPList pbs = new VsqBPList( 2, 0, 24 );
            int premeasure_clock = vsq.getPreMeasureClocks();
            int lastpit = pit.Default;
            int lastpbs = pbs.Default;
            int vpbs = 24;
            int vpit = 0;

            List<int> parts = new List<int>();   // 連続した音符ブロックの先頭音符のクロック位置。のリスト
            parts.Add( premeasure_clock );
            int lastclock = premeasure_clock;
            for ( Iterator itr = vsq.Track[track].getNoteEventIterator(); itr.hasNext(); ) {
                VsqEvent ve = (VsqEvent)itr.next();
                if ( ve.Clock <= lastclock ) {
                    lastclock = Math.Max( lastclock, ve.Clock + ve.ID.Length );
                } else {
                    parts.Add( ve.Clock );
                    lastclock = ve.Clock + ve.ID.Length;
                }
            }

            for ( int i = 0; i < parts.Count; i++ ) {
                int partstart = parts[i];
                int partend = int.MaxValue;
                if ( i + 1 < parts.Count ) {
                    partend = parts[i + 1];
                }

                // まず、区間内の最大ピッチベンド幅を調べる
                double max = 0;
                for ( int j = 0; j < keyclocks.Count; j++ ) {
                    if ( keyclocks[j] < partstart ) {
                        continue;
                    }
                    if ( partend <= keyclocks[j] ) {
                        break;
                    }
                    max = Math.Max( max, Math.Abs( pitch.getValue( keyclocks[j] ) / 10000.0 ) );
                }

                // 最大ピッチベンド幅を表現できる最小のPBSを計算
                vpbs = (int)(Math.Ceiling( max * 8192.0 / 8191.0 ) + 0.1);
                if ( vpbs <= 0 ) {
                    vpbs = 1;
                }
                double pitch2 = pitch.getValue( partstart ) / 10000.0;
                if ( lastpbs != vpbs ) {
                    pbs.add( partstart, vpbs );
                    lastpbs = vpbs;
                }
                vpit = (int)(pitch2 * 8192 / (double)vpbs);
                if ( lastpit != vpit ) {
                    pit.add( partstart, vpit );
                    lastpit = vpit;
                }
                for ( int j = 0; j < keyclocks.Count; j++ ) {
                    int clock = keyclocks[j];
                    if ( clock < partstart ) {
                        continue;
                    }
                    if ( partend <= clock ) {
                        break;
                    }
                    if ( clock != partstart ) {
                        pitch2 = pitch.getValue( clock ) / 10000.0;
                        vpit = (int)(pitch2 * 8192 / (double)vpbs);
                        if ( lastpit != vpit ) {
                            pit.add( clock, vpit );
                            lastpit = vpit;
                        }
                    }
                }
            }
            vsq.Track[track].setCurve( "pit", pit );
            vsq.Track[track].setCurve( "pbs", pbs );
        }

        public void writeAsXml( string file ) {
            //reflectPitch( MasterPitchControl.Pitch );
            using ( FileStream fs = new FileStream( file, FileMode.Create, FileAccess.Write ) ) {
                s_vsq_serializer.Serialize( fs, this );
            }
        }

        public static VsqFileEx readFromXml( string file ) {
            VsqFileEx ret = null;
            using ( FileStream fs = new FileStream( file, FileMode.Open, FileAccess.Read ) ) {
                ret = (VsqFileEx)s_vsq_serializer.Deserialize( fs );
            }

            if( ret == null ){
                return null;
            }
            // ベジエ曲線のIDを播番
            if ( ret.AttachedCurves != null ) {
                foreach ( BezierCurves bc in ret.AttachedCurves.Curves ) {
                    foreach ( CurveType ct in AppManager.CURVE_USAGE ) {
                        List<BezierChain> list = bc[ct];
                        for ( int i = 0; i < list.Count; i++ ) {
                            list[i].ID = i + 1;
                            for ( int j = 0; j < list[i].points.Count; j++ ) {
                                list[i].points[j].ID = j + 1;
                            }
                        }
                    }
                }
            } else {
                for ( int i = 1; i < ret.Track.Count; i++ ) {
                    ret.AttachedCurves.Add( new BezierCurves() );
                }
            }
            return ret;
        }

        public override void write( string file ) {
            base.write( file );
        }

        public override void write( string file, int msPreSend ) {
            base.write( file, msPreSend );
        }

        public class Rescue14xXml {
            public class BezierCurves {
                public BezierChain[] Dynamics;
                public BezierChain[] Brethiness;
                public BezierChain[] Brightness;
                public BezierChain[] Clearness;
                public BezierChain[] Opening;
                public BezierChain[] GenderFactor;
                public BezierChain[] PortamentoTiming;
                public BezierChain[] PitchBend;
                public BezierChain[] PitchBendSensitivity;
                public BezierChain[] VibratoRate;
                public BezierChain[] VibratoDepth;

                public BezierCurves() {
                    Dynamics = new BezierChain[0];
                    Brethiness = new BezierChain[0];
                    Brightness = new BezierChain[0];
                    Clearness = new BezierChain[0];
                    Opening = new BezierChain[0];
                    GenderFactor = new BezierChain[0];
                    PortamentoTiming = new BezierChain[0];
                    PitchBend = new BezierChain[0];
                    PitchBendSensitivity = new BezierChain[0];
                    VibratoRate = new BezierChain[0];
                    VibratoDepth = new BezierChain[0];
                }
            }

            public Boare.Cadencii.AttachedCurve Rescue( string file, int num_track ) {
#if DEBUG
                AppManager.DebugWriteLine( "VsqFileEx.Rescue14xXml.Rescue; file=" + file + "; num_track=" + num_track );
                bocoree.debug.push_log( "   constructing serializer..." );
#endif
                XmlSerializer xs = null;
                try {
                    xs = new XmlSerializer( typeof( List<Boare.Cadencii.VsqFileEx.Rescue14xXml.BezierCurves> ) );
                } catch ( Exception ex ) {
                    bocoree.debug.push_log( "    ex=" + ex );
                }
                if ( xs == null ) {
                    return null;
                }
#if DEBUG
                bocoree.debug.push_log( "    ...done" );
                bocoree.debug.push_log( "    constructing FileStream..." );
#endif
                FileStream fs = new FileStream( file, FileMode.Open );
#if DEBUG
                bocoree.debug.push_log( "    ...done" );
#endif
                AttachedCurve ac = null;
#if !DEBUG
                try {
#endif
#if DEBUG
                bocoree.debug.push_log( "    serializing..." );
#endif
                    List<Boare.Cadencii.VsqFileEx.Rescue14xXml.BezierCurves> list = (List<Boare.Cadencii.VsqFileEx.Rescue14xXml.BezierCurves>)xs.Deserialize( fs );
#if DEBUG
                    bocoree.debug.push_log( "    ...done" );
                    bocoree.debug.push_log( "    (list==null)=" + (list == null) );
                    bocoree.debug.push_log( "    list.Count=" + list.Count );
#endif
                    if ( list.Count >= num_track ) {
                        ac = new AttachedCurve();
                        ac.Curves = new List<Boare.Cadencii.BezierCurves>();
                        for ( int i = 0; i < num_track; i++ ) {
#if DEBUG
                            bocoree.debug.push_log( "    i=" + i );
#endif
                            Boare.Cadencii.BezierCurves add = new Boare.Cadencii.BezierCurves();
                            add.Brethiness = new List<BezierChain>( list[i].Brethiness );
                            add.Brightness = new List<BezierChain>( list[i].Brightness );
                            add.Clearness = new List<BezierChain>( list[i].Clearness );
                            add.Dynamics = new List<BezierChain>();
                            foreach ( BezierChain bc in list[i].Dynamics ) {
                                add.Dynamics.Add( (BezierChain)bc.Clone() );
                            }
                            add.FX2Depth = new List<BezierChain>();
                            add.GenderFactor = new List<BezierChain>( list[i].GenderFactor );
                            add.Harmonics = new List<BezierChain>();
                            add.Opening = new List<BezierChain>( list[i].Opening );
                            add.PortamentoTiming = new List<BezierChain>( list[i].PortamentoTiming );
                            add.Reso1Amp = new List<BezierChain>();
                            add.Reso2Amp = new List<BezierChain>();
                            add.Reso3Amp = new List<BezierChain>();
                            add.Reso4Amp = new List<BezierChain>();
                            add.Reso1BW = new List<BezierChain>();
                            add.Reso2BW = new List<BezierChain>();
                            add.Reso3BW = new List<BezierChain>();
                            add.Reso4BW = new List<BezierChain>();
                            add.Reso1Freq = new List<BezierChain>();
                            add.Reso2Freq = new List<BezierChain>();
                            add.Reso3Freq = new List<BezierChain>();
                            add.Reso4Freq = new List<BezierChain>();
                            add.VibratoDepth = new List<BezierChain>( list[i].VibratoDepth );
                            add.VibratoRate = new List<BezierChain>( list[i].VibratoRate );
                            ac.Curves.Add( add );
                        }
                    }
#if !DEBUG
                } catch ( Exception ex ) {
                    bocoree.debug.push_log( "Rescue14xXml; ex=" + ex );
                    ac = null;
                } finally {
                    if ( fs != null ) {
                        fs.Close();
                    }
                }
#endif
                return ac;
            }
        }

    }

}
