<?php
@require_once __DIR__ . '/common.php';

/**
 * カタログファイルを分析してタグを更新するためのクラス
 */
class parse_catalog {

    /**
     * 既知のタグ
     */
    private $known_tags = [];

    /**
     * データベースアクセス
     */
    private $pdo;

    /**
     * リモートアドレス
     */
    public $hostaddr;

    /**
     * タグ名をデータベースに登録しIDを返す。すでに登録されていれば、そのIDを返す。
     * @param $tag タグ名
     * @return タグID
     */
    function ensure_tag($tag) {
        $tag = trim($tag);
        if (!array_key_exists($tag, $this->known_tags)) {
            $stm = $this->pdo->prepare('select id from tag_entries where tagname = ?');
            $stm->execute([$tag]);
            $tagid = $stm->fetchColumn();
            if ($tagid === false) {
                $ins = $this->pdo->prepare('insert into tag_entries(tagname, hostaddr, sticky)
                    values(:tagname, :hostaddr, 1)');
                $ins->execute([
                    'tagname' => $tag,
                    'hostaddr' => $this->hostaddr
                    ]);
                $tagid = $this->pdo->lastInsertId();
            }
            $this->known_tags[$tag] = $tagid;
        } else {
            $tagid = $this->known_tags[$tag];
        }
        return $tagid;
    }

    /**
     * ZIPに関連づけられているパーツとタグ、ZIPとタグの関連と、パーツ一覧を削除する.
     * @param $zipid ZIPエントリのID
     */
    function clean_parts($zipid) {
        $stm = $this->pdo->prepare('delete from TAG_ZIP_REL where zipid = ?');
        $stm->execute([$zipid]);
        $stm = $this->pdo->prepare('delete from TAG_PARTS_REL where TAG_PARTS_REL.partsid in (
            select PARTS_ENTRIES.id from PARTS_ENTRIES where PARTS_ENTRIES.zipid = ?)');
        $stm->execute([$zipid]);
        $stm = $this->pdo->prepare('delete from PARTS_ENTRIES where zipid = ?');
        $stm->execute([$zipid]);
        $stm = null;
    }

    /**
     * パーツにもZIPにも関連づけられていないタグがあれば、そのタグを削除する.
     */
    function purge_unused_tags() {
        $stm = $this->pdo->prepare('select tagid, sum(cnt) sumcnt from (
            select a.id tagid, count(b.tagid) cnt from tag_entries a left outer join tag_parts_rel b on a.id = b.tagid group by a.id
            union all
            select a.id tagid, count(b.tagid) cnt from tag_entries a left outer join tag_zip_rel b on a.id = b.tagid group by a.id
            ) group by tagid having sumcnt < 1');
        $stm->execute();
        $rows = $stm->fetchAll();
        $stm = null;
        if ($rows !== false) {
            $delstm = $this->pdo->prepare('delete from tag_entries where id = ?');
            foreach ($rows as $row) {
                $tagid = $row['tagid'];
                $delstm->execute([$tagid]);
            }
            $delstm = null;
        }
    }

    /**
     * ZIPに関連づけられたパーツを登録しIDを返す。すでに登録されていれば、そのIDを返す。
     * @param $zipid ZIPエントリのID
     * @param $partsname パーツ名
     * @return パーツのID
     */
    function ensure_partsname($zipid, $partsname) {
        $stm = $this->pdo->prepare(
            'select id from parts_entries where zipid = :zipid and partsname = :partsname');
        $stm->execute([
            'zipid' => $zipid,
            'partsname' => $partsname
        ]);
        $partsid = $stm->fetchColumn();
        $stm = null;

        if ($partsid === false) {
            $ins = $this->pdo->prepare(
                'insert into parts_entries(zipid, partsname) values(:zipid, :partsname)');
            $ins->execute([
                'zipid' => $zipid,
                'partsname' => $partsname
            ]);
            $partsid = $this->pdo->lastInsertId();
            $ins = null;
        }

        return $partsid;
    }

    /**
     * パーツのタグを登録する。すでに登録済みであれば何もしない。
     * @param $partsid パーツID
     * @param $tagid タグID
     */
    function register_parts_tag($partsid, $tagid) {
        $stm = $this->pdo->prepare('insert into TAG_PARTS_REL(partsid, tagid) 
            select :partsid, :tagid where not exists (
                select * from TAG_PARTS_REL A where A.partsid = :partsid and A.tagid = :tagid)');
        $stm->execute([
            'partsid' => $partsid,
            'tagid' => $tagid
        ]);
    }

    /**
     * ZIPのタグを登録する。すでに登録済みであれば何もしない。
     * @param $partsid パーツID
     * @param $tagid タグID
     */
    function register_zip_tag($zipid, $tagid) {
        $stm = $this->pdo->prepare('insert into TAG_ZIP_REL(zipid, tagid) 
            select :zipid, :tagid where not exists (
                select * from TAG_ZIP_REL A where A.zipid = :zipid and A.tagid = :tagid)');
        $stm->execute([
            'zipid' => $zipid,
            'tagid' => $tagid
        ]);
    }

    /**
     * 指定したzipidのタグ情報を削除し、
     * その後、どこにも使われていないタグがあれば、それを削除する。
     * @param zipid ZIPファィルのID
     */
    public function unregister($zipid) {
        $this->pdo = create_pdo();
        $this->pdo->beginTransaction();
        try {
            $this->clean_parts($zipid);
            $this->purge_unused_tags();
            $this->pdo->commit();
            $this->pdo = null;

        } catch (Exception $e) {
            $this->pdo->rollBack();
            $this->pdo = null;
            throw $e;
        }
    }

    /**
     * カタログの文字列リストを解析する
     * @param $lines カタログの文字列のリスト
     * @param $zipid ZIPエントリのID
     */
    public function register($lines, $zipid) {
        $this->pdo = create_pdo();
        $this->pdo->beginTransaction();
        try {
            $this->clean_parts($zipid);

            foreach ($lines as $line) {
                $line = trim($line);
                $pos = strrpos($line, '=');
                if ($pos !== false) {
                    $partsname = substr($line, 0, $pos); // パーツ名別タグ
                    $partsname = trim($partsname);
                    $partsid = 0;
                    if (strlen($partsname) > 0) {
                        $partsid = $this->ensure_partsname($zipid, $partsname);
                    }
                    $tags = explode(',', substr($line, $pos + 1));
                    foreach ($tags as $tag) {
                        $tagid = $this->ensure_tag($tag);
                        $this->register_parts_tag($partsid, $tagid);
                    }
                    
                } else if (strlen($line) > 0) {
                    $tags = explode(',', $line);
                    foreach ($tags as $tag) {
                        $tagid = $this->ensure_tag($line); // 全体タグ
                        $this->register_zip_tag($zipid, $tagid);
                    }
                }
            }
            $this->purge_unused_tags();
            $this->pdo->commit();
            $this->pdo = null;

        } catch (Exception $e) {
            $this->pdo->rollBack();
            $this->pdo = null;
            throw $e;
        }
    }
}

/***
 * ZIPファイルからカタログファイルを抽出し、カタログを解析してデータベースを更新する。
 * @param $zipfile ZIPファイル
 * @param $zipid ZIPエントリのID
 * @param $hostaddr タグ登録時のホストアドレス
 */
function search_zip_catalog($zipfile, $zipid, $hostaddr) {
    $za = new ZipArchive();
    $res = $za->open($zipfile);
    if ($res === TRUE) {
        for ($i = 0; $i < $za->numFiles; $i++) { 
            $stat = $za->statIndex($i); 
            $entryname = $stat['name'];
            $entryname = str_replace('\\', '/', $entryname); // バックスラッシュで入っているzipもあり
            if (!preg_match('/\/$/', $entryname)) {
                // ファイル名とサイズの取り出し
                $lcname = strtolower(pathinfo($entryname, PATHINFO_BASENAME));
                $fsize = $stat['size'];
                if ($lcname == 'catalog.txt' && $fsize > 0) {
                    $fp = $za->getStream($entryname);
                    if ($fp !== false) {
                        // カタログファイルの読み込み(SJIS)
                        $lines = [];
                        while (($line = fgets($fp)) !== false) {
                            $lines[] = mb_convert_encoding($line, 'UTF-8', 'SJIS');
                        }
                        // カタログの解析とデータベースへの登録
                        $pc = new parse_catalog();
                        $pc->hostaddr = $hostaddr;
                        $pc->register($lines, $zipid);
                    }
                    fclose($fp);
                    break;
                }
            }
        }
        $za->close();
    }
}
?>
