package charactermanaj.model.util;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.ObjectOutputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Locale;

import charactermanaj.model.CharacterData;
import charactermanaj.model.io.CharacterDataPersistent;
import charactermanaj.model.io.CharacterDataPersistent.DocInfo;

/**
 * 埋め込みリソースを生成するためのビルド時のヘルパ.<br>
 * @author seraphy
 */
public class MakeEmbeddedResource {

	public static void main(String[] args) throws Exception {
		String fileName = "resources/schema/character.xml.ser";
		if (args.length >= 1) {
			fileName = args[0];
		}
		File file = new File(fileName);
		
		// 読み込む言語データ(ENは標準)
		Locale[] locales = new Locale[] {
				Locale.ENGLISH, // デフォルト
				Locale.JAPANESE, // 日本語固有、(現状、2つしか定義さてれいない)
		};

		// 各言語ごとのデフォルトキャラクターデータを読み込む
		HashMap<String, CharacterData> characterDataMap = new HashMap<String, CharacterData>();
		for (Locale locale : locales) {
			String lang = locale.getLanguage();
			if (characterDataMap.containsKey(lang)) {
				continue; // すでに読み込み済みなのでスキップする.
			}
			CharacterData cd = loadDefaultCharacterData(locale);
			characterDataMap.put(lang, cd);
		}

		// シリアライズする.
		FileOutputStream fos = new FileOutputStream(file);
		try {
			ObjectOutputStream oos = new ObjectOutputStream(fos);
			try {
				oos.writeObject(characterDataMap);
			} finally {
				oos.close();
			}
		} finally {
			fos.close();
		}
		System.out.println("make embedded resource : succeeded. file=" + file);
	}
	
	protected static CharacterData loadDefaultCharacterData(Locale locale) throws Exception {
		// 埋め込みリソースからデフォルトキャラクターデータを構築する.
		CharacterData cd;
		URL defaultCharacter = getEmbeddedResourceURL(CharacterDataPersistent.DEFAULT_CHARACTER_XML);
		InputStream is = defaultCharacter.openStream();
		try {
			DocInfo docInfo = new DocInfo();
			docInfo.setFirstElementName("character");
			docInfo.setNamespace(CharacterDataPersistent.NS);
			docInfo.setVersion("1.0");
			
			CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
			cd = persist.loadCharacterDataFromXML(is, null, docInfo, locale);

		} finally {
			is.close();
		}
		return cd;
	}

	protected static URL getEmbeddedResourceURL(String schemaName) {
		return CharacterDataPersistent.class.getResource(schemaName);
	}
	
}
