package charactermanaj.util;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Level;
import java.util.logging.Logger;

import charactermanaj.Main;

/**
 * 言語リソースを管理する.
 * @author seraphy
 */
public class SetupLocalization extends ResourceLoader {
	
	private final Logger logger = Logger.getLogger(getClass().getName());
	
	public static final String DIRNAME_RESOURCES = "resources";
	
	public static final String CATALOG_NAME = "/resource_list.txt";
	
	private File baseDir;
	
	/**
	 * アプリケーションデータ用ディレクトリを指定して構築する.
	 * @param baseDir データディレクトリ
	 */
	public SetupLocalization(File baseDir) {
		if (baseDir == null || !baseDir.isDirectory()) {
			throw new IllegalArgumentException();
		}
		this.baseDir = baseDir;
	}
	
	/**
	 * リソース一覧を取得する.<br>
	 * クラスパス上の一覧を取得する汎用的な方法はないため、カタログファイルを使用する.<br>
	 * {@link #CATALOG_NAME}で示されるリソース上のファイルにリソース一覧が記録されているものとする.<br>
	 * @return リソース一覧(言語関連リソースのみ)
	 * @throws IOException 失敗
	 */
	protected Collection<String> getResourceList() throws IOException {
		ArrayList<String> resources = new ArrayList<String>();
		URL listURL = Main.class.getResource(CATALOG_NAME);
		InputStream is = listURL.openStream();
		try {
			BufferedReader rd = new BufferedReader(new InputStreamReader(is));
			try {
				String line;
				while ((line = rd.readLine()) != null) {
					line = line.trim();
					if (line.startsWith("#")) {
						continue;
					}
					resources.add(line);
				}
				
			} finally {
				rd.close();
			}
		} finally {
			is.close();
		}
		logger.log(Level.FINE, "resource list: " +resources);
		return resources;
	}
	
	/**
	 * リソースをファイルにコピーする.<br>
	 * @param fromURL
	 * @param toFile
	 * @throws IOException
	 */
	protected void copyResource(URL fromURL, File toFile) throws IOException {
		logger.log(Level.INFO, "copy resource '" + fromURL + "' to '" + toFile + "'");
		File dir = toFile.getParentFile();
		if ( !dir.exists()) {
			if ( !dir.mkdirs()) {
				throw new IOException("can't create directory. " + dir);
			}
		}

		URLConnection conn = fromURL.openConnection();
		conn.setDoInput(true);
		InputStream is = conn.getInputStream();
		try {
			long lastModified = conn.getLastModified();
			OutputStream os = new FileOutputStream(toFile);
			try {
				byte[] buf = new byte[4096];
				for (;;) {
					int rd = is.read(buf);
					if (rd <= 0) {
						break;
					}
					os.write(buf, 0, rd);
				}
			} finally {
				os.close();
			}
			toFile.setLastModified(lastModified);
		} finally {
			is.close();
		}
	}
	
	/**
	 * リソースディレクトリを返す.
	 * @return リソースディレクトリ
	 */
	public File getResourceDir() {
		try {
			return new File(baseDir, DIRNAME_RESOURCES).getCanonicalFile();
		} catch (Exception ex) {
			throw new RuntimeException(ex);
		}
	}
	
	/**
	 * ローカルシステム上のアプリケーションデータディレクトリに言語リソースをコピーする.
	 * @throws IOException 失敗
	 */
	public void setupToLocal() throws IOException {
		File toDir = getResourceDir();
		ClassLoader cl = getDefaultClassLoader();
		for (String resourceName : getResourceList()) {
			URL url = cl.getResource(resourceName);
			if (url != null) {
				File toFile = new File(toDir, resourceName).getCanonicalFile();
				copyResource(url, toFile);

			} else {
				logger.log(Level.WARNING, "missing resource: " + resourceName);
			}
		}
	}
}
