package charactermanaj.model.io;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import charactermanaj.model.AppConfig;
import charactermanaj.model.CharacterData;
import charactermanaj.model.RecentData;
import charactermanaj.util.UserData;
import charactermanaj.util.UserDataFactory;

/**
 * 最後に使用したデータの使用状況を保存・復元するためのクラス
 * @author seraphy
 */
public final class RecentDataPersistent {

	/**
	 * ロガー
	 */
	private static final Logger logger = Logger.getLogger(RecentDataPersistent.class.getName());

	/**
	 * シングルトン
	 */
	private static final RecentDataPersistent inst = new RecentDataPersistent();
	
	/**
	 * 最後に使用したプロファイルのコートベースを保存するファイル名
	 */
	private static final String RECENT_CHARACTER_SER = "recent-character.ser";
	

	/**
	 * プライベートコンストラクタ
	 */
	private RecentDataPersistent() {
		super();
	}
	
	/**
	 * インスタンスを取得する
	 * @return インスタンス
	 */
	public static RecentDataPersistent getInstance() {
		return inst;
	}
	
	/**
	 * 最後に使用したキャラクターデータとして記録する.
	 * @param characterData
	 * @throws IOException 保存できなった場合
	 */
	public void saveRecent(CharacterData characterData) throws IOException {
		if (characterData == null) {
			throw new IllegalArgumentException();
		}
		if (!characterData.isValid()) {
			return;
		}
		
		AppConfig appConfig = AppConfig.getInstance();
		
		RecentData recentData = new RecentData();
		recentData.setAppVersion(appConfig.getSpecificationVersion());
		recentData.setDocBase(characterData.getDocBase());
		
		UserData recentCharacterStore = getRecentCharacterStore();
		recentCharacterStore.save(recentData);
	}
	
	/**
	 * 最後に使用したキャラクターデータを取得する.
	 * @return キャラクターデータ。最後に使用したデータが存在しない場合はnull
	 * @throws IOException 読み込みに失敗した場合
	 */
	public CharacterData loadRecent() throws IOException {
		UserData recentCharacterStore = getRecentCharacterStore();
		if (!recentCharacterStore.exists()) {
			return null;
		}

		RecentData recentData;
		try {
			recentData = (RecentData) recentCharacterStore.load();

		} catch (Exception ex) {
			// RecentData情報の復元に失敗した場合は最後に使用したデータが存在しないものとみなす.
			logger.log(Level.WARNING, "recent data loading failed. " + recentCharacterStore, ex);
			return null;
		}

		CharacterDataPersistent persist = CharacterDataPersistent.getInstance();
		return persist.loadProfile(recentData.getDocBase());
	}
	
	/**
	 * 最後に使用したキャラクタデータの保存先を取得する
	 * @return 保存先
	 */
	protected UserData getRecentCharacterStore() {
		UserDataFactory userDataFactory = UserDataFactory.getInstance();
		UserData recentCharacterStore = userDataFactory.getUserData(RECENT_CHARACTER_SER);
		return recentCharacterStore;
	}
	
	
}
